/*
     This file is part of GNUnet.
     (C) 2001, 2002, 2003, 2004, 2005 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/


/**
 * @file conf/confdata.c
 * @brief GNUnet Setup
 * @author Roman Zippel
 * @author Nils Durner
 */

/*
 * Originally Copyright (C) 2002 Roman Zippel <zippel@linux-m68k.org>
 */

#include <sys/stat.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <limits.h>

#define LKC_DIRECT_LINK
#include "lkc.h"

#include "platform.h"
#include "gnunet_util.h"

int conf_read(const char *name)
{
  char *val;
  struct symbol *sym;
  struct property *prop;
  struct expr *e;
  int i = 0;

  GNUNET_ASSERT(name != NULL);
  FREENONNULL(setConfigurationString("FILES",
				     "gnunet.conf",
				     name));
  readConfigFile(name);	

  for_all_symbols(i, sym) {
    sym->flags |= SYMBOL_NEW | SYMBOL_CHANGED;
    sym->flags &= ~SYMBOL_VALID;

    if (isConfigurationItemSet(sym->sect, sym->name)) {
      val = getConfigurationString(sym->sect, sym->name);
      if (!val)
	val = STRDUP("");

      switch (sym->type) {
      case S_TRISTATE:
	if (*val == 'm') {
	  sym->user.tri = mod;
	  sym->flags &= ~SYMBOL_NEW;
	  break;
	}
      case S_BOOLEAN:
	sym->user.tri = (*val == 'Y') ? yes : no;
	sym->flags &= ~SYMBOL_NEW;
	break;
      case S_STRING:
      case S_INT:
      case S_HEX:
	if (sym->user.val)
	  free(sym->user.val);
	
	if (sym_string_valid(sym, val)) {
	  sym->user.val = STRDUP(val);
	  sym->flags &= ~SYMBOL_NEW;
	}
	else {
	  LOG(LOG_ERROR,
	      _("%s: symbol value `%s' invalid for %s\n"),
	      name,
	      val,
	      sym->name);
	  sym->user.val = NULL;
	  sym->flags |= SYMBOL_NEW;
	}
	
	if (!sym_string_within_range(sym, val))
	  sym->flags |= SYMBOL_NEW;
	
	break;
      default:
	sym->user.val = NULL;
	sym->user.tri = no;
	
      }

      if (sym && sym_is_choice_value(sym)) {
	struct symbol *cs = prop_get_symbol(sym_get_choice_prop(sym));
	switch (sym->user.tri) {
	case no:
	  break;
	case mod:
	  if (cs->user.tri == yes)
	    /* warn? */;
	  break;
	case yes:
	  if (cs->user.tri != no)
	    /* warn? */;
	  cs->user.val = sym;
	  break;
	}
	cs->user.tri = E_OR(cs->user.tri, sym->user.tri);
	cs->flags &= ~SYMBOL_NEW;
      }

      sym_calc_value(sym);
      if (sym_has_value(sym) && !sym_is_choice_value(sym)) {
	if (sym->visible == no)
	  sym->flags |= SYMBOL_NEW;
      }
      if (!sym_is_choice(sym)) {
	FREE(val);
	continue;
      }
      prop = sym_get_choice_prop(sym);
      for (e = prop->expr; e; e = e->left.expr)
	if (e->right.sym->visible != no)
	  sym->flags |= e->right.sym->flags & SYMBOL_NEW;
      FREE(val);
    }
  }

  sym_change_count = 1;

  return 0;
}

int conf_write(const char * filename)
{
  FILE *out = NULL;
  struct symbol *sym;
  struct menu *menu;
  int type;
  const char *str;

  sym_clear_all_valid();

  out = FOPEN(filename, "w+");	
  if (NULL == out)
    return 1;
  fprintf(out,
	  "#%s"
	  "# Automatically generated by gnunet-setup%s"
	  "#%s",
	  NEWLINE,
	  NEWLINE,
	  NEWLINE);
  menu = rootmenu.list;
  while (menu) {
    sym = menu->sym;
    if (!sym) {
      str = menu_get_prompt(menu);
      if (str && strlen(str) > 0) {			
	fprintf(out,
		"%s"
		"#%s"
		"# %s%s"
		"#%s",
		NEWLINE,
		NEWLINE,
		str,
		NEWLINE,
		NEWLINE);
      }
      if (menu->section && strlen(menu->section) > 0)
	fprintf(out,
		"[%s]%s",
		menu->section,
		NEWLINE);
    } else if (!(sym->flags & SYMBOL_CHOICE)) {
      sym_calc_value_ext(sym, 1);
      sym->flags &= ~SYMBOL_WRITE;
      type = sym->type;
      if (type == S_TRISTATE) {
	sym_calc_value_ext(modules_sym, 1);
	if (modules_sym->curr.tri == no)
	  type = S_BOOLEAN;
      }
      switch (type) {
      case S_BOOLEAN:
      case S_TRISTATE:
	switch (sym_get_tristate_value(sym)) {
	case no:
	  fprintf(out,
		  "%s = NO",
		  sym->name);
	  break;
	case mod:
	  fprintf(out,
		  "%s = m",
		  sym->name);
	  break;
	case yes:
	  fprintf(out,
		  "%s = YES",
		  sym->name);
	  break;
	}
	break;
      case S_STRING:
        fprintf(out,
		"%s = \"%s\"",
		sym->name,
		sym_get_string_value(sym));
	break;
      case S_HEX:
	str = sym_get_string_value(sym);
	if (str[0] != '0' || (str[1] != 'x' && str[1] != 'X')) {
	  fprintf(out,
		  "%s = 0x%s",
		  sym->name,
		  str);
	  break;
	}
      case S_INT:
	fprintf(out,
		"%s = %s",
		sym->name,
		sym_get_string_value(sym));
	break;
      }
      fprintf(out,
	      "%s",
	      NEWLINE);
    }
    if (menu->list) {
      menu = menu->list;
      continue;
    }
    if (menu->next)
      menu = menu->next;
    else while ((menu = menu->parent)) {
      if (menu->next) {
	menu = menu->next;
	break;
      }
    }
  }

  fclose(out);
  sym_change_count = 0;

  return 0;
}
