/*
 * Copyright (c) 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.vise;

import java.sql.*;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;

/**
 * Represents a possible connection to a given database. The database
 * is modeled as a DatabaseModel object. Connection and disconnection
 * are done through JDBC.
 *
 * @author Liam Stewart
 * @author Maintained by <a href="mailto:fnasser@redhat.com">Fernando Nasser</a>
 * @version 1.2.0
 *
 * @see com.redhat.rhdb.vise.DatabaseModel
 */
public class ConnectionModel {
	private Connection con;
	private DatabaseModel dbm;
	private EventListenerList listenerList;
	private String action_command = "ConnectionModelAction";
	private ConnectionWorker cw = null;
	private String con_error;
	private String dbver;
	private PlannerOptions popts;
	private Component parent;
	private HashMap hdefs;
	private boolean hasDefsMap;
	
	/**
	 * Creates a new <code>ConnectionModel</code> instance.
	 */
	public ConnectionModel(Component parent)
	{
		this.parent = parent;
		con = null;
		dbm = null;
		popts = new PlannerOptions();
	}

	/**
	 * Creates a new <code>ConnectionModel</code> instance.
	 *
	 * @param d a <code>DatabaseModel</code> value
	 */
	public ConnectionModel(Component parent, DatabaseModel d)
	{
		this.parent = parent;
		con = null;
		dbm = d;
		popts = new PlannerOptions();
	}

	/**
	 * Get the DatabaseModel that the ConnectionModel is currently
	 * dealing with.
	 *
	 * @return a <code>DatabaseModel</code> value
	 */
	public synchronized DatabaseModel getDatabaseModel()
	{
		return dbm;
	}

	/**
	 * Set a new DatabaseModel to deal with. Closes previous
	 * connection, if open, but doesn't open a new connection. To do
	 * that, {@link #openConnection} must be invoked.
	 *
	 * @param d a <code>DatabaseModel</code> value
	 */
	public synchronized void setDatabaseModel(DatabaseModel d)
	{
		closeConnection();
		
		// Reset planner options
		popts = new PlannerOptions();
		
		dbm = d;
	}

	/**
	 * Set planner options to be set before a query is run.
	 * Setting a new database model with {@link com.redhat.rhdb.vise.DatabaseModel#DatabaseModel}
	 * resets these values
	 *
	 * @param opts a <code>PlannerOptions</code> object
	 */
	 public void setPlannerOptions(PlannerOptions opts) {
	 	// If a connection is not open (internal error)
		// print something on the console
	 	if (!isConnectionOpen())
			System.out.println("Internal Error: setPlannerOptions called with no connection");
		
		// We we are trying to set something not available
		// on the current version, warn that it is being ignored
		// Require at least 7.4 to have HashAggregate
		if (!((getBackendMajorVersion() > 7) ||
			(getBackendMajorVersion() == 7) && (getBackendMinorVersion() >= 4)) &&
			(opts.enable_hashagg != null))
				JOptionPane.showMessageDialog(parent,
										  ExplainResources.getString(ExplainResources.WARNING_UNSUPPORTED_OPTION,
																	 ExplainResources.getString(ExplainResources.POPTS_LABEL_HASHAGG)),
										  ExplainResources.getString(ExplainResources.WARNING_UNSUPPORTED_OPTION_TITLE),
										  JOptionPane.WARNING_MESSAGE);
	 
	 	this.popts = opts;
	}

	/**
	 * Gets the planner options to be set before the query is run.
	 *
	 * @return a <code>PlannerOptions</code> value
	 */
	public PlannerOptions getPlannerOptions() {
		return popts;
	}

	/**
	 * Gets the default planner options for this connection.
	 *
	 * @return a <code>PlannerOptions</code> value
	 */
	public PlannerOptions getDefaultPlannerOptions() {
		PlannerOptions defopts = new PlannerOptions();

		if (hasDefsMap)
		{
			defopts.enable_seqscan = new Boolean((String)hdefs.get("enable_seqscan"));
			defopts.enable_indexscan = new Boolean((String)hdefs.get("enable_indexscan"));
			defopts.enable_tidscan = new Boolean((String)hdefs.get("enable_tidscan"));
			defopts.enable_sort = new Boolean((String)hdefs.get("enable_sort"));
			// Only new backends have HashAggregate
			if (hdefs.containsKey("enable_hashagg"))
				defopts.enable_hashagg = new Boolean((String)hdefs.get("enable_hashagg"));
			defopts.enable_nestloop = new Boolean((String)hdefs.get("enable_nestloop"));
			defopts.enable_mergejoin = new Boolean((String)hdefs.get("enable_mergejoin"));
			defopts.enable_hashjoin = new Boolean((String)hdefs.get("enable_hashjoin"));
			defopts.geqo = new Boolean((String)hdefs.get("geqo"));
			defopts.geqo_threshold = new Integer((String)hdefs.get("geqo_threshold"));
			defopts.geqo_pool_size = new Integer((String)hdefs.get("geqo_pool_size"));
			defopts.geqo_effort = new Integer((String)hdefs.get("geqo_effort"));
			defopts.geqo_generations = new Integer((String)hdefs.get("geqo_generations")); 
			defopts.random_page_cost = new Float((String)hdefs.get("random_page_cost"));
			defopts.cpu_tuple_cost = new Float((String)hdefs.get("cpu_tuple_cost"));
			defopts.cpu_index_tuple_cost = new Float((String)hdefs.get("cpu_index_tuple_cost"));
			defopts.cpu_operator_cost = new Float((String)hdefs.get("cpu_operator_cost"));
			defopts.geqo_selection_bias = new Float((String)hdefs.get("geqo_selection_bias"));
		}
		else
		{
			// For backends older than 7.3 we need to use hardcoded values
			defopts.enable_seqscan = new Boolean(true);
			defopts.enable_indexscan = new Boolean(true);
			defopts.enable_tidscan = new Boolean(true);
			defopts.enable_sort = new Boolean(true);
			// Only new backends have HashAggregate
			// Require at least 7.4
			if ((getBackendMajorVersion() > 7) ||
				(getBackendMajorVersion() == 7) && (getBackendMinorVersion() >= 4))
				defopts.enable_hashagg = new Boolean(true);
			defopts.enable_nestloop = new Boolean(true);
			defopts.enable_mergejoin = new Boolean(true);
			defopts.enable_hashjoin = new Boolean(true);
			defopts.geqo = new Boolean(true);
			defopts.geqo_threshold = new Integer(11); // DEFAULT_GEQO_RELS
			defopts.geqo_pool_size = new Integer(0); // DEFAULT_GEQO_POOL_SIZE
			defopts.geqo_effort = new Integer(1);
			defopts.geqo_generations = new Integer(0); 
			defopts.random_page_cost = new Float(4.0); // DEFAULT_RANDOM_PAGE_COST
			defopts.cpu_tuple_cost = new Float(0.01); // DEFAULT_CPU_TUPLE_COST
			defopts.cpu_index_tuple_cost = new Float(0.001); // DEFAULT_CPU_INDEX_TUPLE_COST
			defopts.cpu_operator_cost = new Float(0.0025); // DEFAULT_CPU_OPERATOR_COST
			defopts.geqo_selection_bias = new Float(2.0); // DEFAULT_GEQO_SELECTION_BIAS
		}
		
		return defopts;
	}

	/**
	 * Open a connection. If a connection is already open, close that
	 * connection first. Connection opening is done in a seperate
	 * thread. One a connection is open, an ActionEvent will be
	 * generated.
	 */
	public synchronized void openConnection()
	{
		closeConnection();
		
		if (dbm == null)
			return;

		// ConnectionWorker opens a connection in a seperate thread and
		// then notifies listeners in the event thread
		if (cw == null)
		{
			cw = new ConnectionWorker();
			cw.start();
		}
	}

	/**
	 * Close the current connection, if there is one open. Connection
	 * closing is <em>not</em> done in a seperate thread.
	 */
	public synchronized void closeConnection()
	{
		if (con == null || dbm instanceof DisconnectedDatabaseModel)
			return;

		try {
			if (con.isClosed())
				return;
		} catch (SQLException e) {
			e.printStackTrace();
		}
		
		try {
			con.close();
		} catch (SQLException e) {
			// No need to make noise here
			// e.printStackTrace();
		}
		
		// Reset the backend version field
		dbver = null;
	}

	/**
	 * Is there a connection currently open?
	 *
	 * @return a <code>boolean</code> value
	 */
	public synchronized boolean isConnectionOpen()
	{
		boolean rv = false;

		if (con == null || dbm instanceof DisconnectedDatabaseModel)
			return rv;

		try {
			if (! con.isClosed())
				rv = true;
		} catch (SQLException e) {
			e.printStackTrace();
		}

		return rv;
	}

	/**
	 * Get the current connection.
	 *
	 * @return a <code>Connection</code> value
	 */
	public synchronized Connection getConnection()
	{
		return con;
	}

	/**
	 * Get any connection errors that occurred on connection opening.
	 *
	 * @return a <code>String</code> value
	 */
	public synchronized String getConnectionError()
	{
		return con_error;
	}

	/**
	 * Get the version of the database we've connected to.
	 *
	 * @return a <code>String</code> value
	 */
	public synchronized String getBackendVersion()
	{
		return dbver;
	}

	/**
	 * Get the major version of the database we've connected to.
	 *
	 * @return a <code>int</code> value
	 */
	public synchronized int getBackendMajorVersion()
	{
		String major = dbver.substring(0, dbver.indexOf("."));
		int mver = 0;
		
		try {
			mver = Integer.parseInt(major);
		} catch (NumberFormatException nfe) {
		}
		
		return mver;
	}

	/**
	 * Get the minor version of the database we've connected to.
	 *
	 * @return a <code>int</code> value
	 */
	public synchronized int getBackendMinorVersion()
	{
		int start = dbver.indexOf(".") + 1;
		int end = start;
		int len = dbver.length();
		for (int i = end; i < len; i++)
		{
			if (!Character.isDigit(dbver.charAt(i)))
				break;
			end++;
		}

		String minor = dbver.substring(start, end);
		int mver = 0;

		try {
			mver = Integer.parseInt(minor);
		} catch (NumberFormatException nfe) {
			// What a sick backend would that be...
			System.out.println("Invalid backend version: " + dbver);
		}
		
		return mver;
	}

	/**
	 * Add an ActionListener.
	 *
	 * @param l an <code>ActionListener</code> value
	 */
	public void addActionListener(ActionListener l)
	{
		if (listenerList == null)
		{
			listenerList = new EventListenerList();
		}

		listenerList.add(ActionListener.class, l);
	}

	/**
	 * Remove an ActionListener.
	 *
	 * @param l an <code>ActionListener</code> value
	 */
	public void removeActionListener(ActionListener l)
	{
		if (listenerList == null)
			return;

		listenerList.remove(ActionListener.class, l);
	}

	/**
	 * Set the action command of generated ActionEvents.
	 *
	 * @param s a <code>String</code> value
	 */
	public void setActionCommand(String s)
	{
		action_command = s;
	}
	
	/**
	 * Get the action command of generated ActionEvents.
	 *
	 * @return a <code>String</code> value
	 */
	public String getActionCommand()
	{
		return action_command;
	}

	//
	// private methods
	//

	private void notifyListeners()
	{
		if (listenerList != null)
		{
			Object[] listeners = listenerList.getListenerList();

			for (int i = 1; i < listeners.length; i += 2)
			{
				ActionEvent ev = new ActionEvent(this, ActionEvent.ACTION_PERFORMED, action_command);
				((ActionListener) listeners[i]).actionPerformed(ev);
			}
		}
	}
	
	//
	// inner classes
	//

	private class ConnectionWorker extends Worker {
		public synchronized Object construct()
		{
			if (dbm instanceof DisconnectedDatabaseModel)
			{
				// do nothing
			}
			else if (dbm != null)
			{
				con_error = null;
				// Try and connect to the backend
				try {
					Class.forName(dbm.getDriverClass());
					con = DriverManager.getConnection(dbm.getURL(), dbm.getUser(), dbm.getPassword());
				} catch (ClassNotFoundException ex) {
					con_error = "JDBC Driver (" + dbm.getDriverClass() + ") could not be found.";
				} catch (SQLException ex) {
					con_error = ex.getMessage();
				} catch (Exception ex) {
					con_error = ex.getMessage();
				}
				if (con_error == null)
				{
					// Now get and save the backend version
					try {
						DatabaseMetaData dbmd = con.getMetaData();
						dbver = dbmd.getDatabaseProductVersion();
					} catch (SQLException ex) {
						con_error = ex.getMessage();
					} catch (Exception ex) {
						con_error = ex.getMessage();
					}
				}
				// Must get the default planner options from the backend
				// if possible (backend version 7.3 or newer)
				hasDefsMap = false;
				if ((con_error == null) &&
					(dbver.startsWith("7.3") || dbver.startsWith("7.4")))
				{
					// Now get and save the backend version
					try {
						Statement stmt = con.createStatement();
						ResultSet rs = stmt.executeQuery("SELECT * FROM pg_settings");
						hdefs = new HashMap();
						while (rs.next())
						{
							hdefs.put(rs.getString(1), rs.getString(2));
						}
						hasDefsMap = true;
					} catch (SQLException ex) {
						con_error = ex.getMessage();
					} catch (Exception ex) {
						con_error = ex.getMessage();
					}
				}
				if (con_error == null)
					setPlannerOptions(new PlannerOptions(getDefaultPlannerOptions()));
			}
			return "Finished!";
		}

		public synchronized void finished()
		{
			// need to notify somebody that we've made a connection
			cw = null;
			notifyListeners();
		}
	}
	
}// ConnectionModel
