#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include "config.h"
#include "xmmplayer.h"

GtkWidget *videowin = NULL;
Window videocanvas;
static GdkPixmap *videowin_bg;
static GdkGC *videowin_gc;
gboolean videowin_focus = FALSE, videowin_resizing = FALSE;
gint videowin_resize_x = 0, videowin_resize_y = 0;
PButton *videowin_shade, *videowin_close;
static GList *videowin_wlist = NULL;
gboolean show_wm_decorations = FALSE;

#define VIDEOWIN_WIDTH xmmplayer_config.videowin_width
#define VIDEOWIN_HEIGHT xmmplayer_config.videowin_height
#define VIDEOWIN_X xmmplayer_config.videowin_x
#define VIDEOWIN_Y xmmplayer_config.videowin_y
#define VIDEOWIN_VISIBLE xmmplayer_config.videowin_visible
#define VIDEOWIN_AUTOCLOSE xmmplayer_config.videowin_autoclose

static struct
{
  int w, h;
} videowin_resizeq =
{
-1, -1};

void util_set_cursor (GtkWidget * window);
void videowin_draw_frame (void);

void
videowin_raise (void)
{
  gdk_window_raise (videowin->window);
}

void
videowin_resize (int width, int height)
{
  gint bx, nw;
  GdkPixmap *oldbg;

  bx = (width - 50) / 25;
  nw = (bx * 25) + 50;
  if (nw < 50)
    nw = 50;

  if (height < 18)
    height = 18;

  if (nw == VIDEOWIN_WIDTH && height == VIDEOWIN_HEIGHT)
    return;

  VIDEOWIN_WIDTH = nw;
  VIDEOWIN_HEIGHT = height;

  oldbg = videowin_bg;
  videowin_bg = gdk_pixmap_new (NULL, VIDEOWIN_WIDTH,
				VIDEOWIN_HEIGHT,
				gdk_rgb_get_visual ()->depth);
  gdk_window_set_back_pixmap (videowin->window, videowin_bg, 0);
  widget_list_change_pixmap (videowin_wlist, videowin_bg);
  gdk_pixmap_unref (oldbg);
  XResizeWindow (GDK_DISPLAY (), videocanvas, VIDEOWIN_WIDTH,
		 VIDEOWIN_HEIGHT - 18);
  XClearWindow (GDK_DISPLAY (), videocanvas);

  move_widget (videowin_shade, VIDEOWIN_WIDTH - 21, 3);
  move_widget (videowin_close, VIDEOWIN_WIDTH - 11, 3);

  videowin_draw_frame ();
}

static gboolean
videowin_resize_handler (gpointer data)
{
  GDK_THREADS_ENTER ();
  videowin_resize (videowin_resizeq.w, videowin_resizeq.h);

  videowin_resizeq.w = -1;
  videowin_resizeq.h = -1;

  GDK_THREADS_LEAVE ();
  return FALSE;
}

void
videowin_queue_resize (int width, int height)
{
  if (videowin_resizeq.w == -1)
    g_idle_add_full (G_PRIORITY_HIGH_IDLE,
		     videowin_resize_handler, NULL, NULL);

  videowin_resizeq.w = width;
  videowin_resizeq.h = height;
}

static gboolean
inside_sensitive_widgets (gint x, gint y)
{
  return inside_widget (x, y, videowin_close) ||
    inside_widget (x, y, videowin_shade);
}

void
videowin_release (GtkWidget * widget, GdkEventButton * event,
		  gpointer callback_data)
{
  if (event->button == 3)
    return;

  gdk_pointer_ungrab (GDK_CURRENT_TIME);
  gdk_flush ();

  if (videowin_resizing)
    {
      videowin_resizing = FALSE;
    }
  else if (dock_is_moving (videowin))
    {
      dock_move_release (videowin);
    }
  else
    {
      handle_release_cb (videowin_wlist, widget, event);
      videowin_draw_frame ();
    }
}

void
videowin_press (GtkWidget * widget, GdkEventButton * event,
		gpointer callback_data)
{
  gboolean grab = TRUE;
  int xpos, ypos;

  dock_get_widget_pos (videowin, &xpos, &ypos);

  if ((event->button == 1) && (!show_wm_decorations)
      && event->y > (VIDEOWIN_HEIGHT - 5))
    {
      if (hint_move_resize_available ())
	{
	  hint_move_resize (videowin, event->x_root, event->y_root, FALSE);
	  grab = FALSE;
	}
      else
	{
	  videowin_resizing = TRUE;
	  videowin_resize_x = VIDEOWIN_WIDTH - event->x;
	  videowin_resize_y = VIDEOWIN_HEIGHT - event->y;
	  videowin_raise ();
	}
    }
  else if (event->button == 1 && event->type == GDK_BUTTON_PRESS &&
	   (cfg.easy_move || event->y < 14)
	   && !inside_sensitive_widgets (event->x, event->y))
    {
      gdk_window_raise (videowin->window);
      dock_move_press (dock_window_list, videowin, event, FALSE);
    }
  else
    {
      handle_press_cb (videowin_wlist, widget, event);
      videowin_draw_frame ();
    }

  if (grab)
    gdk_pointer_grab (videowin->window, FALSE,
		      GDK_BUTTON_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
		      GDK_NONE, GDK_NONE, GDK_CURRENT_TIME);
}

void
videowin_motion (GtkWidget * widget, GdkEventMotion * event,
		 gpointer callback_data)
{
  XEvent ev;

  if (videowin_resizing)
    {
      videowin_resize (event->x + videowin_resize_x,
		       event->y + videowin_resize_y);
      gdk_window_set_hints (videowin->window, 0, 0,
			    VIDEOWIN_WIDTH, VIDEOWIN_HEIGHT,
			    VIDEOWIN_WIDTH, VIDEOWIN_HEIGHT,
			    GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE);
      gdk_window_resize (videowin->window, VIDEOWIN_WIDTH, VIDEOWIN_HEIGHT);
      gtk_widget_set_usize (videowin, VIDEOWIN_WIDTH, VIDEOWIN_HEIGHT);
    }
  else if (dock_is_moving (videowin))
    {
      dock_move_motion (videowin, event);
    }
  else
    {
      handle_motion_cb (videowin_wlist, widget, event);
      videowin_draw_frame ();
    }
  gdk_flush ();
  while (XCheckMaskEvent (GDK_DISPLAY (), ButtonMotionMask, &ev));
}

static gboolean
videowin_configure (GtkWidget * window, GdkEventConfigure * event,
		    gpointer data)
{
  if (!GTK_WIDGET_VISIBLE (window))
    return FALSE;

  if (show_wm_decorations || hint_move_resize_available ())
    {
      if (event->width != VIDEOWIN_WIDTH || event->height != VIDEOWIN_HEIGHT)
	videowin_queue_resize (event->width, event->height);
    }

  if (show_wm_decorations)
    gdk_window_get_root_origin (window->window, &VIDEOWIN_X, &VIDEOWIN_Y);
  else
    gdk_window_get_deskrelative_origin (window->window,
					&VIDEOWIN_X, &VIDEOWIN_Y);

  return FALSE;
}

void
videowin_focus_in (GtkWidget * widget, GdkEvent * event,
		   gpointer callback_data)
{
  videowin_close->pb_allow_draw = TRUE;
  videowin_shade->pb_allow_draw = TRUE;
  videowin_focus = TRUE;
  videowin_draw_frame ();
}

void
videowin_focus_out (GtkWidget * widget, GdkEventButton * event,
		    gpointer callback_data)
{
  videowin_close->pb_allow_draw = FALSE;
  videowin_shade->pb_allow_draw = FALSE;
  videowin_focus = FALSE;
  videowin_draw_frame ();
}

void
videowin_set_hints (void)
{
  GdkGeometry geometry;
  GdkWindowHints mask;

  if (!show_wm_decorations && !hint_move_resize_available ())
    return;

  geometry.min_width = 50;
  geometry.base_width = 50;
  geometry.width_inc = 25;
  geometry.height_inc = 1;
  geometry.max_width = (1 << 16) - 1;
  geometry.max_height = (1 << 16) - 1;
  mask = GDK_HINT_MIN_SIZE | GDK_HINT_RESIZE_INC |
    GDK_HINT_MAX_SIZE | GDK_HINT_BASE_SIZE;
  geometry.min_height = 18;
  geometry.base_height = 18;

  gtk_window_set_geometry_hints (GTK_WINDOW (videowin),
				 videowin, &geometry, mask);
}

void videowin_close_cb (void);

gint
videowin_delete (GtkWidget * w, gpointer data)
{
  videowin_close_cb ();
  return TRUE;
}

void
videowin_create_gtk (void)
{
  videowin = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  dock_add_window (dock_window_list, videowin);
  gtk_widget_set_app_paintable (videowin, TRUE);
  show_wm_decorations = cfg.show_wm_decorations;
  if (show_wm_decorations)
    gtk_window_set_policy (GTK_WINDOW (videowin), TRUE, TRUE, FALSE);
  else
    gtk_window_set_policy (GTK_WINDOW (videowin), FALSE, FALSE, TRUE);
  gtk_window_set_title (GTK_WINDOW (videowin), "XMMPlayer");
  gtk_window_set_wmclass (GTK_WINDOW (videowin), "XMMPlayer", "xmms");
  gtk_widget_set_usize (videowin, VIDEOWIN_WIDTH, VIDEOWIN_HEIGHT);
  gtk_widget_set_uposition (videowin, VIDEOWIN_X, VIDEOWIN_Y);
  gtk_widget_set_events (videowin,
			 GDK_FOCUS_CHANGE_MASK | GDK_BUTTON_MOTION_MASK |
			 GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
  gtk_widget_realize (videowin);
  gdk_window_set_icon_name (videowin->window, "XMMPlayer");
  videowin_set_hints ();
  util_set_cursor (videowin);

  gtk_signal_connect (GTK_OBJECT (videowin), "button_press_event",
		      GTK_SIGNAL_FUNC (videowin_press), NULL);
  gtk_signal_connect (GTK_OBJECT (videowin), "button_release_event",
		      GTK_SIGNAL_FUNC (videowin_release), NULL);
  gtk_signal_connect (GTK_OBJECT (videowin), "motion_notify_event",
		      GTK_SIGNAL_FUNC (videowin_motion), NULL);
  gtk_signal_connect (GTK_OBJECT (videowin), "focus_in_event",
		      GTK_SIGNAL_FUNC (videowin_focus_in), NULL);
  gtk_signal_connect (GTK_OBJECT (videowin), "focus_out_event",
		      GTK_SIGNAL_FUNC (videowin_focus_out), NULL);
  gtk_signal_connect (GTK_OBJECT (videowin), "configure_event",
		      GTK_SIGNAL_FUNC (videowin_configure), NULL);
  gtk_signal_connect (GTK_OBJECT (videowin), "delete_event",
		      GTK_SIGNAL_FUNC (videowin_delete), NULL);
/*	gtk_signal_connect(GTK_OBJECT(videowin), "client_event", GTK_SIGNAL_FUNC(videowin_client_event), NULL);
	xmms_drag_dest_set(videowin);
	gtk_signal_connect(GTK_OBJECT(videowin), "drag-data-received", GTK_SIGNAL_FUNC(videowin_drag_data_received), NULL);
	gtk_signal_connect(GTK_OBJECT(videowin), "key-press-event", GTK_SIGNAL_FUNC(videowin_keypress), NULL);
	gtk_signal_connect(GTK_OBJECT(videowin), "selection_received", GTK_SIGNAL_FUNC(selection_received), NULL); */

  if (!show_wm_decorations)
    gdk_window_set_decorations (videowin->window, 0);

  gdk_window_set_back_pixmap (videowin->window, videowin_bg, 0);
}

void
videowin_create_canvas (void)
{
  long black = BlackPixel (GDK_DISPLAY (), DefaultScreen (GDK_DISPLAY ()));

  videocanvas =
    XCreateSimpleWindow (GDK_DISPLAY (),
			 GDK_WINDOW_XWINDOW (videowin->window), 0, 14,
			 VIDEOWIN_WIDTH, VIDEOWIN_HEIGHT - 18, 0, 0, black);
  XMapWindow (GDK_DISPLAY (), videocanvas);
}

void
videowin_draw_frame (void)
{
  int c, y, w = VIDEOWIN_WIDTH, h = VIDEOWIN_HEIGHT, i;
  SkinIndex src = SKIN_PLEDIT;
  gboolean redraw = FALSE;

  if (videowin_focus || !cfg.dim_titlebar)
    y = 0;
  else
    y = 21;

  // title, left corner
  skin_draw_pixmap (videowin_bg, videowin_gc, src, 0, y, 0, 0, 25, 14);

  // title, center
  c = (w - 50) / 25;
  for (i = 0; i < c; i++)
    skin_draw_pixmap (videowin_bg, videowin_gc, src, 127, y, (i * 25) + 25, 0,
		      25, 16);

  // title, right corner
  skin_draw_pixmap (videowin_bg, videowin_gc, src, 153, y, w - 25, 0, 25, 16);

  // bottom
  c = w / 25;
  for (i = 0; i < c; i++)
    skin_draw_pixmap (videowin_bg, videowin_gc, src, 127, 10, (i * 25), h - 4,
		      25, 4);

  lock_widget_list (videowin_wlist);
  draw_widget_list (videowin_wlist, &redraw, FALSE);
  unlock_widget_list (videowin_wlist);

  gdk_window_clear (videowin->window);

  gdk_flush ();
}

void
videowin_hide (void)
{
  if (!videowin)
    return;
  gtk_widget_hide (videowin);
}

void
videowin_close_cb (void)
{
  if (decode_thread != -1)
  	xmms_remote_stop(ctrlsocket_get_session_id());
  videowin_hide ();
/* Not needed no more.
    if (decode_thread == -1)
    videowin_hide (); 
*/
}

void
videowin_iconify (void)
{
  Window xwindow;
  if (!videowin->window)
    return;
  xwindow = GDK_WINDOW_XWINDOW (videowin->window);
  XIconifyWindow (GDK_DISPLAY (), xwindow, DefaultScreen (GDK_DISPLAY ()));
}

void
videowin_create (void)
{
  if (videowin)
    return;

  videowin_bg = gdk_pixmap_new (NULL, VIDEOWIN_WIDTH,
				VIDEOWIN_HEIGHT,
				gdk_rgb_get_visual ()->depth);
  videowin_create_gtk ();
  videowin_gc = gdk_gc_new (videowin->window);
  videowin_create_canvas ();
  gtk_widget_show_all (videowin);

  videowin_close =
    create_pbutton (&videowin_wlist, videowin_bg, videowin_gc,
		    VIDEOWIN_WIDTH - 11, 3, 9, 9, 167, 3, 52, 42,
		    videowin_close_cb, SKIN_PLEDIT);
  videowin_close->pb_allow_draw = FALSE;
  videowin_shade =
    create_pbutton (&videowin_wlist, videowin_bg, videowin_gc,
		    VIDEOWIN_WIDTH - 21, 3, 9, 9, 157, 3, 62, 42,
		    videowin_iconify, SKIN_PLEDIT);
  videowin_shade->pb_allow_draw = FALSE;

  videowin_draw_frame ();
}

void
videowin_destroy (void)
{
  if (!videowin)
    return;

  gtk_widget_destroy (videowin);
  gdk_pixmap_unref (videowin_bg);
  gdk_gc_destroy (videowin_gc);
}

void
videowin_show (void)
{
  if (!videowin)
    return;
  if (!GTK_WIDGET_VISIBLE (videowin))
    gtk_widget_show (videowin);
  else
    videowin_raise ();
}

void
videowin_force_resize (int w, int h, float aspect)
{
  int nw, nh;
  nw = w + (25 - (w % 25));
  if (h > -1)
    nh = (nw * h) / w;
  else
    nh = (int) ((float) nw / aspect);
  videowin_resize (nw, nh + 18);
  gtk_widget_set_usize (videowin, VIDEOWIN_WIDTH, VIDEOWIN_HEIGHT);
  gdk_flush ();
}
