# Part of the A-A-P recipe executive: The parse position class

# Copyright (C) 2002-2003 Stichting NLnet Labs
# Permission to copy and use this file is specified in the file COPYING.
# If this file is missing you can find it here: http://www.a-a-p.org/COPYING

# A ParsePos keeps the position in a file or string where the parser is
# working.

import string

from Error import *
from Util import *
from Process import get_line_marker


class ParsePos:
    """Object to remember a file OR string being parsed, the last line read and
    the position in that line.
    Can't have a "file" and "string" at the same time."""

    def __init__(self, rpstack, file = '', string = ''):
        self.file = file        # file being read
        self.string = string    # string to be parsed
        self.string_idx = 0     # index in string, start of next line
        self.string_len = len(string)
        self.line = ''          # current line (concatenated file lines)
        self.idx = 0            # parsing index in "line"
        self.rpstack = rpstack  # stack of recipes, rpstack[-1] is the current
                                # one and is the current line number in file;
                                # when parsing a string it's offset by the line
                                # number in the recipe where the string came
                                # from

    def getlnum(self):
        """Get the line number of the recipe being parsed."""
        return self.rpstack[-1].line_nr

    def nextline(self):
        """Read a line from "self.file" and handle line continuation.
           When reading a string use "self.string".
           Puts the concatenated line in "self.line", with EOL and backslashes
           removed.
           Returns None in "self.line" when at the end of the file or string.
           Increases the line number self.rpstack[-1].line_nr.
           Skips over empty and comment lines.
           Throws an exception when the last line has a backslash or when
           there is a read error."""

        def getline(fp):
            """Get one line from the file or the string.  Includes the newline
               at the end of the line."""
            if fp.string:
                if fp.string_idx >= len(fp.string):     # end of the string
                    return None
                i = string.find(fp.string, "\n", fp.string_idx)
                if i < 0:
                    line = fp.string[fp.string_idx:] + '\n'
                    i = len(fp.string)
                else:
                    i = i + 1
                    line = fp.string[fp.string_idx:i]
                fp.string_idx = i
                return line
            else:
                line = fp.file.readline()
                if line and line[-1] != '\n':
                    return line + '\n'      # add missing newline
                return line


        try:
            nonwhite = -1
            while 1:
                self.line = getline(self)
                self.rpstack[-1].line_nr = self.rpstack[-1].line_nr + 1
                if not self.line:
                    # Reached end of file
                    self.line = None
                    break
                # concatenate lines ending in a backslash
                while 1:
                    # Remove the trailing NL or CR-NL.
                    # A CR-NL comes from reading a DOS file on Unix.
                    line_len = len(self.line) - 1
                    if line_len > 0 and self.line[line_len - 1] == '\r':
                        self.line = self.line[:line_len - 1]
                        line_len = line_len - 1
                    else:
                        self.line = self.line[:line_len]
                    if line_len < 1 or self.line[line_len - 1] != '\\':
                        break
                    nextline = getline(self)
                    self.rpstack[-1].line_nr = self.rpstack[-1].line_nr + 1
                    if not nextline:
                        from Process import recipe_error
                        recipe_error(self.rpstack,
                                               'last line ends in a backslash')

                    # When the line starts with '@' remove leading '@' from the
                    # continuation line.
                    if nonwhite < 0:
                        nonwhite = skip_white(self.line, 0)
                    if self.line[nonwhite] == '@':
                        i = skip_white(nextline, 0)
                        if i < len(nextline) and nextline[i] == '@':
                            nextline = nextline[i + 1:]

                    self.line = self.line[:-1] + nextline

                # stop reading lines when found a non-blank non-comment line
                i = skip_white(self.line, 0)
                if i < len(self.line):
                    if self.line[i] == '#':
                        # Check for a line marker, used in build commands to
                        # correct for comment and empty lines.
                        n = get_line_marker(self.line, i)
                        if not n is None:
                            self.rpstack[-1].line_nr = n - 1
                    else:
                        break

        except IOError, e:
            raise UserError, _('Cannot read from "') \
                    + self.file.name + '": ' + str(e)

        self.idx = 0
        if not self.line is None:
            self.line_len = len(self.line)


# vim: set sw=4 et sts=4 tw=79 fo+=l:
