;;;***************************************************************
;;;An ACL2 Library of Floating Point Arithmetic

;;;Advanced Micro Devices, Inc.
;;;June 2001
;;;***************************************************************

(in-package "ACL2")

(local (include-book "../support/top"))


;;;**********************************************************************
;;;                       Naturals
;;;**********************************************************************

(in-theory (disable natp))

(defthm natp-compound-recognizer
  (equal (natp x)
         (and (integerp x)
              (>= x 0)))
  :rule-classes :compound-recognizer)

(defthm natp>=0
    (implies (natp x)
	     (>= x 0)))

(defthm natp+
    (implies (and (natp x) (natp y))
	     (natp (+ x y))))

(defthm natp-
    (implies (and (natp x)
		  (natp y)
		  (>= x y))
	     (natp (+ x (* -1 y)))))

(defthm natp*
    (implies (and (natp x) (natp y))
	     (natp (* x y))))

(defun natp-induct (k)
  (if (zp k)
      t
    (natp-induct (1- k))))

(defthm abs-type 
  (>= (abs x) 0)
  :rule-classes :type-prescription)


;;;**********************************************************************
;;;                       FLOOR and CEILING
;;;**********************************************************************

(defun fl (x) (floor x 1))

(defun cg (x) (- (fl (- x))))

(in-theory (disable fl cg))

#|
(defthm floor-fl
    (implies (and (integerp m)
		  (integerp n)
		  (> n 0)
		  (>= m 0))
	     (= (floor m n) (fl (/ m n))))
  :rule-classes ())
|#

;new
(DEFTHM FLOOR-FL
  (EQUAL (FLOOR M N) (FL (/ M N))))

;new
(in-theory (disable floor-fl))

(defthm int-fl-rules
    (implies (rationalp x)
	     (integerp (fl x)))
    :rule-classes (:rewrite :type-prescription))

(defthm int-cg-rules
    (implies (rationalp x)
	     (integerp (cg x)))
    :rule-classes (:rewrite :type-prescription))

#|
(defthm fl-int-2
    (implies (rationalp x)
	     (iff (equal (fl x) x)
		  (integerp x))))
|#

;new
(DEFTHM FL-INT-2
  (EQUAL (EQUAL (FL X) X) (INTEGERP X)))

(defthm cg-int-2
    (implies (rationalp x)
	     (iff (equal (cg x) x)
		  (integerp x))))

(defthm fl-def-linear
    (implies (case-split (rationalp x))
	     (and (<= (fl x) x)
		  (< x (1+ (fl x)))))
  :rule-classes :linear)

(defthm cg-def-linear
    (implies (rationalp x)
	     (and (>= (cg x) x)
		  (> (1+ x) (cg x))))
  :rule-classes :linear)

(defthm fl-unique
    (implies (and (rationalp x)
		  (integerp n)
		  (<= n x)
		  (< x (1+ n)))
	     (equal (fl x) n))
  :rule-classes ())

(defthm cg-unique
    (implies (and (rationalp x)
		  (integerp n)
		  (>= n x)
		  (> (1+ x) n))
	     (equal (cg x) n))
  :rule-classes ())

(defthm fl-monotone-linear
    (implies (and (<= x y)
		  (rationalp x)
		  (rationalp y))
	     (<= (fl x) (fl y)))
  :rule-classes :linear)

(defthm cg-monotone-linear
    (implies (and (rationalp x)
		  (rationalp y)
		  (<= x y))
	     (<= (cg x) (cg y)))
  :rule-classes :linear)

(defthm n<=fl-linear
    (implies (and (<= n x)
		  (rationalp x)
		  (integerp n))
	     (<= n (fl x)))
  :rule-classes :linear)

(defthm n>=cg-linear
    (implies (and (>= n x)
		  (rationalp x)
		  (integerp n))
	     (>= n (cg x)))
  :rule-classes :linear)

(defthm fl+int-rewrite
    (implies (and (integerp n)
		  (rationalp x))
	     (equal (fl (+ x n)) (+ (fl x) n))))

(defthm cg+int-rewrite
    (implies (and (integerp n)
		  (rationalp x))
	     (equal (cg (+ x n)) (+ (cg x) n))))

(defthm fl-cg
    (implies (and (rationalp x)
		  (not (integerp x)))
	     (equal (cg x) (1+ (fl x))))
  :rule-classes ())

(defthm fl/int-rewrite
    (implies (and (integerp n)
		  (> n 0)
		  (rationalp x))
	     (equal (fl (/ (fl x) n))
		    (fl (/ x n)))))

(defthm cg/int-rewrite
    (implies (and (integerp n)
		  (> n 0)
		  (rationalp x))
	     (equal (cg (/ (cg x) n))
		    (cg (/ x n)))))

(defthm floor-m+1
    (implies (and (integerp m)
		  (integerp n)
		  (>= m 0)
		  (> n 0))
	     (= (fl (- (/ (1+ m) n)))
		(1- (- (fl (/ m n))))))
  :rule-classes ())

(defthm floor-2
    (implies (integerp m)
	     (= (fl (- (/ (1+ m) 2)))
		(1- (- (fl (/ m 2))))))
  :rule-classes ())


;;;**********************************************************************
;;;                       EXPONENTIATION
;;;**********************************************************************

(defthm expt-2-positive-rational-type
  (and (rationalp (expt 2 i))
       (< 0 (expt 2 i)))
  :rule-classes ((:type-prescription :typed-term (expt 2 i))))

(defthm expt-2-positive-integer-type
  (implies (<= 0 i)
           (and (integerp (expt 2 i))
                (< 0 (expt 2 i))))
  :rule-classes (:type-prescription))

;the rewrite rule counterpart to expt-2-positive-integer-type
(defthm expt-2-integerp
  (implies (<= 0 i)
           (integerp (expt 2 i))))

(defthm expt-2-type-linear
  (implies (<= 0 i)
           (<= 1 (expt 2 i)))
  :rule-classes ((:linear :trigger-terms ((expt 2 i)))))


#|
(defthm expt-pos
    (implies (integerp x)
	     (> (expt 2 x) 0)))

(in-theory (disable expt-pos))
|#

;perhaps use only expt-2-positive-integer-type
(defthm natp-expt
  (implies (natp n)
           (and (integerp (expt 2 n))
                (< 0 (expt 2 n))))
  :rule-classes (:type-prescription :rewrite))


(defthm expt>=1
    (implies (and (integerp n)
		  (>= n 0))
	     (and (integerp (expt 2 n))
		  (>= (expt 2 n) 1)))
  :rule-classes ())

(defthm expt-weak-monotone
    (implies (and (integerp n)
		  (integerp m))
	     (iff (<= n m)
		  (<= (expt 2 n) (expt 2 m))))
  :rule-classes ())

(defthm expt-strong-monotone
    (implies (and (integerp n)
		  (integerp m))
	     (iff (< n m)
		  (< (expt 2 n) (expt 2 m))))
  :rule-classes ())

(defthm expt+
    (implies (and (integerp n)
		  (integerp m))
	     (= (* (expt 2 m) (expt 2 n))
		(expt 2 (+ m n))))		
  :rule-classes ())

(defthm expt-
    (implies (and (integerp a)
		  (integerp b))
	     (= (/ (expt 2 a) (expt 2 b)) 
		(expt 2 (- a b))))
  :rule-classes ())

(defthm ash-rewrite
    (implies (integerp n)
	     (equal (ash n i)
		    (fl (* n (expt 2 i))))))


;;;**********************************************************************
;;;                         REMAINDERS
;;;**********************************************************************


; Note, all occurence of rem have been replaced by mod to allow nicer rules about bits, etc.

(in-theory (disable mod))

(in-theory (disable natp))

(defthm natp-integerp
    (implies (natp x)
	     (integerp x)))

(defthm integerp-rationalp
    (implies (integerp x)
	     (rationalp x)))

(defthm mod-0
  (implies (natp m)
           (equal (mod m 0) m)))
#|
(defthm rationalp-mod
    (implies (and (rationalp m)
		  (rationalp n))
	     (rationalp (mod m n)))
  :rule-classes :type-prescription)
|#

;new
(DEFTHM RATIONALP-MOD
  (IMPLIES (CASE-SPLIT (RATIONALP M))
           (RATIONALP (MOD M N)))
  :RULE-CLASSES
  (:REWRITE :TYPE-PRESCRIPTION))

(defthm rationalp-mod-rewrite
    (implies (and (rationalp m)
		  (rationalp n))
	     (rationalp (mod m n))))

#|
(defthm integerp-mod
    (implies (and (integerp m)
		  (integerp n))
	     (integerp (mod m n)))
  :rule-classes :type-prescription)

(defthm integerp-mod-rewrite
    (implies (and (integerp m)
		  (integerp n))
	     (integerp (mod m n))))
|#

;new (combined the 2 rules above into one; maybe not a good idea?)
(defthm integerp-mod
  (implies (and (integerp m) (integerp n))
           (integerp (mod m n)))
  :rule-classes
  (:rewrite :type-prescription)
  :hints
  (("goal" :in-theory (enable mod))))

(defthm natp-mod
  (implies (and (natp m)
                (natp n))
           (natp (mod m n)))
  :rule-classes :type-prescription)

(defthm natp-mod-rewrite
  (implies (and (natp m)
                (natp n))
           (natp (mod m n))))

(defthm mod-bnd-1
    (implies (and (natp m)
		  (natp n)
		  (not (= n 0)))
	     (< (mod m n) n))
  :rule-classes :linear)

(defthm mod-bnd-2
    (implies (and (natp m)
		  (natp n))
	     (<= (mod m n) m))
  :rule-classes :linear)

(defthm quot-mod
    (implies (and (natp m)
		  (natp n))
	     (equal (+ (* n (fl (/ m n))) (mod m n))
		    m))
  :rule-classes ())

(defthm mod-mult
    (implies (and (natp m)
		  (natp a)
		  (natp n))
	     (equal (mod (+ m (* a n)) n)
		    (mod m n))))

(defthm mod-sum
    (implies (and (natp a)
		  (natp b)
		  (natp n))
	     (equal (mod (+ a (mod b n)) n)
		    (mod (+ a b) n))))

(in-theory (disable mod-sum))

(defthm mod-mod-sum
    (implies (and (natp a)
		  (natp b)
		  (natp n))
	     (equal (mod (+ (mod a n) (mod b n)) n)
		    (mod (+ a b) n))))

(in-theory (disable mod-mod-sum))

(defthm mod-diff
    (implies (and (natp a)
		  (natp b)
		  (natp n)
		  (>= a b))
	     (equal (mod (- a (mod b n)) n)
		    (mod (- a b) n))))

(defthm mod-equal
    (implies (and (natp m)
		  (natp n)
		  (< m n))
	     (equal (mod m n) m)))

(in-theory (disable mod-equal))

(defthm mod-1
  (implies (natp x)
           (equal (mod x 1) 0)))

(defthm mod-of-mod
    (implies (and (natp x)
		  (natp a)
		  (natp b)
		  (>= a b))
	     (equal (mod (mod x (expt 2 a)) (expt 2 b))
		    (mod x (expt 2 b)))))

(defthm mod-must-be-n
    (implies (and (natp m)
		  (natp n)
		  (not (= m 0))
		  (< m (* 2 n))
		  (= (mod m n) 0))
	     (= m n))
  :rule-classes ())

(defthm mod-prod
    (implies (and (natp m)
		  (natp n)
                  (integerp (* n k)) ;new
                  (integerp (* m k)) ;new
                  (rationalp k)      ;new
                  (> k 0)            ;new
		  (not (= n 0)))
	     (equal (mod (* k m) (* k n))
		    (* k (mod m n)))))

#| old version:
(defthm rem-prod
    (implies (and (natp m)
		  (natp n)
		  (natp k) ;old
		  (not (= n 0)))
	     (equal (rem (* k m) (* k n))
		    (* k (rem m n)))))
|#

(defthm mod-bnd-3
    (implies (and (natp m)
		  (natp n)
		  (natp a)
		  (natp r)
		  (<= (* a n) m)
		  (< m (+ (* a n) r)))
	     (< (mod m n) r))
  ;; Free variables make this rule very weak, but it seems harmless
  ;; enough to make it a :linear rule.
  :rule-classes :linear)

(defthm mod-force
    (implies (and (natp m)
		  (natp n)
		  (natp a)
		  (> (* (1+ a) n) m)
		  (>= m (* a n)))
	     (= (mod m n) (- m (* a n))))
  :rule-classes ())

(defthm mod-equal-int
    (implies (and (natp a)
		  (natp b)
		  (natp n)
		  (= (mod a n) (mod b n)))
	     (integerp (/ (- a b) n)))
  :rule-classes ())

(defthm mod-0-fl
    (implies (and (natp m)
		  (natp n))
	     (iff (= (mod m n) 0)
		  (= m (* (fl (/ m n)) n))))
  :rule-classes ())

(defthm quot-bnd
    (implies (and (natp m)
		  (natp n))
	     (>= m (* (fl (/ m n)) n)))
  :rule-classes :linear)

(defthm mod-0-0
    (implies (and (natp m)
		  (natp n)
		  (natp p)
                  (not (= p 0)))
	     (iff (= (mod m (* n p)) 0)
		  (and (= (mod m n) 0)
		       (= (mod (fl (/ m n)) p) 0))))
  :rule-classes ())


(defthm mod-mult-2
    (implies (and (natp n)
		  (natp a))
	     (equal (mod (* a n) n)
		    0)))

(defthm mod-force-equal
    (implies (and (natp a)
		  (natp b)
		  (natp n)
		  (< (abs (- a b)) n)
		  (= (mod a n) (mod b n)))
	     (= a b))
  :rule-classes ())

(defthm nk>=k-linear
    (implies (and (integerp n)
		  (integerp k)
		  (not (= n 0)))
	     (>= (abs (* n k)) k))
  :rule-classes :linear)

(defthm mod-mod-2
    (implies (natp x)
	     (or (= (mod x 2) 0)
		 (= (mod x 2) 1)))
  :rule-classes ())

(defthm mod-plus-mod-2
    (implies (and (natp x)
		  (natp y))
	     (iff (= (mod (+ x y) 2) (mod x 2))
		  (= (mod y 2) 0)))
  :rule-classes ())

(defthm mod-mod-2-not-equal
    (implies (natp x)
	     (not (= (mod x 2) (mod (1+ x) 2))))
  :rule-classes ())

(defthm x-or-x/2
    (implies (integerp x) 
	     (or (integerp (/ x 2)) (integerp (/ (1+ x) 2))))
  :rule-classes ())

(defthm mod-2*i-rewrite
    (implies (integerp i)
             ;; Rule A3 in fp.lisp suggests using (* 2 i) instead of
             ;; (+ i i).
             (equal (mod (* 2 i) 2) 0)))

(defthm mod-2*i+1
    (implies (integerp i)
	     (not (equal (mod (1+ (* 2 i)) 2) 0)))
  :rule-classes ())

(defthm mod-2*i+1-rewrite
    (implies (natp i)
	     (equal (mod (1+ (* 2 i)) 2) 1)))

(defun fl-half (x)
  (1- (fl (/ (1+ x) 2))))

(in-theory (disable fl-half))

(defthm fl-half-lemma
    (implies (and (integerp x)
		  (not (integerp (/ x 2))))
	     (= x (1+ (* 2 (fl-half x)))))
  :rule-classes ())


;;;**********************************************************************
;;;                   MISCELLANEOUS INEQUALITIES
;;;**********************************************************************

(defthm delta1-a
    (implies (and (rationalp x)
		  (rationalp y)
		  (rationalp d)
		  (>= y 0)
		  (>= x (+ y y))
		  (>= d 0))
	     (>= (- x (* y (+ 1 d)))
		 (* (- x y) (- 1 d))))
  :rule-classes ())

(defthm delta1-b
    (implies (and (rationalp x)
		  (rationalp y)
		  (rationalp d)
		  (>= y 0)
		  (>= x (+ y y))
		  (>= d 0))
	     (<= (- x (* y (- 1 d)))
		 (* (- x y) (+ 1 d))))
  :rule-classes ())

(defthm delta2
    (implies (and (rationalp x)
		  (rationalp y)
		  (rationalp d)
		  (>= (* x d) 0))
	     (>= (+ x (* y (- 1 d)))
		 (* (+ x y) (- 1 d))))
  :rule-classes ())

(defthm exp+1
    (implies (and (integerp m)
		  (integerp n)
		  (<= n m))
	     (> (* (- 1 (expt 2 m)) (- 1 (expt 2 n)))
		(- 1 (expt 2 (1+ m)))))
  :rule-classes ())

(defthm exp+2
    (implies (and (integerp n)
		  (integerp m)
		  (<= n m)
		  (<= m 0))
	     (< (* (1+ (expt 2 m)) (1+ (expt 2 n)))
		(1+ (expt 2 (+ m 2)))))
  :rule-classes ())

(defthm exp-invert
    (implies (and (integerp n)
		  (<= n -1))
	     (<= (/ (- 1 (expt 2 n)))
		 (1+ (expt 2 (1+ n)))))
  :rule-classes ())

(defthm sq-sq
    (implies (and (rationalp a)
		  (rationalp b)
		  (rationalp p)
		  (integerp n)
		  (<= (* (- 1 (expt 2 n)) p) (* a a))
		  (<= (* a a) p)
		  (<= (* b b) (* (expt 2 (- (* 2 n) 2)) p)))
	     (>= (* (- a b) (- a b))
		 (* (- 1 (expt 2 (1+ n))) p)))
  :rule-classes ())
