// ---------------------------------------------------------------------------
// - Sort.cpp                                                                -
// - aleph:txt library - array sorting implementation                        -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Sort.hpp"
#include "Boolean.hpp"
#include "Exception.hpp"

namespace aleph {

  // this function compare an object with another one
  static bool qsort_cmpobj (Runnable* robj, Object* ref, Object* dst) {
    if ((ref == nilp) && (dst == nilp)) return true;
    if ((ref == nilp) && (dst != nilp)) return false;
    Object* obj   = ref->oper (robj, Object::LTH, dst);
    Boolean* bobj = dynamic_cast <Boolean*> (obj);
    bool result   = (bobj == nilp) ? false : bobj->toboolean ();
    Object::cref (obj);
    return result;
  }

  // this function makes a partition of the array and returns the 
  // pivot index
  static long qsort_partition (Runnable* robj, Vector* argv, 
			       const long first, const long last) {
    // initialize the pivot index, last index and next (unknown) index
    long pidx = first;
    long lidx = first;
    long uidx = lidx + 1;
    // get the pivot object - and protect it
    Object* pvt = Object::iref (argv->get (pidx));
    // swap until we have reached the last element
    while (uidx <= last) {
      Object* ref = Object::iref (argv->get (uidx));
      if (qsort_cmpobj (robj, ref, pvt) == true) {
	lidx++;
	argv->set (uidx, argv->get (lidx));
	argv->set (lidx, ref);
      }
      Object::tref (ref);
      // the intel is in the other region
      uidx++;
    }
    // place the pivot in proper position
    argv->set (first, argv->get (lidx));
    argv->set (lidx,  pvt);
    Object::tref (pvt);
    // the pivot index is now the last index
    return lidx;
  }

  // this function performs a recursive quick sort
  static void qsort_vector (Runnable* robj, Vector* argv, 
			    const long first, const long last) {
    if (first >= last) return;
    long pidx = qsort_partition (robj, argv, first, last);
    qsort_vector (robj, argv, first, pidx - 1);
    qsort_vector (robj, argv, pidx + 1, last);
  }

  // quick sort library function

  Object* atxt_qsort (Runnable* robj, Nameset* nset, Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 1) 
      throw Exception ("argument-error", "invalid arguments with sort");
    // get the argument vector
    Vector* vobj = dynamic_cast <Vector*> (argv->get (0));
    if (vobj == nilp)
      throw Exception ("type-error", "invalid object with sort");
    vobj->wrlock ();
    try {
      long len = (vobj == nilp) ? 0 : vobj->length ();
      qsort_vector (robj, vobj, 0, len - 1);
      vobj->unlock ();
      delete argv;
      return nilp;
    } catch (...) {
      vobj->unlock ();
      delete argv;
      throw;
    }
  }
}
