// ---------------------------------------------------------------------------
// - Integer.cpp                                                             -
// - standard object library - integer class implementation                  -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Real.hpp"
#include "Input.hpp"
#include "Output.hpp"
#include "Vector.hpp"
#include "Recycle.hpp"
#include "Boolean.hpp"
#include "Character.hpp"
#include "Exception.hpp"
#include "cmth.hpp"
#include "ccnv.hpp"
#include "cstr.hpp"

namespace aleph {

  // the integer supported quarks
  static const long QUARK_OR       = String::intern ("or");
  static const long QUARK_OPP      = String::intern ("++");
  static const long QUARK_OMM      = String::intern ("--");
  static const long QUARK_ADD      = String::intern ("+");
  static const long QUARK_SUB      = String::intern ("-");
  static const long QUARK_MUL      = String::intern ("*");
  static const long QUARK_DIV      = String::intern ("/");
  static const long QUARK_EQL      = String::intern ("==");
  static const long QUARK_NEQ      = String::intern ("!=");
  static const long QUARK_LTH      = String::intern ("<");
  static const long QUARK_LEQ      = String::intern ("<=");
  static const long QUARK_GTH      = String::intern (">");
  static const long QUARK_GEQ      = String::intern (">=");
  static const long QUARK_AEQ      = String::intern ("+=");
  static const long QUARK_SEQ      = String::intern ("-=");
  static const long QUARK_MEQ      = String::intern ("*=");
  static const long QUARK_DEQ      = String::intern ("/=");
  static const long QUARK_ABS      = String::intern ("abs");
  static const long QUARK_AND      = String::intern ("and");
  static const long QUARK_SHL      = String::intern ("shl");
  static const long QUARK_SHR      = String::intern ("shr");
  static const long QUARK_XOR      = String::intern ("xor");
  static const long QUARK_MOD      = String::intern ("mod");
  static const long QUARK_NOT      = String::intern ("not");
  static const long QUARK_ODDP     = String::intern ("odd-p");
  static const long QUARK_EVENP    = String::intern ("even-p");
  static const long QUARK_ZEROP    = String::intern ("zero-p");

  // the integer recycler
  static Recycle recycler;

  // allocate a new integer
  void* Integer::operator new (const t_size size) {
    return recycler.pop (size);
  }

  // delete an integer
  void Integer::operator delete (void* handle) {
    recycler.push (handle);
  }

  // create a new default integer
  
  Integer::Integer (void) {
    d_value = 0;
  }

  // create a new integer from a long integer

  Integer::Integer (const t_long value) {
    d_value = value;
  }

  // create a new integer from a string

  Integer::Integer (const String& value) {
    bool  status = false;
    const char* data = value.tochar ();
    d_value = c_atoll (data,status);
    delete [] data;
    if (status == false) 
      throw Exception ("literal-error",
		       "illegal string integer number",value);
  }
  
  // copy constructor for this integer

  Integer::Integer (const Integer& that) {
    d_value = that.d_value;
  }

  // return the class name

  String Integer::repr (void) const {
    return "Integer";
  }

  // return a literal representation of this integer

  String Integer::toliteral (void) const {
    return tostring ();
  }

  // get a string representation on this integer

  String Integer::tostring (void) const {
    char*  buffer = c_lltoa (d_value);
    String result (buffer);
    delete [] buffer;
    return result;
  }

  // return a clone of this integer

  Object* Integer::clone (void) const {
    return new Integer (*this);
  }

  // return the integer serial code

  t_byte Integer::serialid (void) const {
    return SERIAL_INTG_ID;
  }

  // serialize this integer

  void Integer::wrstream (Output& os) const {
    rdlock ();
    t_byte data[8];
    c_ohton (d_value, data);
    os.write ((char*) data, 8);
    unlock ();
  }

  // deserialize this integer

  void Integer::rdstream (Input& is) {
    wrlock ();
    t_byte data[8];
    for (long i = 0; i < 8; i++) data[i] = (t_byte) is.read ();
    d_value = c_ontoh (data);
    unlock ();
  }
  
  // set an integer with a value

  Integer& Integer::operator = (const long value) {
    d_value = value;
    return *this;
  }

  // set an integer with a value

  Integer& Integer::operator = (const Integer& value) {
    d_value = value.d_value;
    return *this;
  }

  // compare an integer with a native value

  bool Integer::operator == (const long value) const {
    return (d_value == value);
  }

  bool Integer::operator != (const long value) const {
    return (d_value != value);
  }

  // compare two integers

  bool Integer::operator == (const Integer& value) const {
    return (d_value == value.d_value);
  }

  bool Integer::operator != (const Integer& value) const {
    return (d_value != value.d_value);
  }

  bool Integer::operator < (const Integer& value) const {
    return (d_value < value.d_value);
  }

  bool Integer::operator <= (const Integer& value) const {
    return (d_value <= value.d_value);
  }
  
  bool Integer::operator > (const Integer& value) const {
    return (d_value > value.d_value);
  }

  bool Integer::operator >= (const Integer& value) const {
    return (d_value >= value.d_value);
  }

  // compute the opposite of the current integer
  
  Integer operator - (const Integer& x) {
    return Integer (-x.d_value);
  }
  
  // add two integers together from two integers

  Integer operator + (const Integer& x, const Integer& y) {
    return Integer (x.d_value + y.d_value);
  }

  // substract two integers together from two integers
  
  Integer operator - (const Integer& x, const Integer& y) {
    return Integer (x.d_value - y.d_value);
  }
  
  // multiply two integers together from two integers
  
  Integer operator * (const Integer& x, const Integer& y) {
    return Integer (x.d_value * y.d_value);
  }
    
  // divide two integers together from two integers
  
  Integer operator / (const Integer& x, const Integer& y) {
    if (y.d_value == 0) throw Exception ("division-by-zero");
    return Integer (x.d_value / y.d_value);
  }

  // return the remainder of this number with its argument

  Integer operator % (const Integer& x, const Integer& y) {
    if (y.d_value == 0) throw Exception ("division-by-zero");
    t_long result = x.d_value % y.d_value;
    return Integer (result);
  }

  // add an integer to this one

  Integer& Integer::operator += (const Integer& x) {
    d_value += x.d_value;
    return *this;
  }

  // substract an integer to this one
  
  Integer& Integer::operator -= (const Integer& x) {
    d_value -= x.d_value;
    return *this;
  }

  // multiply an integer with this one
  
  Integer& Integer::operator *= (const Integer& x) {
    d_value *= x.d_value;
    return *this;
  }

  // divide an integer with this one
  
  Integer& Integer::operator /= (const Integer& x) {
    if (x.d_value == 0) throw Exception ("division-by-zero");
    d_value /= x.d_value;
    return *this;
  }

  // return the absolute value of this number

  Integer Integer::abs (void) const {
  t_long result = (d_value < 0) ? -d_value : d_value;
  return Integer (result);
  }

  // return a native integer value

  t_long Integer::tointeger (void) const {
    return d_value;
  }

  // extract new integer from a string

  t_long Integer::tointeger (const String& value) {
    bool  status = false;
    const char* data = value.tochar ();
    t_long result = c_atoll (data,status);
    delete [] data;
    if (status == false) 
      throw Exception ("literal-error",
		       "illegal string integer number",value);
    return result;
  }

  // return a random number

  t_long Integer::random (void) {
    return c_random ();
  }

  // evaluate an object to an integer value

  t_long Integer::evalto (Runnable* robj, Nameset* nset, Object* object) {
    Object* obj = (object == nilp) ? nilp : object->eval (robj, nset);
    Integer* val = dynamic_cast <Integer*> (obj);
    if (val == nilp) throw Exception ("type-error", "nil object to evaluate");
    return val->tointeger ();
  }

  // create a new integer in a generic way

  Object* Integer::mknew (Vector* argv) {
    if ((argv == nilp) || (argv->length () == 0)) return new Integer;
    if (argv->length () != 1) 
      throw Exception ("argument-error", 
		       "too many argument with integer constructor");
    // try to map the integer argument
    Object* obj = argv->get (0);
    if (obj == nilp) return new Integer;

    // try an integer object
    Integer* ival = dynamic_cast <Integer*> (obj);
    if (ival != nilp) return new Integer (*ival);

    // try a real object
    Real* rval = dynamic_cast <Real*> (obj);
    if (rval != nilp) return new Integer (rval->tointeger ());

    // try a character object
    Character* cval = dynamic_cast <Character*> (obj);
    if (cval != nilp) return new Integer (cval->tochar ());

    // try a string object
    String* sval = dynamic_cast <String*> (obj);
    if (sval != nilp) return new Integer (*sval);

    // illegal object
    throw Exception ("type-error", "illegal object with integer constructor",
		     obj->repr ());
  }

  // operate this integer with another object

  Object* Integer::oper (Runnable* robj, t_oper type, Object* object) {
    Integer* iobj = dynamic_cast <Integer*> (object);
    Real*    dobj = dynamic_cast <Real*>    (object);
    switch (type) {
    case Object::ADD:
      if (iobj != nilp) return new Integer (d_value + iobj->d_value);
      if (dobj != nilp) return new Real    (d_value + dobj->d_value);
      break;
    case Object::SUB:
      if (iobj != nilp) return new Integer (d_value - iobj->d_value);
      if (dobj != nilp) return new Real    (d_value - dobj->d_value);
      break;
    case Object::MUL:
      if (iobj != nilp) return new Integer (d_value * iobj->d_value);
      if (dobj != nilp) return new Real    (d_value * dobj->d_value);
      break;
    case Object::DIV:
      if (iobj != nilp) return new Integer (*this / *iobj);
      if (dobj != nilp) return new Real    (*this / *dobj);
      break;
    case Object::EQL:
      if (iobj != nilp) return new Boolean (d_value == iobj->d_value);
      if (dobj != nilp) return new Boolean (d_value == dobj->d_value);
      break;
    case Object::NEQ:
      if (iobj != nilp) return new Boolean (d_value != iobj->d_value);
      if (dobj != nilp) return new Boolean (d_value != dobj->d_value);
      break;
    case Object::GEQ:
      if (iobj != nilp) return new Boolean (d_value >= iobj->d_value);
      if (dobj != nilp) return new Boolean (d_value >= dobj->d_value);
      break;
    case Object::GTH:
      if (iobj != nilp) return new Boolean (d_value > iobj->d_value);
      if (dobj != nilp) return new Boolean (d_value > dobj->d_value);
      break;
    case Object::LEQ:
      if (iobj != nilp) return new Boolean (d_value <= iobj->d_value);
      if (dobj != nilp) return new Boolean (d_value <= dobj->d_value);
      break;
    case Object::LTH:
      if (iobj != nilp) return new Boolean (d_value < iobj->d_value);
      if (dobj != nilp) return new Boolean (d_value < dobj->d_value);
      break;
    case Object::MINUS:
      return new Integer (-d_value);
      break;
    }
    throw Exception ("type-error", "invalid operand with integer",
		     Object::repr (object));
  }

  // set an object to this integer

  Object* Integer::vdef (Runnable* robj, Nameset* nset, Object* object) {
    Integer* iobj = dynamic_cast <Integer*> (object);
    if (iobj != nilp) {
      d_value = iobj->d_value;
      return this;
    }
    Real* fobj = dynamic_cast <Real*> (object);
    if (fobj != nilp) {
      d_value = fobj->tointeger ();
      return this;
    }
    throw Exception ("type-error", "invalid object with integer vdef",
		     Object::repr (object));
  }

  // apply this integer with a set of arguments and a quark

  Object* Integer::apply (Runnable* robj, Nameset* nset, const long quark,
			  Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_OPP) {
	d_value++;
	return this;
      }
      if (quark == QUARK_OMM) {
	d_value--;
	return this;
      }
      if (quark == QUARK_ABS) {
	t_long val = (d_value >= 0) ? d_value : -d_value;
	return new Integer (val);
      }
      if (quark == QUARK_EVENP) {
	bool result = ((d_value & 1) == 0) ? true : false;
	return new Boolean (result);
      }
      if (quark == QUARK_ODDP) {
	bool result = ((d_value & 1) == 1) ? true : false;
	return new Boolean (result);
      }
      if (quark == QUARK_ZEROP) {
	bool result = (d_value == 0);
	return new Boolean (result);
      }
      if (quark == QUARK_NOT) {
	Object* result = new Integer (~d_value);
	return result;
      } 
    }

    // dispatch one argument
    if (argc == 1) {
      if (quark == QUARK_ADD) return oper (robj, Object::ADD, argv->get (0));
      if (quark == QUARK_SUB) return oper (robj, Object::SUB, argv->get (0));
      if (quark == QUARK_MUL) return oper (robj, Object::MUL, argv->get (0));
      if (quark == QUARK_DIV) return oper (robj, Object::DIV, argv->get (0));

      if (quark == QUARK_AEQ) {
	t_long val = argv->getint (0);
	d_value += val;
	return this;
      }
      if (quark == QUARK_SEQ) {
	t_long val = argv->getint (0);
	d_value -= val;
	return this;
      }
      if (quark == QUARK_MEQ) {
	t_long val = argv->getint (0);
	d_value *= val;
	return this;
      }
      if (quark == QUARK_DEQ) {
	t_long val = argv->getint (0);
	if (val == 0) throw Exception ("divide-error", "division by zero");
	d_value = d_value / val;
	return this;
      }
      if (quark == QUARK_EQL) {
	Object* result = oper (robj, Object::EQL, argv->get (0));
	return result;
      }
      if (quark == QUARK_NEQ) {
	Object* result = oper (robj, Object::NEQ, argv->get (0));
	return result;
      }
      if (quark == QUARK_LTH) {
	Object* result = oper (robj, Object::LTH, argv->get (0));
	return result;
      }
      if (quark == QUARK_LEQ) {
	Object* result = oper (robj, Object::LEQ, argv->get (0));
	return result;
      }
      if (quark == QUARK_GTH) {
	Object* result = oper (robj, Object::GTH, argv->get (0));
	return result;
      }
      if (quark == QUARK_GEQ) {
	Object* result = oper (robj, Object::GEQ, argv->get (0));
	return result;
      }
      if (quark == QUARK_MOD) {
	t_long val = argv->getint (0);
	Object* result = new Integer (d_value % val);
	return result;
      }
      if (quark == QUARK_SHL) {
	t_long val = argv->getint (0);
	Object* result = new Integer (d_value << val);
	return result;
      }
      if (quark == QUARK_SHR) {
	t_long val = argv->getint (0);
	Object* result = new Integer (d_value >> val);
	return result;
      }
      if (quark == QUARK_XOR) {
	t_long val = argv->getint (0);
	Object* result = new Integer (d_value ^ val);
	return result;
      }
      if (quark == QUARK_AND) {
	t_long val = argv->getint (0);
	Object* result = new Integer (d_value & val);
	return result;
      }
      if (quark == QUARK_OR) {
	t_long val = argv->getint (0);
	Object* result = new Integer (d_value | val);
	return result;
      }
    } 

    // call the literal method
    return Literal::apply (robj, nset, quark, argv);
  }
}
