# This module is a part of Alicq instant messenger package
# It provides core of basic graphical interface of Alicq: tree window
#
# Author: Ihar Viarheichyk

package require BWidget

#module {hideoffline hideempty}
variable alias unknown

event add <<Hide-Offline>> <Control-o>
event add <<Hide-Empty>> <Control-i>
event add <<ContextCurrent>> <3> <Control-1>
event add <<ContextSelected>> <F9>
event add <<Item-Activate>> <Double-1> <Return>
event add <<QuickSearch>> <Key-slash> <Control-Key-f>
event add <<SearchCancel>> <FocusOut> <Key-Escape>

option add *quick.Checkbutton.selectColor "" widgetDefault
option add *quick.Checkbutton.indicatorOn 0
option add *status.compound none widgetDefault
option add *quickPanel true widgetDefault
option add *rosterContactColor black widgetDefault
option add *localContactColor blue widgetDefault

namespace eval meta {
	set author "Ihar Viarheichyk <iverg@mail.ru>"
	set name "Contact list"
	set description "Basic group and contact list implementation"
	set icon img:group
	array set grouping {
		type variant default Plain property Global:UI:Tree|Grouping
		description "Grouping rule" save exit menu {Interface Grouping}
	}
	array set quick_search {
		type {boolean cache} default 0 menu {Search {In Contact List}}
		weight .099
	}
}

namespace eval hide {
	namespace eval meta {
		array set empty {
			type boolean default 0 property Global:UI:Tree|HideEmpty
			description "Hide empty groups" save exit
			menu {Interface "Hide empty groups"}
			
		}
		array set offline {
			type boolean default 0
			menu {Interface "Hide offline contacts"}
			property Global:UI:Tree|HideOffline
			description "Hide offline contacts" save exit
		}
	}
}

# Visibility filters
namespace eval filter {
	# Contacts with pending messages are always visible
	proc _pending {uid} {
		if {[info exists [ref $uid](Pending)]} {
			return -code break
		} else { return 1}
	}
}

namespace eval [ref Group:common]::meta {
	array set Open	{type boolean default 0 save exit}
}
namespace eval [ref Group:status] {}

proc MyDraw {path} { 
	_draw_tree_old $path
	#set im [.tr.c create image 0 0 -anchor nw -image [Bitmap::get /usr/share/pixmaps/Snail.xpm]]
	#.tr.c lower $im
}

proc DeltaY {} {
	set max 0
	foreach x [image names] {
		set h [image height $x]
		if {[string match img:* $x] && $h>$max} {set max $h}
	}
	incr max 2
}

# Main tree window
proc MainWindow {} {
	wm group . .
	wm command . [file join [pwd] $::argv0]
	# Handle geometry command-line option
	if {![info exists ::geometry] &&
	    [set gm [option get . geometry Geometry]]!=""} {wm geometry . $gm }
	# Handle state of main window on startup: normal or iconic	 
	if {[set state [option get . state State]]!=""} { wm state . $state }
	set searchVar [namespace current]::quick_search
	# Create scrolled tree widget
	frame .quick
	if {[option get . quickPanel Boolean]=="true"} {
		grid .quick -columnspan 2 -sticky we
	}
	grid columnconfigure .quick 8 -weight 1
	grid [checkbutton .quick.offline -image img:offline\
		-variable [namespace current]::hide::offline]\
		-row 0 -column 1 -padx 2 -ipadx 2 -ipady 2 -sticky w
	ui::tooltip .quick.offline $hide::meta::offline(description) <<Hide-Offline>>
	grid [checkbutton .quick.empty -image img:group\
		-variable [namespace current]::hide::empty]\
		-row 0 -column 2 -padx 2 -ipadx 2 -ipady 2 -sticky w
	ui::tooltip .quick.empty $hide::meta::empty(description) <<Hide-Empty>>
	grid [checkbutton .quick.search -text [mc Search] -variable $searchVar]\
		-row 0 -column 10 -padx 2 -ipadx 2 -ipady 2 -sticky e
	ui::tooltip .quick.search "Search in contact list" <<QuickSearch>>
	trace variable $searchVar w [nc QuickSearch .qsearch .tr.c]
	set sw [ScrolledWindow .sw -relief flat -borderwidth 0]
	set tree [Tree .tr -deltay [DeltaY]\
		-dragenabled yes -dropenabled yes -dropcmd [nc DropCmd]\
		-opencmd [nc OpenCmd 1] -closecmd [nc OpenCmd 0]]
	$sw setwidget $tree
	foreach id {Text Image} {
		$tree bind${id} <Double-1> [nc DoubleClick]
		$tree bind${id} <<ContextCurrent>> [nc PopupMenu %X %Y]
	}
	foreach x {roster local} { 
		variable _${x} [option get .tr ${x}ContactColor ContactColor] 
	}
	bind .tr.c <<ContextSelected>> [nc PopupSelected %X %Y]

	set menu1 [menubutton .main -text [mc "Menu"]\
		-direction above -menu .main.m]
	menu .main.m -type normal -tearoff no
	set menu2 [menubutton .status -text [mc offline]\
		-menu .status.m -image img:offline]
	set search [FastSearch .qsearch .tr $searchVar]
	grid $sw -columnspan 2 -row 3 -sticky news
	grid $menu2 -column 0 -row 5 -sticky nws -padx 2 
	grid $menu1 -column 1 -row 5 -sticky nes -padx 2
	grid columnconfigure . 1 -weight 1
	grid rowconfigure . 3 -weight 1

	bind .tr.c <<QuickSearch>> [list set $searchVar 1]
	bind .tr.c <<Popup>> { puts "Key %K" }

	# Small hack to allow mouse wheel scrolling in tree widget
	bind .tr.c <4> ".tr.c yview scroll -2 units"
	bind .tr.c <5> ".tr.c yview scroll 2 units"
	
	if {[wm protocol . WM_DELETE_WINDOW]==""} {
		wm protocol . WM_DELETE_WINDOW {Event exit}
	}
	bind . <Return> [namespace code  {DoubleClick [.tr selection get]}]
	bind [winfo class .] <FocusIn> { focus .tr.c }
}

proc QuickSearch {name main var args} {
	upvar 1 $var mode
	set current [winfo viewable $name]
	if {$current==$mode} return 
	if {$mode} {
		grid $name -column 0 -columnspan 2 -row 4 -sticky we
		$name selection range 0 end
		focus $name
	} else { 
		grid forget $name 
		focus $main
	}
}

proc FastSearch {top tree var} {
	entry $top -validate key -vcmd [nc DoSearch $top %P $tree]
	bind $top <<SearchCancel>> [list set $var 0]
	set top
}

proc DoSearch {entry val tree} {
	if {$val==""} { return 1}
	set found [list]
	set idx [string length $val]
	foreach uid [select Contact] {
		set alias [get $uid Alias]
		set uin [lindex [split $uid :] end]
		if {[string equal -nocase -length $idx $val $alias]} {
			lappend found $uid
		} elseif {[string equal -length $idx $val $uin]} {
			lappend found $uid
		} 	
	}
	if {[llength $found]} { return [ShowFound $tree $found] }
	return 0
}

proc ShowFound {tree items} {
	variable mapping
	foreach uid $items {
		if {![info exists mapping($uid)]} continue
		set r [lindex $mapping($uid) 0]
		if {[$tree visible $r]} break
	}
	if {![info exists r]} { return 0 }
	if {![$tree visible $r]} { RecOpen $tree $r }
	$tree see $r
	$tree selection set $r
	return 1
}

proc RecOpen {tree node} {
	if {$node=="root"} return
	$tree itemconfigure $node -open 1
	RecOpen $tree [NodeParent $node]
}

proc OpenCmd {val node} { set [ref [UidByNode $node]](Open) $val }

proc DoubleClick {node} {
	if {[set uid [UidByNode $node]]=={}} return
	# If  node is group: open/close
	if [string match Group:common:* $uid] {
		.tr itemconfigure $node -open\
			[set [ref $uid](Open) [expr {[get $uid Open 0]^1}]]
	} else { Event $uid|Send:text $uid|Send:text }
}

proc CreateMenus {} {
	menu .status.m -type normal -tearoff no
	foreach status {online away dnd occ ffc na invisible offline} {
		.status.m add command -image img:$status -label [mc $status]\
			-command "Event SetStatus $status"
		set [ref Group:status:$status](Alias) [mc $status]
	}
}

# --- Handle in-tree item renaming
handler {*|Rename} Rename {id} {
	variable CurrentNode
	set uid [lindex [split $id |] 0]
	.tr edit $CurrentNode [get $uid Alias] [nc ItemModify $uid]
}

proc ItemModify {uid text} {
	set [ref $uid](Alias) $text
	return 1
}

# --- Handle moving items between groups
proc DropCmd {dest src type op datatype data} {
	# Don't accept unknown types and don'w allow loops
	if {$type=="widget" && $dest==".tr" ||
	    [lindex $type 0]=="position" && [lindex $type 1]=="root"} {
		set uid [UidByNode $data]
		set parent [UidByNode [NodeParent $data]]
		if {[string match Group:common:* $parent]} {
			set parent [lindex [split $parent :] end]
			set [ref $uid](Groups) [grep x [get $uid Groups {}]\
				{$x!=$parent}]
		}
		return
	}
	if {![string match node* $type]||\
	     [string match ${data}* [lindex $type 1]]} return
	set dest [UidByNode [lindex $type 1]]
	switch -glob -- $dest {
		Group:common:* {
			set uid [UidByNode $data]
			if {$uid==$dest} return
			set dest [lindex [split $dest :] end]
			set old [lindex [split [UidByNode\
					[NodeParent $data]] :] end]
			# Set filter expression depending of drop type
			if {$op!={copy}} {set exp {$x!=$old}} else {set exp 1}
			append exp {&& $x!=$dest}
			set [ref $uid](Groups) [concat\
				[grep x [get $uid Groups {}] $exp] $dest]
		}
	}
}

# --- Handle deletion of an item
handler {*|Delete} Delete {id} {
	variable CurrentNode
	variable mapping
	set uid [UidByNode $CurrentNode]
	if {[set pos [lsearch -exact $mapping($uid) $CurrentNode]]>=0} {
		set lst [lreplace $mapping($uid) $pos $pos]
		if {![llength $lst]} { 
			set name [get $uid Alias]
			if {[tk_messageBox -type yesno -title "Removing $name"\
				-message "Remove $name from contact list?"]
					!="yes"} return
			unset [ref $uid](Groups)
			after idle [list unset [ref $uid]]
		} else {
			set parent [lindex [split\
				[UidByNode [NodeParent $CurrentNode]] :] end]
			set [ref $uid](Groups) [grep x [get $uid Groups {}] {
				$x!=$parent
			}]
		}
	}
}

# --- Context menu handling
proc PopupSelected {x y} {
	set sel [.tr selection get]
	if {[llength $sel]} { PopupMenu $x $y [lindex $sel 0] }
}

proc PopupMenu {x y node} {
	variable CurrentNode
	Event Menu .context [UidByNode $node]
	if {![winfo exists .context]||[.context index last]=="none"} return
	set CurrentNode $node
	tk_popup .context [expr $x-10] [expr $y-5]
}

handler ConfigLoaded ConfigLoaded {args} {
	variable grouping
	set meta::grouping(values) [map x [info procs Grouping::*] { namespace tail $x }]
	if {![info exists [ref Group:common:other]]} { Group other Other }
	
	update idletasks
	if {[string is true $hide::offline]} trigger_offline
	trace variable grouping w [nc Regroup]
	trace variable hide::empty w [nc Remap]
	trace variable hide::offline w [nc trigger_offline]

	trace variable [ref Me](Status) w [nc UpdateStatus]
	trace variable [ref Me](Alias) w [nc WindowTitle]
	UpdateStatus

	foreach x {contacts groups} {
		set sort::meta::${x}(values)\
			[map y [info procs sort::${x}::*] {namespace tail $y}]
		SortAlias $x [set sort::$x]
		trace variable sort::$x w [nc resort] 
	}
	foreach x [select Contact] { CommonMonitor $x; ContactMonitor $x}
	foreach x [select Group:common] { CommonMonitor $x }
	hook {New:Contact:* New:Group:common:*} [nc Mapping] 0.4
	hook {New:Contact:* New:Group:common:*} [nc CommonMonitor] 0.6
	hook New:Contact:* [nc ContactMonitor]
	Regroup
	Event Menu .main.m
	handler ModuleLoaded recalcMenu {module} { Event Menu .main.m } 0.99
	# If no menu created by external modul add at least exit entry
	if {[.main.m index last]=="none"} { 
		.main.m add command -label [mc Exit] -command {Event exit}
	}
}

proc CommonMonitor {uid} {
	redraw $uid
	trace variable [ref $uid](Alias) w [nc onAlias $uid]
	trace variable [ref $uid](rid) wu [nc onRid $uid]
	trace variable [ref $uid](Groups) w [nc Mapping $uid]
	trace variable [ref $uid] u [nc Hide $uid]
}

proc ContactMonitor {uid} {
	trace variable [ref $uid](Status) w [list after idle [nc onStatus $uid]]
	trace variable [ref $uid](Pending) w [nc DoPending $uid]
}

proc onRid {uid name1 name2 op} { 
	redraw $uid -fill
	if {$op=="u" && [info exists [ref $uid](Groups)]} {
		trace variable [ref $uid](rid) wu [nc onRid $uid]
	}
}

proc onAlias {uid args} {
	redraw $uid -text
	DoSort $uid
}

proc onStatus {uid args} {
	redraw $uid -image -helptext
	foreach x [get $uid Groups] {delayed [nc redraw Group:common:$x -text]}
	#if {[string is true $hide::offline] } { Mapping $uid }
	Mapping $uid 
	if {$sort::contacts=="Status"} { DoSort $uid }
}

proc Hide {uid ref field op} {
	if {$field==""} { MappingChanged $uid [list] }
}

proc FlashIcon {uid stage} {
	redraw $uid [list -image\
		img:[expr {$stage?[get $uid Status offline]:"message"}]]
	set stage
}

# --- Flashing icon when pending messages present
proc NoPending {uid args} {
	unhook Flashing [nc FlashIcon $uid]
	FlashIcon $uid 1
	Flash less
	Mapping $uid
	trace variable [ref $uid](Pending) w [nc DoPending $uid]
	# Check if there other invisble contacts with pending messages
	delayed [nc SeeNext]
}

proc DoPending {uid args} {
	variable mapping
	if {![info exists mapping($uid)]} { Mapping $uid }
	See $uid
	hook Flashing [nc FlashIcon $uid]
	Flash more
	trace vdelete [ref $uid](Pending) w [nc DoPending $uid]
	trace variable [ref $uid](Pending) u [nc NoPending $uid]
}

# Find contact havind pending message and make it viewable in the tree
proc SeeNext {} {
	set uid [lindex [select Contact {[info exists Pending]}] 0]
	if {$uid!=""} { See $uid }
}

# Resisively open parent groups and make any of object nodes visible
proc See {uid} {
	variable mapping
	if {[info exists mapping($uid)]} {
		foreach x $mapping($uid) { if {[.tr visible $x]} break }
		if {![.tr visible $x]} { RecOpen .tr $x}
		.tr see $x
	}
}

# Namespace containing different grouping policies
namespace eval Grouping {
	# No grouping (default)
	proc Plain {uid} {
		expr {[string match Contact:* $uid]?{root}:{}}
	}
	# Use grous
	proc Groups {uid} {
		set ref [ref $uid]
		if { ![info exists ${ref}(Groups)] } { return "" }
		set g [set ${ref}(Groups)] 
		expr {[llength $g]?[map x $g {set _ Group:common:$x}]:{root}}
	}
	# Group by status
	proc Status {uid} {
		switch -glob -- $uid {
			Contact:* { return Group:status:[get $uid Status offline]}
			Group:status:* {return root}
			* {return {}}
		}
	}
	# Divide into two groups: online and offline contacts
	proc Online {uid} {
		switch -glob -- $uid {
			Contact:* {
			   return Group:status:[expr {([get $uid Status offline]==\
			   		{offline})?{offline}:{online}}]
			}
			Group:status:online {return root}
			Group:status:offline {return root}
			* {return {}}
		}
	}
}

proc Regroup {args} {
	variable grouping
	interp alias {} grouper {} [namespace current]::Grouping::$grouping
	# Hide lines in plain mode and show if necessary in others
	if {$grouping=="Plain"} {
		.tr configure -showlines 0 -deltax 0
	} else {
		.tr configure -showlines [option get .tr showlines D]\
		 	      -deltax [option get .tr deltax D]
	}
	Remap
}

proc Remap {args} {
	set list [select Contact]
	if {![string is true $hide::empty]} {
		set list [concat $list [select Group]]
	} else { HideEmpty }
	foreach id $list { Mapping $id }
}

proc HideEmpty {} {
	variable mapping
	foreach {key val} [array get mapping Group:*] {
		set n [lindex $val 0]
		if {$n!="" && [.tr exists $n] && ![llength [.tr nodes $n]]} {
			MappingChanged $key {}
			after idle [nc HideEmpty]
		}
	}
}

proc configlist {uid args} {
	variable all_node_options
	if {![llength $args]} { set args $all_node_options }
	set lst [list]
	foreach x $args {
		if {[llength $x]==2} { set lst [concat $lst $x]
		} else { lappend lst $x [node$x $uid] }
	}
	set lst
}

proc UpdateNode {node config} {
	if {![llength $config]} return 
	set parent [NodeParent $node]
	if {![.tr exists $parent]} {MappingChanged [UidByNode $parent] $parent}
	if {![.tr exists $node]} { 
		set pos 0
		foreach n [.tr nodes $parent] {
			if {[comparator $n $node]>=0} break else { incr pos }
		}
		.tr insert $pos $parent $node 
	}
	eval [list .tr itemconfigure $node] $config
}

proc RemoveNode {node} { 
	if {![.tr exists $node]} return
	.tr selection remove $node
	.tr delete $node 
	if {[string is true $hide::empty]} {
		set parent [NodeParent $node]
		if {$parent!="root" && ![llength [.tr nodes $parent]]} {
			MappingChanged [UidByNode $parent] {}
		}
	}
}

proc UidByNode {node} { lindex [split $node "\0"] end }

proc NodeParent {node} {string range $node 0 [expr [string last "\0" $node]-1]}

proc redraw {uid args} {
	variable mapping
	if {[info exists mapping($uid)]} {
		set config [eval [list configlist $uid] $args]
		foreach node $mapping($uid) { UpdateNode $node $config }
	}
}

proc node-text {uid} { 
	if {[string match Group:common:* $uid]} {
		set text [get $uid Alias]
		foreach x {all online} { set $x 0 }
		set g [lindex [split $uid :] end]
		foreach x [select Contact "\[info exists Groups\] && \[lsearch \$Groups $g\]!=-1"] {
			incr all
			if {[get $x Status offline]!="offline"} { incr online }
		}
		append text " - $online/$all"
	} else { get $uid Alias }
}

proc node-image {uid} {
	set img group
	if {[string match Contact:* $uid]} { 
		set img [get $uid Status offline] 
	}
	if {[nil? $img]} { set img offline }
	return img:$img
}

proc node-fill {uid} {
	upvar #0 [ref $uid](rid) rid
	set [namespace current]::[expr {[info exists rid]?"_roster":"_local"}]
}

proc node-open {uid} { 
	if {[string match Group:* $uid]} { 
		get $uid Open 0
	} else { return -code continue }
}

if [package vsatisfies [package present BWidget] 1.6] {
	proc node-helptype {uid} { return balloon }

	proc node-helptext {uid} {
		upvar #0 [ref $uid] info
		foreach {class type id} [split $uid :] break
		set t "$type id: $id"
		if {[info exists info(Client)]} {
			foreach {ver client unicode} $info(Client) break
			append t "\n$client, ICQv${ver}\n"\
				[expr ${unicode}?"Unicode":"Encoding: [get $uid encoding [set [pref {icq encoding}]]]"]
		}
		if {[info exists info(client:name)]} {
			append t "\n$info(client:name)" 
		}
		if {[info exists info(Status)]} {
			append t "\n[mc Status]: [mc $info(Status)]"
		}
		if {[info exists info(description)]} {
			append t "($info(description))"
		}
		if {[info exists info(IP)]} { append t "\nIP: $info(IP)" }
		if {[info exists info(dc)] && [info exists info(LocalIP)]} {
			append t "\nDC IP: $info(LocalIP)"
		}
		set t
	}
} else {
	proc node-helptext {args} { return -code continue }
}

# Sort namespace includes all available sorting methods
namespace export UidByNode
namespace eval sort {
	namespace eval meta {
		variable contacts
		array set contacts {
			type variant default Status save exit
			description "Sorting rule for contacts"
			menu {Interface "Sort contacts by"}
			property Global:UI:Tree|Sort:Contacts
		}
		variable groups
		array set groups {
			type variant default Identifier save exit
			description "Sorting rule for groups"
			menu {Interface "Sort groups by"}
			property Global:UI:Tree|Sort:Groups
		}
	}
	
	namespace export *
	# No sorting at all
	proc Unsorted {s1 s2} {return 0}
	# Sort by identifier (e.g UIN for ICQ contacts, group ID for groups)
	proc Identifier {s1 s2} { string compare $s1 $s2 }
	# Sort by alphabet (case-sensitive)
	proc Alphabet {s1 s2} {
		string compare	[get $s1 Alias] [get $s2 Alias]
	}
	# Sort by alphabet (case-insensitive)
	proc {Alphabet,case-insensitive} {s1 s2} {
		string compare -nocase [get $s1 Alias] [get $s2 Alias]
	}
	# Contact-specific sorting
	namespace eval contacts {
		namespace import [namespace parent]::*
		# Sort by status and then by alphabet
		array set weight {ffc a online b occ c dnd d away e
				na f invisible g offline z}
		proc Status {s1 s2} {
			variable weight
			set st1 [get $s1 Status offline]
			set st2 [get $s2 Status offline]
			set res [string compare $weight($st1) $weight($st2)]
			if {$res} {set res} else { Alphabet $s1 $s2 }
		}
	}
	#Group-specific sorting
	namespace eval groups {
		namespace import [namespace parent]::*

		if 0 {
		proc Unsorted {s1 s2} {
			puts "compare $s1 and $s2"
			set w1 [get $s1 weight 0.5]
			set w2 [get $s2 weight 0.5]
			expr {($w1<$w2)?-1:(($w1==$s2)?0:1)}
		}
		}
	}
}

proc comparator {n1 n2} {
	foreach x {1 2} {
		set node$x [UidByNode [set n$x]]
		set type$x [lindex [split [set node$x] :] 0]
	}
	if {[set res [string compare $type1 $type2]]} { return $res }
	sort${type1} $node1 $node2
}

proc sort {x} {
	if {![.tr exists $x]} return
	.tr reorder $x [lsort -command comparator [.tr nodes $x]]
	# workaround for nonupdating tree in Plain mode
	if {$x=={root}} {.tr insert end root _fake_; .tr delete _fake_}
}

proc resort {name args} {
	variable mapping
	upvar 1 $name val
	SortAlias [namespace tail $name] $val
	foreach {uid nodes} [array get mapping Group:*] { 
		foreach node $nodes { delayed [nc sort $node] }
	}
	delayed [nc sort root]
}

proc SortAlias {name val} {
	set ns [string totitle [string range $name 0 end-1]]
	set nc [namespace current]
	interp alias {} ${nc}::sort$ns {} ${nc}::sort::${name}::$val
}

proc DoSort {uid} {
	variable mapping
	if {![info exists mapping($uid)]} return
	foreach x $mapping($uid) { delayed [nc sort [NodeParent $x]] }
}

set all_node_options [list]
foreach x [info commands node-*] { 
	lappend all_node_options [string range $x 4 end]
}

#Handle tree nodes mapping changes
proc Mapping {uid args} { 
	MappingChanged $uid [mapper $uid] 
	if {[string match Group:common:* $uid]} {
		set g [lindex [split $uid :] end]
		foreach x [select Contact "\[lsearch \$Groups $g\]!=-1"] {
			MappingChanged $x [mapper $x] 
		}
		redraw $uid -text
	} elseif {[string match Contact:* $uid]} {
		foreach x [get $uid Groups] { delayed [nc redraw Group:common:$x -text] }
	}
}

proc MappingChanged {uid new} {
	variable mapping
	set old [list]
	if {[info exists mapping($uid)]} { set old $mapping($uid) }
	if {$old==$new} return
	set mapping($uid) $new
	if {![llength $new]} { unset mapping($uid) }
	foreach {_ Update Remove} [lcompare $old $new] break
	foreach x $Remove { RemoveNode $x }
	if {[llength $Update]} {
		set config [configlist $uid]
		foreach x $Update { UpdateNode $x $config }
	}
}

proc mapper {uid} {
	foreach x [info commands filter::*] { if {![$x $uid]} { return {}} }
	set lst [list]
	foreach x [grouper $uid] {
		set m [expr {($x=="root")?"root":[mapper $x]}]
		foreach y $m { lappend lst "$y\0$uid" }
	}
	set lst
}

proc trigger_offline {args} {
	if {[string is true $hide::offline]} {
		proc filter::offline {uid} { 
			set ref [ref $uid]
			expr {![string match Contact:* $uid] ||
			       [info exists ${ref}(Status)] &&
			       [set ${ref}(Status)]!="offline"}
		}
	} else { rename filter::offline {} }
	foreach x [select Contact {
		![info exists Status] || $Status=="offline"}] { Mapping $x }
}

proc trigger {name} { set $name [expr [string is true [set $name]]^1] }

proc UpdateStatus {args} {
	upvar #0 [ref Me](Status) status
	.status configure -image img:$status -text [mc $status]
	ui::tooltip .status [mc $status]
	WindowTitle
}
proc WindowTitle {args} {
	wm title . "Alicq: [get Me Alias] [get Me Status]"
}

resource icons {group away dnd ffc invisible message na occ offline online url}

MainWindow
CreateMenus
bind . <<Hide-Offline>> [nc trigger hide::offline]
bind . <<Hide-Empty>> [nc trigger hide::empty]

