#file This module is a part of Alicq ICQ client
# It is used to highlight URLs in incoming messages
# and invoke proper handler if user clicks on URL.
#
# Author: Ihar Viarheichyk

option add *urlColor blue widgetDefault

# This program names are used to guess default browser in Unix
array set browsers {
	graphical {mozilla firefox konqueror opera netscape galeon}
	text {links lynx}
	terminals {xterm aterm rxvt eterm konsole}
}
namespace eval meta {
	set author "Ihar Viarheichyk <iverg@mail.ru>"
	set description "Highlight and invoke proper browsers for URLs"
	set name "URL handlers"
	set requires dialogs
	set capabilities {UI unloadable}
	set revision [string map {$ "" " " "" Revision: ""} {$Revision: 1.24 $}]
	set iconfile /usr/share/pixmaps/netscape.png
	set configuration-objects url
}
# Namespace to contain all url objects
namespace eval [ref url] {
	namespace eval meta {
		array set regexp {
			description "Regular expression" save change
			weight .1
		}
		array set handler { 
			type {file executable} description "External handler"
			save change weight .2
		}
		array set script { type script description "Internal handler" 
			save change weight .3
		}
	}
	# Default protocol regexps
	foreach x {http https ftp gopher} { set ${x}(regexp) "$x://\[^ \]+" }
	foreach x {mailto news} { set ${x}(regexp) "$x:\[^ \]+" }
	set file(regexp) "file:/\[^ \]+"
	set UNC(regexp) "\\\\\[^ \]+"
}

# Check resource database for known URL types
foreach x [select url {
		![info exists handler] && ![info exists script]}] {
	set type [string totitle [namespace tail [ref $x]]]
	set val [option get . handler$type UrlHandler]
	if {$val!=""} { set [ref $x](handler) $val }
}

# If no handler for http is set, trying determine it automatically
handler ConfigLoaded defaultBrowser {args} {
	variable browsers
	# In Windows default system handler for URL can be invoked via
	# start command
	if {$::tcl_platform(platform)=="windows" && 
	    ![info exists [ref url:default](handler)]} { 
	    	set [ref url:default](handler) start }

	set ref [ref url:http]
	if {[info exists ${ref}(script)]||[info exists ${ref}(handler)]} return

	# Take into account BROWSER environment variable
	if {[info exists ::env(BROWSER)]} {
		foreach x {http ftp file} { 
			set [ref url:${x}](handler) $::env(BROWSER)
		}
	} elseif {$::tcl_platform(platform)=="unix"} {
		# In Unix ther is no default browser, trying to guess
		# Check popular graphical browsers first
		if {[set x [External $browsers(graphical)]]!=""} {
			set [ref url:http](handler) $x
		} else {
			# If graphical browser not found try text ones
			# First detect terminal program
			if {[set term [External $browsers(terminals)]]!="" &&
			    [set x [External $browsers(text)]]!=""} {
				set [ref url:http](handler) [list $term -e $x]
			}
		}	
	}
}

handler ConfigLoaded clearBrowsers {args} {
	unset [namespace current]::browsers
} 0.99

# Invoke URL handler
proc UrlHandler {txt} {
	set tags [$txt tag names current]
	set url [eval $txt get [$txt tag prevrange url "current + 1c"]]
	# Determine all possible types for this URL
	set types [list]
	for {set i 0} {[set i [lsearch -start $i $tags url:*]]!=-1} {incr i} {
		lappend types [lindex [split [lindex $tags $i] :] end]
	}
	# For each URL type check if handler set and invoke it if possible
	foreach x [concat $types default] {
		# Try internal script first
		foreach {n scr} {script {$y $url} handler {exec $y $url &}} {
			set var [ref url:$x]($n)
			if {![info exists $var]} continue
			set y [set $var]
			if {$y==""} continue
			if {[catch { eval [subst -nocommands $scr] } v]}\
				{Event Log error $v} else return
		}
	}
}

# Detect URLs in message and tag them
handler DisplayMessage FindUrls {w from end} {
	foreach item [select url {[info exists regexp]}] {
		set exp [set [ref $item](regexp)]
		if {$exp==""} continue
		set type [lindex [split [lindex [split $item |] 0] :] end]
		set i $from
		while {[set i [$w search -count l -regexp $exp $i $end]]!={}} {
			$w tag add url $i "$i + $l char"
			$w tag add url:$type $i "$i + $l char"
			set i [$w index "$i + $l char"]
		}	
	}		
	$w tag raise url
}

# Initialize url-specific tag handling
handler ChatOpen Configure {_ w args} { 
	$w tag configure url -foreground [option get $w urlColor UrlColor]\
		-underline yes
	$w tag bind url <Enter> {%W configure -cursor hand2}
	$w tag bind url <Leave> {%W configure -cursor {}}
	$w tag bind url <1> [namespace code {UrlHandler %W}]
}

