// matchers.cc
//
//  Copyright 2000,2001 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  Grammer for the condition language (probably not very correctly done, but
// close enough :) ):
//
//  CONDITION := CONDITION-LIST
//  CONDITION-LIST := CONDITION-AND-GROUP '|' CONDITION-LIST
//                 := CONDITION-AND-GROUP
//  CONDITION-AND-GROUP := CONDITION-ATOM CONDITION-AND-GROUP
//                      := CONDITION-ATOM
//  CONDITION-ATOM := '(' CONDITION-LIST ')'
//                 := '!' CONDITION-ATOM
//                 := '~'field-id [string]
//                 := string

#include "matchers.h"
#include "apt.h"
#include "tasks.h"

#include "../aptitude.h"

#include <apt-pkg/error.h>
#include <apt-pkg/pkgcache.h>
#include <apt-pkg/pkgsystem.h>
#include <apt-pkg/version.h>

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <regex.h>
#include <sys/types.h>

using namespace std;

inline bool string_matches(regex_t *pattern, string s)
  // Altering this to support globs or regexps is all that's needed to add
  // support for those to the program (if that's done, the inline should be
  // removed :) )
{
  regmatch_t amount;

  return !regexec(pattern, s.c_str(), 1, &amount, 0);
}

typedef bool (*mfunc)(regex_t *pattern, pkgCache::PkgIterator pkg, pkgCache::VerIterator ver);

template<mfunc f>
class pkg_string_matcher:public pkg_matcher
{
  regex_t pattern;

  pkg_string_matcher()
  {
  }

  bool compile(string _pattern, bool flag_errors)
  {
    int err=regcomp(&pattern, _pattern.c_str(), REG_ICASE|REG_EXTENDED|REG_NOSUB);
    if(err!=0)
      {
	if(flag_errors)
	  {
	    size_t needed=regerror(err, &pattern, NULL, 0);

	    char *buf=new char[needed+1];

	    regerror(err, &pattern, buf, needed+1);

	    _error->Error("Regex compilation error: %s", buf);

	    delete buf;
	  }

	return false;
      }
    else
      return true;
  }
public:
  static pkg_string_matcher<f> *init(string _pattern,
				     bool flag_errors)
  {
    pkg_string_matcher<f> *rval=new pkg_string_matcher<f>;

    if(!rval->compile(_pattern, flag_errors))
      {
	delete rval;
	return NULL;
      }
    else
      return rval;
  }

  ~pkg_string_matcher()
  {
    regfree(&pattern);
  }

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return f(&pattern, pkg, ver);
  }
};

bool name_matches(regex_t *pattern, pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
{
  return string_matches(pattern, pkg.Name());
}
typedef pkg_string_matcher<name_matches> pkg_name_matcher;

bool description_matches(regex_t *pattern, pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
{
  return !ver.end() && string_matches(pattern, apt_package_records->Lookup(ver.FileList()).LongDesc());
}
typedef pkg_string_matcher<description_matches> pkg_description_matcher;

bool maintainer_matches(regex_t *pattern, pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
{
  return !ver.end() && string_matches(pattern, apt_package_records->Lookup(ver.FileList()).Maintainer());
}
typedef pkg_string_matcher<maintainer_matches> pkg_maintainer_matcher;

bool section_matches(regex_t *pattern, pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
{
  return !ver.end() && ver.Section() && string_matches(pattern, ver.Section());
}
typedef pkg_string_matcher<section_matches> pkg_section_matcher;

bool version_matches(regex_t *pattern, pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
{
  return !ver.end() && ver.VerStr() && string_matches(pattern, ver.VerStr());
}
typedef pkg_string_matcher<version_matches> pkg_version_matcher;

bool task_matches(regex_t *pattern, pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
{
  list<string> *l=get_tasks(pkg);

  if(!l)
    return false;

  for(list<string>::iterator i=l->begin();
      i!=l->end();
      ++i)
    if(string_matches(pattern, *i))
      return true;

  return false;
}
typedef pkg_string_matcher<task_matches> pkg_task_matcher;

//  Package-file info matchers.  Match a package if any of its
// available files (for all versions) match the given criteria.
//
//  Should I use templates?
bool origin_matches(regex_t *pattern,
		    pkgCache::PkgIterator pkg,
		    pkgCache::VerIterator ver)
{
  for(pkgCache::VerIterator v=pkg.VersionList(); !v.end(); ++v)
    for(pkgCache::VerFileIterator f=v.FileList(); !f.end(); ++f)
      {
	pkgCache::PkgFileIterator cur=f.File();

	if(!cur.end() && cur.Origin() && string_matches(pattern,
							cur.Origin()))
	  return true;
      }

  return false;
}
typedef pkg_string_matcher<origin_matches> pkg_origin_matcher;

bool archive_matches(regex_t *pattern,
		     pkgCache::PkgIterator pkg,
		     pkgCache::VerIterator ver)
{
  if(ver.end() || ver.FileList().end())
    return false;

  for(pkgCache::VerIterator v=pkg.VersionList(); !v.end(); ++v)
    for(pkgCache::VerFileIterator f=v.FileList(); !f.end(); ++f)
      {
	pkgCache::PkgFileIterator cur=f.File();

	if(!cur.end() && cur.Archive() && string_matches(pattern,
							 cur.Archive()))
	  return true;
      }

  return false;
}
typedef pkg_string_matcher<archive_matches> pkg_archive_matcher;

class pkg_auto_matcher:public pkg_matcher
{
public:
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return
      (!pkg.CurrentVer().end() || (*apt_cache_file)[pkg].Install()) &&
      ((*apt_cache_file)->get_ext_state(pkg).install_reason!=aptitudeDepCache::manual);
  }
};

class pkg_broken_matcher:public pkg_matcher
{
public:
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    if(ver.end())
      return false;
    else
      {
	aptitudeDepCache::StateCache &state=(*apt_cache_file)[pkg];
	return state.NowBroken() || state.InstBroken();
      }
  }
};

class pkg_priority_matcher:public pkg_matcher
{
  pkgCache::State::VerPriority type;
public:
  pkg_priority_matcher(pkgCache::State::VerPriority _type)
    :type(_type) {}

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    if(ver.end())
      return false;
    else
      return ver->Priority == type;
  }
};

// Matches packages with unmet dependencies of a particular type.
class pkg_broken_type_matcher:public pkg_matcher
{
  pkgCache::Dep::DepType type; // Which type to match
public:
  pkg_broken_type_matcher(pkgCache::Dep::DepType _type)
    :type(_type) {}

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    if(ver.end())
      return false;
    else
      {
	pkgCache::DepIterator dep=ver.DependsList();

	while(!dep.end())
	  {
	    // Skip to the end of the Or group to check GInstall
	    while(dep->CompareOp & pkgCache::Dep::Or)
	      ++dep;

	    if(dep->Type==type &&
	       !((*apt_cache_file)[dep]&pkgDepCache::DepGInstall))
	      // Oops, it's broken..
	      return true;

	    ++dep;
	  }
      }
    return false;
  }
};

// This matches packages based on the action that will be taken with them.
//
// It will treat a request for a non-auto type as also being a request
// for the auto type.
class pkg_action_matcher:public pkg_matcher
{
  pkg_action_state type;
  bool require_purge;
public:
  pkg_action_matcher(pkg_action_state _type, bool _require_purge)
    :type(_type), require_purge(_require_purge)
  {
  }

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    if(require_purge &&
       ((*apt_cache_file)[pkg].iFlags & pkgDepCache::Purge) == 0)
      return false;
    else
      {
	pkg_action_state thetype=find_pkg_state(pkg);

	switch(type)
	  {
	  case pkg_install:
	    return thetype==pkg_install || thetype==pkg_auto_install;
	  case pkg_hold:
	    return !pkg.CurrentVer().end() && ((*apt_cache_file)[pkg].Held() || (*apt_cache_file)->get_ext_state(pkg).selection_state==pkgCache::State::Hold);
	  case pkg_remove:
	    return thetype==pkg_remove || thetype==pkg_auto_remove ||
	      thetype==pkg_unused_remove;
	  default:
	    return thetype==type;
	  }
      }
  }
};

class pkg_virtual_matcher:public pkg_matcher
// Matches *pure* virtual packages -- ie, those that /have/ no versions.
{
public:
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return pkg.VersionList().end();
  }
};

class pkg_installed_matcher:public pkg_matcher
// Matches packages which are installed..slightly quirky in that it matches
// *any* version of an installed package.
{
public:
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return !pkg.CurrentVer().end();
  }
};

class pkg_essential_matcher:public pkg_matcher
// Matches essential packages
{
public:
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return
      (pkg->Flags&pkgCache::Flag::Essential)==pkgCache::Flag::Essential ||
      (pkg->Flags&pkgCache::Flag::Important)==pkgCache::Flag::Important;
  }
};

class pkg_configfiles_matcher:public pkg_matcher
// Matches a package which was removed but has config files remaining
{
public:
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return pkg->CurrentState==pkgCache::State::ConfigFiles;
  }
};

// Matches packages with a dependency on the given pattern.
// FIXME: make this more general (for recommends, eg?)
class pkg_dep_matcher:public pkg_matcher
{
private:
  pkg_matcher *pattern;
  pkgCache::Dep::DepType type;

public:
  pkg_dep_matcher(pkgCache::Dep::DepType _type,
		  pkg_matcher *_pattern)
    :pattern(_pattern), type(_type)
  {
  }
  ~pkg_dep_matcher() {delete pattern;}

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    assert(!pkg.end());
    if(ver.end())
      return false;

    for(pkgCache::DepIterator dep=ver.DependsList(); !dep.end(); ++dep)
      {
	if( (type == dep->Type) ||
	    (type==pkgCache::Dep::Depends &&
	     dep->Type==pkgCache::Dep::PreDepends))
	  {
	    // See if a versionless match works,.
	    if(dep.TargetPkg().VersionList().end() &&
	       pattern->matches(dep.TargetPkg(), dep.TargetPkg().VersionList()))
	      return true;

	    for(pkgCache::VerIterator i=dep.TargetPkg().VersionList(); !i.end(); i++)
	      if(_system->VS->CheckDep(i.VerStr(), dep->CompareOp, dep.TargetVer()))
		{
		  if(pattern->matches(dep.TargetPkg(), i))
		    return true;
		}
	  }
      }

    return false;
  }
};

class pkg_or_matcher:public pkg_matcher
{
  pkg_matcher *left,*right;
public:
  pkg_or_matcher(pkg_matcher *_left, pkg_matcher *_right):left(_left),right(_right) {}
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return left->matches(pkg, ver) || right->matches(pkg, ver);
  }
  ~pkg_or_matcher() {delete left; delete right;}
};

class pkg_and_matcher:public pkg_matcher
{
  pkg_matcher *left,*right;
public:
  pkg_and_matcher(pkg_matcher *_left, pkg_matcher *_right):left(_left),right(_right) {}
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return left->matches(pkg, ver) && right->matches(pkg, ver);
  }
  ~pkg_and_matcher() {delete left; delete right;}
};

class pkg_not_matcher:public pkg_matcher
{
  pkg_matcher *child;
public:
  pkg_not_matcher(pkg_matcher *_child):child(_child) {}
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return !child->matches(pkg, ver);
  }
  ~pkg_not_matcher() {delete child;}
};

// Matches packages that were garbage-collected.
class pkg_garbage_matcher:public pkg_matcher
{
public:
  pkg_garbage_matcher() {}

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    if(ver.end())
      return false;
    else
      return (*apt_cache_file)->get_ext_state(pkg).garbage;
  }
};

// A dummy matcher that matches any package.
class pkg_true_matcher:public pkg_matcher
{
public:
  pkg_true_matcher() {}

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return true;
  }
};

// A dummy matcher that matches no packages.
class pkg_false_matcher:public pkg_matcher
{
public:
  pkg_false_matcher() {}

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return false;
  }
};

// Matches packages which have a dependency of the given type declared
// on them by a package matching a given pattern.
//
// Traces through Provided packages as well.
class pkg_revdep_matcher:public pkg_matcher
{
  pkgCache::Dep::DepType type;
  pkg_matcher *pattern;
public:
  pkg_revdep_matcher(pkgCache::Dep::DepType _type,
		     pkg_matcher *_pattern):type(_type), pattern(_pattern)
  {
  }

  ~pkg_revdep_matcher()
  {
    delete pattern;
  }

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    // Check direct dependencies.
    for(pkgCache::DepIterator d=pkg.RevDependsList(); !d.end(); ++d)
      {
	if((d->Type==type ||
	    (type==pkgCache::Dep::Depends && d->Type==pkgCache::Dep::PreDepends)) &&
	   (!d.TargetVer() || (!ver.end() &&
			       _system->VS->CheckDep(ver.VerStr(), d->CompareOp, d.TargetVer()))) &&
	   pattern->matches(d.ParentPkg(), d.ParentVer()))
	  return true;
      }

    // Check dependencies through virtual packages.  ie, things that Depend
    // on stuff this package [version] Provides.
    if(!ver.end())
      for(pkgCache::PrvIterator p=ver.ProvidesList(); !p.end(); ++p)
	{
	  for(pkgCache::DepIterator d=p.ParentPkg().RevDependsList();
	      !d.end(); ++d)
	    {
	      // Only unversioned dependencies can match here.
	      if(d->Type==type && !d.TargetVer() &&
		 pattern->matches(d.ParentPkg(), d.ParentVer()))
		return true;
	    }
	}

    return false;
  }
};


/** Matches packages that provide a package that matches the given
 *  pattern.
 */
class pkg_provides_matcher:public pkg_matcher
{
  pkg_matcher *pattern;
public:
  pkg_provides_matcher(pkg_matcher *_pattern):pattern(_pattern) {}

  ~pkg_provides_matcher() 
  {
    delete pattern;
  }

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    if(ver.end())
      return false;

    for(pkgCache::PrvIterator p=ver.ProvidesList(); !p.end(); ++p)
      {
	// Assumes no provided version.
	if(pattern->matches(p.ParentPkg(), pkgCache::VerIterator(*apt_cache_file)))
	  return true;
      }

    return false;
  }
};

/** Matches packages which are provided by a package that fits the
 *  given pattern.
 */
class pkg_revprv_matcher:public pkg_matcher
{
  pkg_matcher *pattern;
public:
  pkg_revprv_matcher(pkg_matcher *_pattern):pattern(_pattern) {}

  ~pkg_revprv_matcher() 
  {
    delete pattern;
  }

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    for(pkgCache::PrvIterator p=pkg.ProvidesList(); !p.end(); ++p)
      {
	if(pattern->matches(p.OwnerPkg(), p.OwnerVer()))
	  return true;
      }

    return false;
  }
};

//  Now back from the dead..it seems some people were actually using it ;-)
//
// Matches (non-virtual) packages which no installed package declares
// an "important" dependency of the given type on.
//
// Note that the notion of "importantness" is affected by the current
// settings!
class pkg_norevdep_matcher:public pkg_matcher
{
public:
  pkg_norevdep_matcher() {}

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    if(ver.end())
      return false;
    else
      {
        pkgCache::DepIterator dep=pkg.RevDependsList();

        while(!dep.end())
          {
            if((*apt_cache_file)->GetPolicy().IsImportantDep(dep) &&
               !dep.ParentVer().ParentPkg().CurrentVer().end())
              return false;

            ++dep;
          }

        return true;
      }
  }
};

// Matches (non-virtual) packages which no installed package declares
// a dependency of the given type on.
class pkg_norevdep_type_matcher:public pkg_matcher
{
  pkgCache::Dep::DepType type; // Which type to match
public:
  pkg_norevdep_type_matcher(pkgCache::Dep::DepType _type)
    :type(_type) {}

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    if(ver.end())
      return false;
    else
      {
	pkgCache::DepIterator dep=pkg.RevDependsList();

	while(!dep.end())
	  {
	    // Return false if the depender is installed.
	    if(dep->Type==type &&
	       !dep.ParentVer().ParentPkg().CurrentVer().end())
	      return false;

	    ++dep;
	  }
      }
    return true;
  }
};

class pkg_new_matcher:public pkg_matcher
{
public:
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    // Don't match virtual packages.
    if(pkg.VersionList().end())
      return false;
    else
      return (*apt_cache_file)->get_ext_state(pkg).new_package;
  }
};

class pkg_upgradable_matcher:public pkg_matcher
{
public:
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return !pkg.CurrentVer().end() && (*apt_cache_file)[pkg].Upgradable();
  }
};

// Parses a dependency type.  Returns (ick) -1 if the type is not
// recognized.
pkgCache::Dep::DepType parse_deptype(string s)
{
  if(!strcasecmp(s.c_str(), "depends"))
    return pkgCache::Dep::Depends;
  if(!strcasecmp(s.c_str(), "predepends"))
    return pkgCache::Dep::PreDepends;
  if(!strcasecmp(s.c_str(), "recommends"))
    return pkgCache::Dep::Recommends;
  else if(!strcasecmp(s.c_str(), "suggests"))
    return pkgCache::Dep::Suggests;
  else if(!strcasecmp(s.c_str(), "conflicts"))
    return pkgCache::Dep::Conflicts;
  else if(!strcasecmp(s.c_str(), "replaces"))
    return pkgCache::Dep::Replaces;
  else // ewww.
    return (pkgCache::Dep::DepType) -1;
}

pkg_matcher *parse_condition_list(string s, unsigned int &loc,
				  bool flag_errors);

// Returns a substring up to the first metacharacter, including escaped
// metacharacters (parentheses, ~, |, and !)
//
// Advances loc to the first character of 's' following the escaped string.
std::string parse_substr(string s, unsigned int &loc)
{
  std::string rval="";
  bool done=false;

  do
    {
      while(loc<s.size() &&
	    s[loc]!='(' &&
	    s[loc]!=')' &&
	    s[loc]!='!' &&
	    s[loc]!='~' &&
	    s[loc]!='|')
	{
	  rval+=s[loc];
	  ++loc;
	}

      // We quit because we ran off the end of the string or saw a
      // metacharacter.  If the latter case and it was a tilde-escape,
      // add the escaped character to the string and continue.
      if(loc<s.size()-1 && s[loc]=='~' &&
	 (s[loc+1]=='(' || s[loc+1]==')' ||
	  s[loc+1]=='!' || s[loc+1]=='~' ||
	  s[loc+1]=='|'))
	{
	  rval+=s[loc+1];
	  loc+=2;
	}
      else
	done=true;
    } while(!done);

  return rval;
}

pkg_matcher *parse_atom(string s, unsigned int &loc,
			bool flag_errors)
{
  std::string substr;

  while(loc<s.size() && isspace(s[loc]))
    loc++;

  while(loc<s.size() && s[loc]!='|' && s[loc]!=')')
    {
      if(s[loc]=='!')
	{
	  loc++;
	  pkg_matcher *atom=parse_atom(s, loc, flag_errors);
	  if(!atom)
	    return NULL;
	  return new pkg_not_matcher(atom);
	}
      else if(s[loc]=='(')
	// Recurse into the list
	{
	  loc++;
	  pkg_matcher *lst=parse_condition_list(s, loc,
						flag_errors);
	  if(!lst)
	    return NULL;

	  if(!(loc<s.size() && s[loc]==')'))
	    {
	      if(flag_errors)
		_error->Error(_("Unmatched '('"));
	      delete lst;
	      return NULL;
	    }
	  else
	    {
	      loc++;
	      return lst;
	    }
	}
      else if(s[loc]=='~')
	{
	  if(loc+1==s.size())
	    {
	      loc++;
	      return pkg_name_matcher::init("~", flag_errors);
	    }
	  else
	    {
	      unsigned int prevloc=loc;
	      loc+=2;
	      switch(s[prevloc+1])
		// Nested switch statements, mmmm...
		// Ok, there really is a reason here.  For all of the match
		// types that need a string argument, some prefix code (see
		// below) is needed to find the string's end.  But this would
		// be worse than unnecessary for the others.  So I have this
		// double check -- first test for anything that doesn't need
		// the prefix code, then work out which of the other forms
		// we have.
		{
		case 'v':
		  return new pkg_virtual_matcher;
		case 'b':
		  return new pkg_broken_matcher;
		case 'g':
		  return new pkg_garbage_matcher;
		case 'c':
		  return new pkg_configfiles_matcher;
		case 'i':
		  return new pkg_installed_matcher;
		case 'E':
		  return new pkg_essential_matcher;
		case 'M':
		  return new pkg_auto_matcher;
		case 'N':
		  return new pkg_new_matcher;
		case 'U':
		  return new pkg_upgradable_matcher;
		case 'P':
		case 'C':
		  {
		    pkg_matcher *m=parse_atom(s, loc, flag_errors);

		    if(!m)
		      return NULL;
		    
		    switch(s[prevloc+1])
		      {
		      case 'C':
			return new pkg_dep_matcher(pkgCache::Dep::Conflicts, m);
		      case 'P':
			return new pkg_provides_matcher(m);
		      }
		  }
		case 'D':
		case 'R':
		  {
		    bool do_provides=false;
		    pkgCache::Dep::DepType type=pkgCache::Dep::Depends;
		    string::size_type nextloc=loc;

		    while(nextloc<s.size() && isalpha(s[nextloc]))
		      ++nextloc;

		    if(nextloc<s.size() && s[nextloc]==':')
		      {
			string tname(s, loc, nextloc-loc);

			loc=nextloc+1;

			if(!strcasecmp(tname.c_str(), "provides"))
			  do_provides=true;
			else
			  {
			    type=parse_deptype(tname.c_str());

			    if(type==-1)
			      {
				char buf[512];
				snprintf(buf, 512, _("Unknown dependency type: %s"),
					 tname.c_str());
				if(flag_errors)
				  _error->Error("%s", buf);
				return NULL;
			      }
			  }
		      }

		    pkg_matcher *m=parse_atom(s, loc, flag_errors);

		    if(!m)
		      return NULL;

		    switch(s[prevloc+1])
		      {
		      case 'D':
			if(do_provides)
			  return new pkg_provides_matcher(m);
			else
			  return new pkg_dep_matcher(type, m);
		      case 'R':
			if(do_provides)
			  return new pkg_revprv_matcher(m);
			else
			  return new pkg_revdep_matcher(type, m);
		      }
		  }
		default:
		  substr=parse_substr(s, loc);
		  switch(s[prevloc+1])
		    {
		    case 'a':
		      {
			// Match packages to be installed
			if(!strcasecmp(substr.c_str(), "install"))
			  return new pkg_action_matcher(pkg_install, false);

			// Match packages to be upgraded
			else if(!strcasecmp(substr.c_str(), "upgrade"))
			  return new pkg_action_matcher(pkg_upgrade, false);

			// Match packages to be removed OR purged
			else if(!strcasecmp(substr.c_str(), "remove"))
			  return new pkg_action_matcher(pkg_remove, false);

			// Match packages to be purged
			else if(!strcasecmp(substr.c_str(), "purge"))
			  return new pkg_action_matcher(pkg_remove, true);

			// Match packages to be reinstalled
			else if(!strcasecmp(substr.c_str(), "reinstall"))
			  return new pkg_action_matcher(pkg_reinstall, false);

			// Match held packages
			else if(!strcasecmp(substr.c_str(), "hold"))
			  return new pkg_action_matcher(pkg_hold, false);

			else
			  {
			    char buf[512];
			    snprintf(buf, 512, _("Unknown action type: %s"),
				     substr.c_str());
			    if(flag_errors)
			      _error->Error("%s", buf);
			    return NULL;
			  }
		      }
		    case 'A':
		      return pkg_archive_matcher::init(substr, flag_errors);
		    case 'B':
		      {
			pkgCache::Dep::DepType ptype=parse_deptype(substr);

			if(ptype!=-1)
			  return new pkg_broken_type_matcher(ptype);
			else
			  {
			    char buf[512];
			    snprintf(buf, 512, _("Unknown dependency type: %s"),
				     substr.c_str());
			    if(flag_errors)
			      _error->Error("%s", buf);
			    return NULL;
			  }
		      }
		    case 'd':
		      return pkg_description_matcher::init(substr, flag_errors);
		    case 'F':
		      return new pkg_false_matcher;
		    case 'm':
		      return pkg_maintainer_matcher::init(substr, flag_errors);
		    case 'n':
		      return pkg_name_matcher::init(substr, flag_errors);
		    case 'O':
		      return pkg_origin_matcher::init(substr, flag_errors);
		    case 'p':
		      {
			pkgCache::State::VerPriority type;

			const char *s=substr.c_str();

			if(strcasecmp(s, "important") == 0 ||
			   (apt_cache_file &&
			    strcasecmp(s, (*apt_cache_file)->GetCache().Priority(pkgCache::State::Important)) == 0))
			  type=pkgCache::State::Important;
			else if(strcasecmp(s, "required") == 0 ||
				(apt_cache_file &&
				 strcasecmp(s, (*apt_cache_file)->GetCache().Priority(pkgCache::State::Required)) == 0))
			  type=pkgCache::State::Required;
			else if(strcasecmp(s, "standard") == 0 ||
				(apt_cache_file &&
				 strcasecmp(s, (*apt_cache_file)->GetCache().Priority(pkgCache::State::Standard)) == 0))
			  type=pkgCache::State::Standard;
			else if(strcasecmp(s, "optional") == 0 ||
				(apt_cache_file &&
				 strcasecmp(s, (*apt_cache_file)->GetCache().Priority(pkgCache::State::Optional)) == 0))
			  type=pkgCache::State::Optional;
			else if(strcasecmp(s, "extra") == 0 ||
				(apt_cache_file &&
				 strcasecmp(s, (*apt_cache_file)->GetCache().Priority(pkgCache::State::Extra)) == 0))
			  type=pkgCache::State::Extra;
			else
			  {
			    if(flag_errors)
			      {
				char buf[512];

				snprintf(buf, 512, _("Unknown priority %s"),
					 substr.c_str());

				_error->Error("%s", buf);
			      }

			    return NULL;
			  }

			return new pkg_priority_matcher(type);
		      }
		    case 's':
		      return pkg_section_matcher::init(substr, flag_errors);
		    case 't':
		      return pkg_task_matcher::init(substr, flag_errors);
		    case 'T':
		      return new pkg_true_matcher;
		    case 'V':
		      return pkg_version_matcher::init(substr, flag_errors);
		    default:
		      {
			char buf[512];
			snprintf(buf, 512, _("Unknown pattern type: %c"), s[1]);
			if(flag_errors)
			  _error->Error("%s", buf);
			return NULL;
		      }
		    }
		}
	    }
	}
      else
	{
	  return pkg_name_matcher::init(parse_substr(s, loc),
					flag_errors);
	}
    }

  // If we get here, the string was empty.
  return NULL;
}

pkg_matcher *parse_and_group(string s, unsigned int &loc,
			     bool flag_errors)
{
  pkg_matcher *rval=NULL;
  while(loc<s.size() && isspace(s[loc]))
    loc++;

  while(loc<s.size() && s[loc]!='|' && s[loc]!=')')
    {
      pkg_matcher *atom=parse_atom(s, loc, flag_errors);
      if(!atom)
	{
	  delete rval;
	  return NULL;
	}

      if(rval==NULL)
	rval=atom;
      else
	rval=new pkg_and_matcher(rval, atom);

      while(loc<s.size() && isspace(s[loc]))
	loc++;
    }
  if(rval==NULL && flag_errors)
    _error->Error(_("Unexpected empty expression"));
  return rval;
}

pkg_matcher *parse_condition_list(string s, unsigned int &loc,
				  bool flag_errors)
  // This (and the grammer) should be optimized to avoid deep recursion
  // what does the above line mean? -- DNB 11/21/01
{
  pkg_matcher *grp=parse_and_group(s, loc, flag_errors);
  if(!grp)
    return NULL;

  while(loc<s.size() && isspace(s[loc]))
    loc++;

  while(loc<s.size() && s[loc]!=')')
    {
      if(loc<s.size() && s[loc]=='|')
	{
	  loc++;
	  pkg_matcher *grp2=parse_condition_list(s, loc,
						 flag_errors);
	  if(!grp2)
	    {
	      delete grp;
	      return NULL;
	    }
	  return new pkg_or_matcher(grp, grp2);
	}
      else
	// We should never come here in a well-formed expression
	{
	  delete grp;
	  if(flag_errors)
	    _error->Error(_("Badly formed expression"));
	  return NULL;
	}
      // Or here.
      while(loc<s.size() && isspace(s[loc]))
	loc++;
    }
  // Nothing else?  Well..
  return grp;
}

pkg_matcher *parse_pattern(string s, bool flag_errors)
{
  unsigned int loc=0;

  // Just filter blank strings out immediately.
  while(loc!=s.size() && isspace(s[loc]))
    loc++;

  if(loc==s.size())
    return NULL;

  pkg_matcher *rval=parse_condition_list(s, loc, flag_errors);

  if(rval==NULL)
    return NULL;
  else if(loc!=s.size())
    {
      if(flag_errors)
	_error->Error(_("Unexpected ')'"));
      delete rval;
      return NULL;
    }
  else
    return rval;
}
