#ifndef _PRSCOREPAINTER_CPP_
#define _PRSCOREPAINTER_CPP_

#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "prScorePainter.h"
#include "lyrics.h"
#include "note.h"
#include "scoreChord.h"
#include "scoreGroup.h"
#include "scoreBar.h"

PrScorePainter::PrScorePainter() {
  c_geometry = new ChordGeometry;
  g_geometry = new GroupGeometry;
}

PrScorePainter::~PrScorePainter() {
  delete c_geometry;
  delete g_geometry;
}

int PrScorePainter::indent(ScoreBar * bar) {
  if (bar->multiple()) return bar->commonIndent();
  else return indentSystem(bar);
}

const char * PrScorePainter::programName(int prg) const {
  return gmNames[prg];
}


void PrScorePainter::resetSigns(int key) {
  for (int i=0;i<7;i++) scrSigns[i]=allSigns[key+7][i];
}


void PrScorePainter::initChordGeometry(int nwidth, int yscale) {
  // chord
  c_geometry->ypos_old      = 0;
  c_geometry->ypos         = 0;
  c_geometry->middle_stem  = false;
  c_geometry->neighbour_offset = false;
  c_geometry->yAverage     = 0;
  c_geometry->num_of_notes = 0;
  c_geometry->yBottom      = 0;
  c_geometry->yTop         = 299;
  c_geometry->xPos         = 0;
  c_geometry->stemPos      = 0;
  c_geometry->stemDir      = STEM_AUTO;
  c_geometry->lyrics       = 0;
  c_geometry->nwidth       = nwidth;
  c_geometry->yscale       = yscale;
}

void PrScorePainter::initGroupGeometry(ScoreGroup * group, int xoff, int nwidth) {
  // group
  g_geometry->elements     = 0;
  g_geometry->dir          = 0;
  g_geometry->tuplet       = 0;
  g_geometry->xTupletLeft  = xoff + int(0.5*nwidth);
  g_geometry->xTupletRight = xoff;
  g_geometry->yTupletLeft  = 0;
  g_geometry->yTupletRight = 0;
  g_geometry->yBottomS     = 0;
  g_geometry->yTopS        = 0;
  if (group->tuplet() != 0) g_geometry->tuplet = group->tuplet();
}


void PrScorePainter::use(Note * note, ScoreChord * chord) {
  c_geometry->enh      = note->enh();
  c_geometry->len      = chord->duration();
  c_geometry->display_len = chord->display();
  c_geometry->step     = note->pitch() % 12;
  c_geometry->ypos_old = c_geometry->ypos;
  c_geometry->ypos     = invPitch[note->pitch()];
  c_geometry->sgn      = sign[c_geometry->step];
  c_geometry->lyrics   = note->lyrics();


  /*
  if (note->tupletBase() != 0) {
    c_geometry->display_len = int(c_geometry->len * note->tuplet()*2.0/(note->tuplet()+1));
  } else {
    c_geometry->display_len = c_geometry->len;
    }*/

  if (c_geometry->enh != 0) {
    c_geometry->ypos += enhF[c_geometry->enh+2][c_geometry->step];
    c_geometry->sgn   = enhS[c_geometry->enh+2][c_geometry->step];
  }
  int sgMem = scrSigns[c_geometry->ypos%7]; // Sign
  scrSigns[c_geometry->ypos%7] = c_geometry->sgn;
  if (sgMem == c_geometry->sgn) {
    c_geometry->sgn = 0;
  } else {
    if (c_geometry->sgn==0) {
      c_geometry->sgn = 3;
    }
  }
  if (clef != 0) c_geometry->ypos += yClef[clef];
  if (c_geometry->ypos < 1) c_geometry->ypos = 1;

  c_geometry->ypos *= c_geometry->yscale;

  // if notes are only one line apart, move:
  if (abs(c_geometry->ypos_old - c_geometry->ypos) < 2*c_geometry->yscale) {
    if ( c_geometry->neighbour_offset == true )
      c_geometry->neighbour_offset = false;
    else
	c_geometry->neighbour_offset = true;
    c_geometry->middle_stem = true;
  } else {
    c_geometry->neighbour_offset = false;
  }

  // build the y-average, set bottom/top:
  if (c_geometry->yTop > c_geometry->ypos) c_geometry->yTop = c_geometry->ypos;
  if (c_geometry->yBottom < c_geometry->ypos) c_geometry->yBottom = c_geometry->ypos;
  c_geometry->yAverage += c_geometry->ypos;
  c_geometry->num_of_notes++;
}

void PrScorePainter::use(ScoreObjectType t, ChordGeometry * cg, int xpos) {
  //
  // group
  //
  switch (t) {
  case BREAK:
    if (g_geometry->tuplet != g_geometry->elements)
      if (g_geometry->tuplet != 0) g_geometry->xTupletRight = xpos + int(0.5*c_geometry->nwidth);
    break;

  case SINGLE:
    g_geometry->yBottomS   = cg->yBottom;
    g_geometry->yTopS      = cg->yTop;
    if (g_geometry->tuplet != 0) g_geometry->xTupletRight = xpos + int(0.5*c_geometry->nwidth);
    break;

  case MULTIPLE:
    if (!cg->middle_stem) {
      g_geometry->xPosLeft[g_geometry->elements]  = cg->xPos;
      g_geometry->xPosRight[g_geometry->elements] = cg->xPos + cg->nwidth;
    } else {
      g_geometry->xPosLeft[g_geometry->elements]  = cg->stemPos;
      g_geometry->xPosRight[g_geometry->elements] = cg->stemPos;
    }
    g_geometry->yBottom[g_geometry->elements]   = cg->yBottom;
    g_geometry->yTop[g_geometry->elements]      = cg->yTop;
    g_geometry->flags[g_geometry->elements]     = flags(cg->display_len);

    if (g_geometry->tuplet != 0) g_geometry->xTupletRight = c_geometry->xPos;

    g_geometry->elements++;
    if (g_geometry->elements > MAX_ELEMENTS_PER_GROUP) { cout << "PANIC: PRSCOREPAINTER: max number of elements per group is reached!" << endl; g_geometry->elements--; }
    break;

  default:
    cout << "PrScorePainter::use() PANIC: wrong type given!" << endl;
  }
}


void PrScorePainter::makeGroupGeometry(bool hbeams) {
  int top = 299;
  int topInd = 0;
  int bottom = 0;
  int bottomInd = 0;
  double slope = 0;
  int start = 0;

  if (g_geometry->elements < 2) {
    bottom = g_geometry->yBottomS;
    top = g_geometry->yTopS;
  }

  for (int i=0; i<g_geometry->elements; i++) {
    if (top > g_geometry->yTop[i]) {
      top    = g_geometry->yTop[i];
      topInd = i;
    }
    if (bottom < g_geometry->yBottom[i]) {
      bottom    = g_geometry->yBottom[i];
      bottomInd = i;
    }
  }
  //
  // when the lowest notes is more distant to the center than the highest note, draw the stems up:
  //
  if (bottom - TOP_DOWN_TRANSITION*c_geometry->yscale > TOP_DOWN_TRANSITION*c_geometry->yscale - top) {
    //
    // STEM UP
    //
    g_geometry->dir = STEM_UP;
    start = top - stemLength();

    if (!hbeams) {
      slope = (g_geometry->yTop[g_geometry->elements-1] - g_geometry->yTop[0]) * 1.0 / (g_geometry->xPosRight[g_geometry->elements-1] - g_geometry->xPosRight[0]);
      start -= int(slope*(g_geometry->xPosRight[topInd] - g_geometry->xPosRight[0]));
    }
    setUpY(start, slope);

    if (g_geometry->tuplet != 0) {
      if (g_geometry->elements > 0) {
	g_geometry->yTupletLeft = g_geometry->yBottom[0] + 18;
	g_geometry->yTupletRight = g_geometry->yBottom[g_geometry->elements-1] + 18;
      } else {
	g_geometry->yTupletLeft = g_geometry->yBottomS + 18;
	g_geometry->yTupletRight = g_geometry->yBottomS + 18;
      }
    }

  } else {
    //
    // STEM DOWN
    //
    g_geometry->dir = STEM_DOWN;
    start = bottom + stemLength();

    if (!hbeams) {
      slope = (g_geometry->yBottom[g_geometry->elements-1] - g_geometry->yBottom[0]) * 1.0 / (g_geometry->xPosLeft[g_geometry->elements-1] - g_geometry->xPosLeft[0]);
      start -= int(slope*(g_geometry->xPosLeft[bottomInd] - g_geometry->xPosLeft[0]));
    }
    setDownY(start, slope);

    if (g_geometry->tuplet != 0) {
      if (g_geometry->elements > 0) {
	g_geometry->yTupletLeft = g_geometry->yTop[0] - 12;
	g_geometry->yTupletRight = g_geometry->yTop[g_geometry->elements-1] - 12;
      } else {
	g_geometry->yTupletLeft = g_geometry->yTopS - 12;
	g_geometry->yTupletRight = g_geometry->yTopS - 12;
      }
    }

  }

}

void PrScorePainter::setUpY(int start, double slope) {
  for (int i=0; i<g_geometry->elements; i++)
    g_geometry->yPos[i] = i==0 ? start : start + int(slope*(g_geometry->xPosRight[i]-g_geometry->xPosRight[0]));
}

void PrScorePainter::setDownY(int start, double slope) {
  for (int i=0; i<g_geometry->elements; i++)
    g_geometry->yPos[i] = i==0 ? start : start + int(slope*(g_geometry->xPosLeft[i]-g_geometry->xPosLeft[0]));
}



void PrScorePainter::makeChordGeometry(int xoff) {
  //
  // chord
  //
  c_geometry->xPos = xoff;

  if (c_geometry->middle_stem) {
    c_geometry->stemPos = xoff + c_geometry->nwidth;
  } else {
    c_geometry->yAverage /= c_geometry->num_of_notes;
    if (c_geometry->yAverage > c_geometry->yscale*TOP_DOWN_TRANSITION) c_geometry->stemPos = xoff + c_geometry->nwidth;
    else c_geometry->stemPos = xoff;
  }
  if (c_geometry->yAverage > c_geometry->yscale*TOP_DOWN_TRANSITION)
    c_geometry->stemDir = STEM_UP;
  else
    c_geometry->stemDir = STEM_DOWN;
}

int PrScorePainter::flags(int len) const {
  if (len<0) len = c_geometry->display_len;
  return 10-int(1.000001*log(len/3)/log(2))-3;
}

int PrScorePainter::lengthOrd(int len) const {
  if (len<0) {
    len = c_geometry->display_len;
  } else if (g_geometry->tuplet != 0) {
    // this is mainly because breaks do not know if they're tuplets, but the group does:
    len = int(len*g_geometry->tuplet*2.0/(g_geometry->tuplet+1));
  }
  return 10-int(1.000001*log(len/3)/log(2))-1; // 0=whole, 1=half, 2=quater, 3=eigth, etc
}

const char * PrScorePainter::string(int i) {
  sprintf(txt, "%d", i);
  return txt;
}

const char * PrScorePainter::lyrics() const {
  if (c_geometry->lyrics != 0)
    return c_geometry->lyrics->get();
  else
    return 0;
}

int PrScorePainter::dot(int l) const {
  if (l == 0) l = c_geometry->len;
  int x0 = int(1.000000001 * log(l/3)/log(2)); l = l-3*int(pow(2,x0));
  if (l<3) l = 3;
  int x1 = int(1.000000001 * log(l/3)/log(2)); l = l-3*int(pow(2,x1));
  if (l<3) l = 3;
  int x2 = int(1.000000001 * log(l/3)/log(2)); l = l-3*int(pow(2,x2));
  return (x0-x1==1)+(x1-x2==1);
}

int PrScorePainter::stemLength() const {
  return c_geometry->yscale * STEM_LENGTH;
}

bool PrScorePainter::hasStem() const {
  return ( lengthOrd() > 0 );
}

bool PrScorePainter::hasFlags() const {
  return ( lengthOrd() > 2 );
}


#endif
