
/*
 *  Diverse Bristol midi routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * This code should open the midi device (working with ALSA raw midi only for
 * the moment (9/11/01)), and read data from it. Not sure how it will be read,
 * either buffers, events, or perhaps just raw data. At some point in the 
 * development this will become a separate thread in the synth code.
 */
#include <sys/poll.h>
#include <stdlib.h>
#include <stdio.h>

#include "bristolmidi.h"

//#define DEBUG

extern bristolMidiMain bmidi;

/*
 * This is the ALSA code. Should separate it out already.
 */
bristolMidiALSAOpen(char *devname, int flags, int chan, int messages,
int (*callback)(), void *param, int dev, int handle)
{
	int nfds;

#ifdef DEBUG
	printf("bristolMidiALSAOpen(%s)\n", devname);
#endif
#ifdef ADLIB
#if ADLIB >= 0
	/*
	 * Configure blocking IO. We could select over the input for clarity, but
	 * since we are now a separate thread we are not going to worry about this.
	 */
#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR == 9)
	bmidi.dev[dev].flags = SND_SEQ_OPEN_INPUT;
	if (snd_rawmidi_open(&bmidi.dev[dev].driver.alsa.handle, NULL,
		devname, bmidi.dev[dev].flags) != 0)
#else
	bmidi.dev[dev].flags = SND_RAWMIDI_OPEN_INPUT;
	if (snd_rawmidi_open(&bmidi.dev[dev].driver.alsa.handle,
		atoi(devname), 0, bmidi.dev[dev].flags) != 0)
#endif
	{
		/* 
		 * Print a debug message, this is problematic.
		 */
		printf("Could not open the MIDI interface.\n");
		return(BRISTOL_MIDI_DRIVER);
	}

#if (SND_LIB_MINOR == 0 && SND_LIB_MAJOR < 6)
#ifdef DEBUG
	if (snd_rawmidi_info(bmidi.dev[dev].driver.alsa.handle,
		&bmidi.dev[dev].driver.alsa.info) != 0)
		printf("Could not get MIDI info.\n");
	else {
		printf("	Handle: %i\n", dev);
		printf("	type	%i\n", bmidi.dev[dev].driver.alsa.info.type);
		printf("	flags	%i\n", bmidi.dev[dev].driver.alsa.info.flags);
		printf("	id	%s\n", bmidi.dev[dev].driver.alsa.info.id);
		printf("	name	%s\n", bmidi.dev[dev].driver.alsa.info.name);
	}
#endif
#endif

#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR == 9)
	/*
	 * We need to get a file descriptor for this interface, since we are going
	 * to use it with select() over several interfaces.
	if ((bmidi.dev[dev].fd = open("/dev/midi", O_RDONLY)) < 0)
		printf("Issue getting MIDI fd\n");
	 */
	if ((nfds =
		snd_rawmidi_poll_descriptors_count(bmidi.dev[dev].driver.alsa.handle))
		< 1)
		printf("issue getting descriptors: %i\n", nfds);
	else {
		struct pollfd *pfds;

		printf("Found %i descriptor\n", nfds);

		pfds = (struct pollfd *) malloc(sizeof (struct pollfd) * nfds);
		snd_rawmidi_poll_descriptors(bmidi.dev[dev].driver.alsa.handle,
			pfds, nfds);

		bmidi.dev[dev].fd = pfds[0].fd;
	}

#else
	if ((bmidi.dev[dev].fd = snd_rawmidi_file_descriptor(
		bmidi.dev[dev].driver.alsa.handle)) < 0)
		printf("Issue getting MIDI fd\n");
#endif

	bmidi.dev[dev].flags |= BRISTOL_CONN_MIDI;

	return(handle);
#else /* ADLIB */
	return(-1);
#endif /* ADLIB */
#endif /* ADLIB */
}

bristolMidiALSAClose(int handle)
{
#ifdef DEBUG
	printf("bristolMidiALSAClose()\n");
#endif

#ifdef ADLIB
#if ADLIB >= 0
	/*
	 * Check to see if the associated device has multiple handles associated
	 */
	if (bmidi.dev[bmidi.handle[handle].dev].handleCount > 1)
	{
		bmidi.dev[bmidi.handle[handle].dev].handleCount--;
		bristolFreeHandle(handle);

		return(BRISTOL_MIDI_OK);
	}

	snd_rawmidi_close(bmidi.dev[bmidi.handle[handle].dev].driver.alsa.handle);

	bristolFreeDevice(bmidi.handle[handle].dev);
	bristolFreeHandle(handle);
#endif /* ADLIB */
#endif /* ADLIB */
	return(BRISTOL_MIDI_OK);
}

/*
 * Note that we should not read on a handle - we should read on a device, and
 * then forward to all the handles.
 */
bristolMidiALSARead(int dev, bristolMidiMsg *msg)
{
	int parsed, space;

#ifdef DEBUG
	printf("bristolMidiALSARead(%i)\n", dev);
#endif

	/*
	 * See if we can read more data from the midi device. We need to make sure
	 * we have buffer capacity, and if so attempt to read as many bytes as we
	 * have space for. We need to treat the buffer as endless, ie, rotary. It
	 * does make it pretty ugly if we come to large sysex messages.
	 */
	if ((space = BRISTOL_MIDI_BUFSIZE - bmidi.dev[dev].bufcount) > 0)
	{
		int offset, count;

		/*
		 * We have space in our buffer, but we may have a wrap condition. If 
		 * this is the case, only partially read to the end of the buffer, then
		 * read further into the buffer. Requires two read ops. Bummer.
		 */

		/*
		 * Generate offset for new data, with buffer size corrections.
		 */
		if ((offset = bmidi.dev[dev].bufindex + bmidi.dev[dev].bufcount)
			>= BRISTOL_MIDI_BUFSIZE)
			offset -= BRISTOL_MIDI_BUFSIZE;
		/*
		 * Read new data, if we have any
		 */
		if (bmidi.dev[dev].flags & BRISTOL_CONTROL_SOCKET) {
			count = read(bmidi.dev[dev].fd, &bmidi.dev[dev].buffer[offset], 1);
			if (count == 0)
			{
				int j;

				printf("	freedev %i\n", dev);
				/*
				 * Free any handles associated with this device.
				 */
				for (j = 0; j < BRISTOL_MIDI_HANDLES; j++)
					if (bmidi.handle[j].dev == dev)
						bristolFreeHandle(j);

				bristolFreeDevice(dev);
				return(BRISTOL_MIDI_OK);
			}
		} else {
#ifdef ADLIB
#if ADLIB >= 0
			if (bmidi.dev[dev].flags & BRISTOL_CONN_MIDI)
				count = snd_rawmidi_read(bmidi.dev[dev].driver.alsa.handle,
					&bmidi.dev[dev].buffer[offset], 1);
			else
#endif
#endif
				count = read(bmidi.dev[dev].fd,
					&bmidi.dev[dev].buffer[offset], 1);
		}
		if (count < 1)
		{
			if (bmidi.dev[dev].bufcount == 0)
			{
				printf("return - no data in buffer\n");
				msg->command = -1;
				return(BRISTOL_MIDI_DEV);
			} else
				count = 0;
		}
#ifdef DEBUG
		else
			printf("BYTE %x on %i\n", bmidi.dev[dev].buffer[offset],
				bmidi.dev[dev].fd);
#endif

		bmidi.dev[dev].bufcount++;
	}

	/*
	 * I really want to parse these, and then forward the messages to callback
	 * routines registered by the bristolMidiOpen() command?
	 */
	if ((parsed = bristolMidiRawToMsg(&bmidi.dev[dev].buffer[0],
			bmidi.dev[dev].bufcount, bmidi.dev[dev].bufindex, dev, msg)) > 0)
	{
		/* 
		 * Reduce count of available space.
		 */
		if ((bmidi.dev[dev].bufcount -= parsed) < 0)
		{
			/*
			 * We have an issue here, count has gone negative. Reset all buffer
			 * pointers, and just write an error for now.
			 */
			bmidi.dev[dev].bufcount = 0;
			bmidi.dev[dev].bufindex = 0;
			printf("Issue with buffer capacity going negative\n");
		}
		/*
		 * Wrap index as necessary.
		 */
		if ((bmidi.dev[dev].bufindex += parsed) >= BRISTOL_MIDI_BUFSIZE)
			bmidi.dev[dev].bufindex -= BRISTOL_MIDI_BUFSIZE;

		/*
		 * We should now go through all the handles, and find out who has
		 * registered for these events - then forward them the event via the
		 * callback. For now we are just going to send it to the one existing
		 * callback.
		 */
		msg->params.bristol.from = dev;
		if (msg->command != 0x0ff)
			checkcallbacks(msg);
	} else
		msg->command = -1;

	return(BRISTOL_MIDI_OK);
}

checkcallbacks(bristolMidiMsg *msg)
{
	int i, message = 1 << ((msg->command & 0x70) >> 4);

	for (i = 0; i < BRISTOL_MIDI_HANDLES; i++)
	{
		if (bmidi.handle[i].callback == NULL)
			continue;

		if (bmidi.dev[bmidi.handle[i].dev].flags & BRISTOL_ACCEPT_SOCKET)
			continue;

		if (bmidi.handle[i].messagemask & message)
		{
			if (msg->command == (MIDI_SYSEX & MIDI_COMMAND_MASK))
			{
				/*
				 * SYSEX should only be sent on a single channel, the one on 
				 * which it arrived.
				 */
				if (msg->params.bristol.from == bmidi.handle[i].dev)
				{
					bmidi.handle[i].callback(msg, bmidi.handle[i].param);
				}
			} else {
				unsigned char hold = msg->params.bristol.from;

				msg->params.bristol.from = i;
				bmidi.handle[i].callback(msg, bmidi.handle[i].param);
				msg->params.bristol.from = hold;
			}
		}
	}
}

