/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_layout_fs.pike,v 1.67.2.3 2004/02/18 13:08:59 vida Exp $
 */

#include <module.h>
#include <camas/screens.h>
#include <camas/globals.h>

inherit "module";
inherit "caudiumlib";

//
//! module: CAMAS: Layout Filesystem
//!  Filesystem Layout for CAMAS.<br /><br />
//!  This module will fetch the layout files, one file per CAMAS screen.<br />
//!  e.g.: if the filename is '<b>layout.rxml</b>', the path '<b>/path/to/layout</b>',
//!  <br /> and the screen '<b>compose</b>', <br />CAMAS will look for
//!  '<b>/path/to/layout/compose/layout.rxml</b>'.<br /> If the layout is not found,
//!  the default will be used ('<b>/path/to/layout/layout.rxml</b>').<br />
//!  If there is no default file, the layout from <i>camas_layout_default</i> module
//!  will be used.<br />The name of the layout is used for the dialog boxes and
//!  the preferences file.
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_PROVIDER
//! cvs_version: $Id: camas_layout_fs.pike,v 1.67.2.3 2004/02/18 13:08:59 vida Exp $
//

constant cvs_version = "$Id: camas_layout_fs.pike,v 1.67.2.3 2004/02/18 13:08:59 vida Exp $";
constant module_type = MODULE_PROVIDER;
constant module_name = "CAMAS: Layout Filesystem";
constant _module_doc = "This module will fetch the layout files, one file per CAMAS screen.<br /><br />"
                       "Here are a few things to note regarding layout modules:<ul>"
                       "<li>If the filename is '<b>layout.rxml</b>', the path '<b>/path/to/layout</b>',<br />"
                       "and the screen '<b>compose</b>', CAMAS will look for '<b>/path/to/layout/compose/layout.rxml</b>'.<br />"
                       "If the layout is not found, the default will be used ('<b>/path/to/layout/layout.rxml</b>').<br />"
                       "You can also choose to use framesets for your layout. As for the layout files, you can put "
                       "a default frameset, e.g.: <b>/path/to/layout/frameset.rxml</b> or specify a particular frameset for"
                       " a given screen, e.g.: <b>/path/to/layout/compose/frameset.rxml</b>.</li>"
                       "<li>Most of the global configuration is done using the XML "
		                   "configuration file. You can get a sample one from the 'Simplest' layout.</li>"
                       "<li>You must reload this module or restart Caudium if you added it.</li></ul><br/>";

constant module_doc    = "Layout Filesystem for CAMAS.<br /><br />" + _module_doc + 
                         "The name of the layout is used for the dialog boxes and the preferences file.<br />"
                         "If there is no default file, the layout from the <i>camas_layout_default</i> module will be used.<br /><br />";
constant module_unique = 0; // A copy of this module per layout
constant thread_safe = 1;		// should be

#if constant(thread_create)
object global_lock = Thread.Mutex();
#endif

mapping (string: mapping) screens = ([ ]);  // mapping for storing screens (name:data)
mapping (string: string)  config  = ([ ]);  // mapping for storing config  (label:data)
string layout_name;                         // My name

string  gbuttonsyntax   = "";               // Storage for gbutton syntax
mapping basicbuttonlist = ([ ]);            // Storage for camas-basic button list
array(object) filewatches = ({ });          // The filewatches objects
string location;                            // like QUERY(location) but updated at the right time;)

int files_loaded = 0;

// Early CAMAS 1.2 layouts contained:
//  - only one file for frameless layouts
//  - a file for the frameset and a file for the frames for framed layouts
// Now, layouts have a file per frame for consistency and speed up parsing
// of framed layouts
// 
// Compatibility with early layouts is available through the following define
#define EARLY_LAYOUT_COMPAT 1

// The size of the random name of the layout to generate
#define RANDOM_NAME_SIZE 10

/* Caudium modules methods */

//! method: void create ()
//!  Constructor
void create ()
{
  
#ifdef CAMAS_DEBUG
  defvar ("debug", 0, "Debug", TYPE_FLAG, "Sends debug information into Caudium error log. Note that debug at the start of this module will be available "
   "even if you set this option to 'No'.");
#endif

  defvar ("location", "etc/include/camas/layouts/simplest/", "Layout path", TYPE_DIR, "The path to search for layout files.");

  defvar ("update_method", "FileWatch", "Layout reload method", TYPE_STRING_LIST,
          "The method we use to reload the layout files if they are modified "
          "<ul><li>Filewatch: The files are reloaded after at most 60 seconds</li>"
          "<li>Fstat: The files are reloaded automatically and instantaneously</li>"
          "<li>None: The files are reloaded when you reload this module or restart the server</li></ul>"
          "<br/>All methods have their drawbacks, Fstat use more resources but is usefull when "
          "developping a template. Filewatch uses less ressources than Fstat but is not as fast as None "
          "and None is the fastest.",
          ({ "FileWatch", "Fstat", "None" }));

  // average user should not have to change the config file name
  defvar("configfilename",
          "config.xml",
          "Config file name",
          TYPE_LOCATION|VAR_EXPERT,
          "This is the file name of the config file. CAMAS will use this file for configuring your layout.");
}

//! method: int|string check_variables (string var, mixed value)
//!  Checks the variables set into the CIF fields
//! arg: string var
//!  The variable to check
//! arg: mixed value
//!  The data set for the variable
int|string check_variable (string var, mixed value)
{
  switch (var)
  {
    case "location": 
      files_loaded = 0;
      location = value;
      if(!config_file_path() || !file_stat(config_file_path()))
        return "Can't find configuration file " + QUERY(configfilename)
          + " for Camas layout in this directory";
      // unregister my old name and config from the layout manager
      my_configuration ()->get_provider ("camas_layout_manager")->remove_layout (this_object ());
      register_myname();
      break;
    case "_name":
      // unregister my old name and config from the layout manager 
      my_configuration ()->get_provider ("camas_layout_manager")->remove_layout (this_object ());
      register_myname(value);
      break;
  }
  return 0;
}

// set my layout name
void set_layout_name()
{
  layout_name = QUERY(_name);
}

// set my name, load configuration file from the FS, set my name in the CIF
// and register to the layout manager
void register_myname(void|string myname)
{
  // Load and check config
  load_config();
  check_config();
  string name;
  if(myname && sizeof(myname))
    name = myname;
  else
    name = config->name;
  if(!this_object()->is_default)
    defvar("_name", name, " Module name", TYPE_STRING|VAR_MORE,
               "An optional name. Set to something to remind you what "
               "the module really does.");
  layout_name = name;
  // Load the files in memory
  load_layout_files ();

  // Some button modes don't need to load a file
  if(config->buttonmode != "icon" && config->buttonmode != "form-button")
    load_buttons();
  my_configuration()->get_provider ("camas_layout_manager")->add_layout (this_object ());
}

//! method: void start()
//!  Function called when the module is started
void start (int num, object conf)
{
  // this modules depends on the layout manager
  module_dependencies (conf, ({ "camas_layout_manager" }));
  location = QUERY(location);
  if(Stdio.is_dir(location))
  {
    set_layout_name();
    register_myname(layout_name);
    if (layout_name == "")
      report_error("A CAMAS layout with no name can't be used\n");
  }
}

//! method: void stop ()
//!  Function called when the module is stopped
void stop ()
{
  if (layout_name != "")
    my_configuration ()->get_provider ("camas_layout_manager")->remove_layout (this_object ());
}

//! method: string status ()
//!  Displays status in the caudium configuration interface
string status ()
{
  string ret = "CAMAS Layout Filesystem: " + layout_name + "<br /><br />";

  // screens
  ret += "<b>Screens:</b>";
  if (sizeof (screens))
  {
    ret += "<ul>";
    foreach (sort (indices (screens)), string s)
    {
      ret += "\n<li>" + s + "<ul>";
      foreach(indices(screens[s]), string frame)
      {
        ret += "<li>From file " + screens[s][frame]->path + 
          "<br/>Last modified at " + ctime(screens[s][frame]->modified) + "</li>\n";
      }
      ret += "</ul>";
      if (screens[s]->default_screen)
        ret += " <font size=-1><i>(loaded from the default file)</i></font>";
      ret += "</li>";
    }
    ret += "</ul>";
  }
  else
    ret += " none";

  // buttons
  ret += "<br /><b>Buttons:</b>";
  if (config->buttonmode && sizeof(config->buttonmode))
    ret += "<ul><li>Using the "+config->buttonmode+" button mode</li></ul>";
  else
    ret += " none";

  ret += "<br /><b>Other informations:</b><table border=\"1\">";
  // remove the buttonmode index from the infos mapping
  mapping(string:string) infos = config - ([ "buttonmode": 0 ]);
  foreach(indices(infos), string property)
    ret += "<tr><td>" + property + "</td><td>" + infos[property] + "</td></tr>";
  ret += "</table>";
  return ret;
}

//! method: string query_provides ()
//!  Returns the name of the service this module provides
string query_provides ()
{
  return "camas_layout";
}

// This module needs it's own debug function because we can't do QUERY(debug)
// in start()
void write_debug(string msg)
{
#ifdef CAMAS_DEBUG
  report_debug("CAMAS DEBUG\t"+__FILE__+"@"+__LINE__+": "+msg+"\n");
#endif
}

/* Management */

//! method: string name ()
//!  Returns the name of the layout 
string name ()
{
  return layout_name;
}

//! method: string layout_root ()
//!  Returns root directory for the layout files in the filesystem
string layout_root ()
{
  return location + "_www/";
}

//! method: string config_file_path ()
//!  Returns the filesystem path for the layout config file
string config_file_path ()
{
  string file = combine_path(location, QUERY(configfilename));
  write_debug("config_file: "+file);
  return file;
}

//! method: void load_config ()
//!  Loads config file for the layout
void load_config()
{
  string file = config_file_path();
  array fstat = file_stat (file);

  write_debug("load_config: "+file);

  if(fstat)
  {
    parse_config();
  }
  else
  {
    report_error("Config file is missing. Layout '" + name() + "' will not work!\n");
  }
}

//! method: void parse_config ()
//!  Parse the config file and stores the config into a mapping
void parse_config ()
{
  object(Parser.XML.Tree.Node) rootnode    = Parser.XML.Tree.parse_file(config_file_path());
  // TODO: identify nodes given their name
  object(Parser.XML.Tree.Node) datanode; 
  
  if(objectp(rootnode))
    datanode = rootnode[1];
  else
  {
    report_error("CAMAS layout FS: not OK for datanode \n");
    return;
  }
#if constant(thread_create)
  object lock = global_lock->lock();
#endif
  mixed err = catch {
    // TODO: check the version attribute for future versions of config files
    for(int count=0; count<datanode->count_children(); count++)
    {
      if(datanode[count]->get_node_type() == Parser.XML.Tree.XML_ELEMENT)
      {
        string value;
        string index = datanode[count]->get_tag_name();
        array replacements;
        if(index != "description")
          replacements = ({ "\n", "\r" });
        else
          replacements = ({ "", "" }); 
        value = replace(datanode[count]->value_of_node(), replacements, ({ "", "" }));
        value = String.trim_whites(value);
        config += ([ index: value ]);
      }
    }
  };
#if constant(thread_create)
      destruct(lock);
#endif
  if(err)
    report_error("error in camas_layout_fs.pike: %s\n", describe_backtrace(err));
  write_debug(sprintf("config: %O\n",config));
}

//! method: void check_config ()
//!  Checks the config extracted from the config file
void check_config()
{
#if constant(thread_create)
  object lock = global_lock->lock();
#endif
  mixed err = catch {

    // put a random random name if none has been specified
    if(!config->name)
    {
      string foo = " " * RANDOM_NAME_SIZE;
      for(int i=0; i<RANDOM_NAME_SIZE; i++)
        foo[i]=(random(25)+97);
      config->name = foo;
    }
    
    config->framesetfilename = config->framesetfilename || "frameset.rxml";
    config->layoutfilename = config->layoutfilename || "layout.rxml";
    config->description = config->description || "Layout without a description";
    config->author = config->author || "unknown";
    config->license = config->license || "unknown";
    
    // TODO: ask the formbutton modules the correct values and check against it
    config->buttonmode = config->buttonmode || "form-button";
  };
#if constant(thread_create)
  destruct(lock);
#endif
  if(err)
    report_error("error in camas_layout_fs.pike: %s\n", describe_backtrace(err));
}

//! method: void flush_rxml_caches(string screen)
//!  Flush the RXML caches in Camas tags for the given screen
void flush_rxml_caches(string screen)
{
  object camas_tags = my_configuration()->get_provider("camas_tags");
  
  // flush the cache for the root tags and containers
  if(camas_tags->parser)
    camas_tags->parser->flush_cache();
  
  // flush this screen rxml cache
  array camas_tags_screen = camas_tags->program_screens[upper_case(screen)];
  
  if(camas_tags_screen && objectp(camas_tags_screen[0]) && 
      objectp(camas_tags_screen[0]->parser))
    camas_tags_screen[0]->parser->flush_cache();
}

/* Layouts */

//! method: string layout_folder_path (string screen)
//!  Returns the path for the folder for the screen
//! arg: string screen
//!  Screen name.
string layout_folder_path(string screen)
{
  return combine_path(location, screen);
}

//! method: string layout_file_path (string screen, string camasframe)
//!  Returns the path for the screen file
//! arg: string screen
//!  Screen name. When void, default screen.
//! arg: string camasframe
//!  CAMAS frame. When void, default to _top.
string layout_file_path(string screen, string camasframe)
{
  return combine_path(layout_folder_path(screen), camasframe+".rxml");
}

//! method: void load_frame(string screen, string frame, string frame_file)
//!  Loads the given frame for the given screen from the file system
//! arg: string screen
//!  The screen to load
//! arg: string frame
//!  The frame to load
//! arg: string frame_file
//!  The file where to find the data for the screen/frame 
void load_frame(string screen, string frame, string frame_file)
{
  if(frame_file[sizeof(frame_file)-5..]==".rxml" && file_stat(frame_file))
  {
    write_debug("Loading frame " + frame_file + " for screen " + screen);
    screens[screen] +=
    ([
      frame :
           ([
             "data"     : Stdio.read_bytes(frame_file),
             "modified" : file_stat(frame_file)[3],
             "path"     : frame_file,
           ])
    ]);
    if(QUERY(update_method) == "FileWatch")
    {
      // we load all the files but it should not happen often and we don't have
      // a way to give load_frame the arguments it needs
      filewatches += ({ FileWatch.Callout(frame_file, 60, load_layout_files) }); 
    }
  }
}

//! method: void load_layout_files ()
//!  Loads screen config and button files for the layout
void load_layout_files ()
{
  // load screens
  foreach (indices (screennames), int screen_id)
  {
    string screen = screennames[screen_id][0];
    string _layout_folder_path = layout_folder_path(screen);

    // Loop over all the known screens
    if(Stdio.is_dir(_layout_folder_path)) 
    {
      screens += ([ screen: ([ ]) ]);
    
      array frames_files = get_dir(_layout_folder_path);

      // find all the frames for the current screen
      foreach(frames_files, string frame_file)
        load_frame(screen, frame_file[..sizeof(frame_file)-6],
            combine_path(_layout_folder_path, frame_file));
    }
  }

  // find all the default frames
  string _layout_folder_path = layout_folder_path("");
  array frames_files = get_dir(_layout_folder_path);
  foreach(frames_files, string frame_file)
    load_frame("_default", frame_file[..sizeof(frame_file)-6],
        combine_path(_layout_folder_path, frame_file));
  files_loaded = 1;
}

//! method: string get_layout_data (int screen_id, string camasframe)
//!  Returns the data for the screen
//! arg: int screen_id
//!  ID of the screen
//! arg: string camasframe
//!  Name of the current camasframe
string get_layout_data (int screen_id, string camasframe)
{
  string screen = screennames[screen_id][0];
  return load_screen(screen, camasframe); 
}

//! method: string load_screen(screen, camasframe)
//!  Method for loading screen for a file
string load_screen(string screen, string camasframe)
{
  string data;
  if(!config || !sizeof(config))
  {
    report_error("Can't load screen '" + screen + "', config file not found for layout"
        " '" + name() + "'");
    return "Fatal error: Config file not found for layout '" + name() + "'";
  }

  // Frame file for the screen
  // /path/to/layout/screen/camasframe.rxml
  data = fetch_data(screen, camasframe);
  if(stringp(data))
    return data;

  // Default frame file at the root of the layout
  // /path/to/layout/camasframe.rxml
  data = fetch_data("_default", camasframe);
  if(stringp(data))
    return data;

#ifdef EARLY_LAYOUT_COMPAT
  if(camasframe=="_top")
  {
    string framesetfile = config->framesetfilename -= ".rxml";
    
    // Compat frameset file for the screen
    // /path/to/layout/screen/frameset.rxml
    data = fetch_data(screen, framesetfile);
    if(stringp(data))
      return data;
   
    // Compat default frameset file at the root of the layout
    // /path/to/layout/frameset.rxml
    data = fetch_data("_default", framesetfile);
    if(stringp(data))
      return data;
  }
#endif
  
  string layoutfile = config->layoutfilename -= ".rxml";
  
  // Default layout file for the screen
  // /path/to/layout/screen/layout.rxml
  data = fetch_data(screen, layoutfile);
  if(stringp(data))
    return data;
   
  // Default default layout file at the root of the layout
  // /path/to/layout/layout.rxml
  data = fetch_data("_default", layoutfile);
  if(stringp(data))
      return data;
  
  report_error("CAMAS Layout FS: No layout found for screen "+screen+" and frame "+camasframe+"\n");
}

//! method: int|string fetch_data(string screen, string frame)
//!  Check in the cached layout if data is available.
//! arg: string screen
//!  The screen for which data is wanted for
//! arg: string frame
//!  The frame for which data is wanted for
//! returns:
//!  Layout data for the screen/frame is found
//!  0 is no data is found
int|string fetch_data(string screen, string frame)
{
  if(
      mappingp(screens[screen]) &&
      mappingp(screens[screen][frame]) &&
      stringp(screens[screen][frame]->data))
  {
    if(QUERY(update_method) == "Fstat")
    {
      array fstat = file_stat((screens[screen][frame]->path));
      int newer = (fstat) ? (fstat[3] > screens[screen]->modified) : 0;
      if(newer)
      {
        load_frame(screen, frame, screens[screen][frame]->path);
        flush_rxml_caches (screen);
      }
    }
    return screens[screen][frame]->data;  
  }

  write_debug("layout for screen '"+screen+"' and frame '"+frame+"' does not exist\n");
  
  return 0;
}

/* Buttons */

//! method: string button_mode ()
//!  Returns button mode of the layout
string button_mode ()
{
  CDEBUG("buttonmode: "+config->buttonmode);
  return (string)config->buttonmode;
}

//! method: string buttons_file (string mode)
//!  Return the filesystem path for the layout button file given the button mode
string buttons_file (string mode)
{
  string file = QUERY (location);

  switch(mode)
  {
  case "gbutton":
    file += "gbutton.txt";
    break;

  case "camas-mode-basic":
    file += "camasbasic.txt";
    break;

	case "icon":
  case "form-button":
  default:
    file = ""; // path is set to nothing since there is no button file
  }

  return file;
}

//! method: void load_buttons ()
//!  Loads buttons file for the layout
void load_buttons()
{
#if constant(thread_create)
  object lock = global_lock->lock();
#endif
  mixed err = catch {
    string file = buttons_file(config->buttonmode);
    array fstat = file_stat (file);

    write_debug("load_buttons: " + file);

    if(fstat)
    {
      write_debug("Button file found");

      switch(config->buttonmode)
      {
      case "gbutton":
        gbuttonsyntax = Stdio.read_bytes(file);
        write_debug("load_buttons: gbuttonsyntax = "+gbuttonsyntax);
        break;

      case "camas-mode-basic":
        basicbuttonlist = CAMAS.FormButtons.Basic.File(file)->doindex();
        write_debug("load_buttons: basicbuttonlist = "+sprintf("%O",basicbuttonlist));
        break;
      }
      //buttonmode = CAMAS.FormButtons.Basic.File(file)->get_mode();
      //CDEBUG("button mode: "+buttonmode);
    }
    else
    {
      write_debug(sprintf("Buttons file %O is missing. Reverting to form-button.", file));
      config->buttonmode = "form-button";
    }
  };
#if constant(thread_create)
  destruct(lock);
#endif
  if(err)
    report_error("error in camas_layout_fs.pike: %s\n", describe_backtrace(err));
}

//! method: get_gbutton_syntax ()
//!   Returns the syntax for gbutton
string get_gbutton_syntax ()
{
  CDEBUG("gbutton syntax: "+gbuttonsyntax);
  return gbuttonsyntax;
}

//! method: get_basic_buttonlist ()
//!  Returns the mapping containing the basic button data
mapping get_basicbutton_list ()
{
  return basicbuttonlist;
}

//! method: get_icon_extension ()
//!  Returns the extension of the icons
//!  Only used when the buttonformat is set to icon
string get_icon_extension()
{
	return config->iconformat ? config->iconformat : "";	
}

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: debug
//! Sends debug information into Caudium error log
//!  type: TYPE_FLAG
//!  name: Debug
//
//! defvar: name
//! The name of the layout. Don't use the name 'Default' or you will experience conflicts and errors with the default layout.
//!  type: TYPE_STRING
//!  name: Name
//
//! defvar: location
//! The path to search for layout files.
//!  type: TYPE_DIR
//!  name: Layout path
//
//! defvar: configfilename
//! This is the file name of the config file. CAMAS will use this file for configuring your layout.
//!  type: TYPE_LOCATION
//!  name: Config file name
//

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

