(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** The tree widget to display directories. *)

let file_exists f = 
  try let _ = Unix.stat f in true
  with _ -> false


let is_prefix f1 f2 =
  let len1 = String.length f1 in
  let len2 = String.length f2 in
  (len1 < len2) &&
  (String.sub f2 0 len1) = f1
    

class ['a] box (behav : 'a Ocvs_behav.tree_behaviour) =
  let vbox = GPack.vbox () in
  let wscroll = GBin.scrolled_window
      ~vpolicy: `AUTOMATIC
      ~hpolicy: `AUTOMATIC
      ~packing: (vbox#pack ~expand: true)
      () 
  in
  let wtree = GTree.tree ~packing: wscroll#add_with_viewport () in
  let _ = wtree#set_selection_mode `SINGLE in
  
  object(self)
    val mutable selection = (None : string option)
    val mutable selected_node = (None : GTree.tree_item option)

    method box = vbox

    method selection = selection

    method select_dir dir item =
      selection <- Some dir ;
      selected_node <- Some item ;
      behav#select dir

    method unselect_dir dir =
      selection <- None ;
      selected_node <- None ;
      behav#unselect dir

    method insert_node wt dirname basename =
      let complete_name = Filename.concat dirname basename in
      let item = GTree.tree_item
          ~label: basename
          ~show: true
          ()
      in
      let _ = wt#append item in
      let _ = item#connect#select (fun () -> self#select_dir complete_name item) in
      let _ = item#connect#deselect (fun () -> self#unselect_dir complete_name) in
      let _ = item#connect#expand (fun () -> behav#add_expanded_dir complete_name) in
      let _ = item#connect#collapse (fun () -> behav#remove_expanded_dir complete_name) in
      let subdirs = Ocvs_misc.get_cvs_directories complete_name in
      match subdirs with
        [] ->
          ()
      | l ->
          let wt_fils = GTree.tree () in
	  wt_fils#set_selection_mode `SINGLE ;
          let _ = item#set_subtree wt_fils in

          if behav#expand_dir complete_name then item#expand () ;

          List.iter 
	    (self#insert_node wt_fils complete_name)
            (List.sort compare l);
          (* connect the press on button 3 for contextual menu *)
	  let _ = wt_fils#event#connect#button_press ~callback:
	    (
	     fun ev ->
	       GdkEvent.Button.button ev = 3 &&
	       GdkEvent.get_type ev = `BUTTON_PRESS &&
	       (
		GToolbox.popup_menu 
		  ~button: 3
		  ~time: 0 
		  ~entries: (List.map 
			       (fun (s,f) -> `I (s, f))
			       (behav#menu_ctx self#selection)
			    );
		true
	       )
	    )
	  in
	  ()
	  
    method update =
      (
       match selection with
	 None -> ()
       | Some dir ->
	   selection <- None ;
	   self#unselect_dir dir
      );
      selected_node <- None;
      let _ = wtree#remove_items (wtree#children) in
      List.iter (self#insert_node wtree "") behav#roots
     
    method update_selected_dir =
      match selection, selected_node with
      |	Some d, Some item ->
	  let _ = 
	    match item#subtree with 
	      None -> ()
	    | Some _ -> item#remove_subtree () 
	  in
	  let subdirs = Ocvs_misc.get_cvs_directories d in
	  (
	   match subdirs with
	     [] ->
	       ()
	   | l ->
	       let wt_fils = GTree.tree () in
	       wt_fils#set_selection_mode `SINGLE ;
	       let _ = item#set_subtree wt_fils in
               let _ = item#expand () in
               List.iter 
		 (self#insert_node wt_fils d)
		 (List.sort compare l)
	  );
	  self#select_dir d item
      |	_ ->
	  ()

    method cvs_update_dir =
      match selection, selected_node with
      |	Some d, Some item ->
	  (
	   (* A VOIR : demander les autorisations pour les fichiers ? *)
	   try let _ = behav#cvs_update_dir d in ()
	   with Failure s -> GToolbox.message_box Ocvs_messages.error s
	  );
	  self#update_selected_dir
	    (* A VOIR : mettre  jour les lments dans data ? 
	       Non, car behav est au courant des fichiers modifis
	       et se mettra  jour tout seul.*)
      |	_, _ ->
	  ()

    method cvs_commit_dir =
      match selection, selected_node with
      |	Some d, Some item ->
	  (
	   (* A VOIR : demander les autorisations pour les fichiers ? *)
	   let com_opt = GToolbox.input_text
	      Ocvs_messages.enter_comment 
	      (Ocvs_messages.enter_comment_commit^" : ")
	   in
	   match com_opt with
	     None -> ()
	   | Some comment ->
	       (
		try let _ = behav#cvs_commit_dir ~comment: comment d in ()
		with Failure s -> 
		  GToolbox.message_box Ocvs_messages.error s
	       );
	       self#update_selected_dir
	       (* A VOIR : mettre  jour les lments dans data ? 
		  Non, car behav est au courant des fichiers modifis
		  et se mettra  jour tout seul.*)
	  )
      |	_, _ ->
	  ()

    method cvs_tag_dir =
      match selection, selected_node with
      |	Some d, Some item ->
	  (
	   (* A VOIR : demander les autorisations pour les fichiers ? *)
	   let tag_opt = GToolbox.input_string
	       Ocvs_messages.m_tag_dir
	       (Ocvs_messages.enter_tag_for_dir d)
	   in
	   match tag_opt with
	     None -> ()
	   | Some tag ->
	       let confirm s = 
		 (GToolbox.question_box 
		    ~title: Ocvs_messages.mConfirm
		    ~buttons: [ Ocvs_messages.mYes ; Ocvs_messages.mNo ]
		    s) = 1
	       in
	       try behav#cvs_tag_dir confirm tag d
	       with Failure s -> GToolbox.message_box Ocvs_messages.error s
	  )
      |	_, _ ->
	  ()

    method cvs_add_dir =
      match selection, selected_node with
	Some d, Some item ->
	  (
	   match GToolbox.select_file ~dir: (ref d) ~title: Ocvs_messages.add_dir () with
	     Some new_d ->
	       (
		try
		  if file_exists new_d then
		    (
		     behav#cvs_add_dir new_d;
		     if is_prefix d new_d then 
		       self#update_selected_dir 
		     else 
		       self#update
		    )
		  else
		    (* ask for confirmation to create the directory *)
		    match GToolbox.question_box 
			~title: Ocvs_messages.add_dir
			~buttons: [ Ocvs_messages.mYes ; Ocvs_messages.mNo ]
			(Ocvs_messages.should_create_dir new_d)
		    with
		      1 ->
			behav#cvs_create_and_add_dir new_d;
			if is_prefix d new_d then 
			  self#update_selected_dir 
			else 
			  self#update
		    | _ ->
			()
		with
		  Failure s ->
		    GToolbox.message_box Ocvs_messages.error s
	       )
	   | None ->
	       ()
	  )
      | _ ->
	  ()

    method private real_cvs_add_files binary =
      let start_dir = 
	match selection with 
	  None -> 
	    (
	     match behav#roots with
	       [] -> Unix.getcwd ()
	     | s :: _ -> s
	    )
	| Some d -> d 
      in
      (* A VOIR : a changer quand on aura la slection multiple
	 dans select_files *)
      let add f =
	let (ok, ko) = behav#cvs_add_files ~binary: binary [f] in
	match ok with
	  [] ->
	    GToolbox.message_box Ocvs_messages.error
	      (Ocvs_messages.error_add_files ko)
	| _ ->
	    ()
      in
      (
       match GToolbox.select_file
	   ~dir: (ref start_dir)
	   ~title: Ocvs_messages.add_files
	   ()
       with
	 Some f -> add f
       | None -> ()
      );
      match selection, selected_node with
      | Some d, Some i -> 
	  self#unselect_dir d;
	  self#select_dir d i
      | _ ->
	  ()

    method cvs_add_files = self#real_cvs_add_files false
    method cvs_add_binary_files = self#real_cvs_add_files true

    initializer

      (* connect the press on button 3 for contextual menu *)
      let _ = wtree#event#connect#button_press ~callback:
	(
	 fun ev ->
	   GdkEvent.Button.button ev = 3 &&
	   GdkEvent.get_type ev = `BUTTON_PRESS &&
	   (
	    GToolbox.popup_menu
	      ~button: 3
	      ~time:0 
	      ~entries: (List.map 
			   (fun (s,f) -> `I (s, f))
			   (behav#menu_ctx self#selection)
			);
	    true
	   )
	)
      in

      self#update
  end
