(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** Data for the description of a report. *)

open Rep_desc

type branch = Then | Else

(** Create the given empty file. *)
let create_file f =
  try let oc = open_out f in close_out oc
  with Sys_error s -> prerr_endline s

(** Open the given file, creating it if it does not exist. *)
let open_file ?(bin=false) f = 
  (try ignore (Unix.stat f)
  with Unix.Unix_error _ -> create_file f
  );
  (if bin then open_in_bin else open_in) f 


class data ?(gui=true) file =
  object (self)
    val mutable report = { rep_params = [] ; rep_header = "" ; rep_eles = []}
    val mutable changed = false
    val mutable file = file

    method report = report

    method set_changed b = changed <- b
    method changed = changed
    method set_file f = file <- f
    method save =
      try
	if !Rep_args.binary_output then
	  (
	   let chanout = open_out_bin file in
	   output_value chanout report ;
	   close_out chanout 
	  )
	else
	  (
	   let chanout = open_out file in
	   let fmt = Format.formatter_of_out_channel chanout in
	   Format.pp_open_box fmt 0;
	   Rep_desc.xprint_report fmt report; 
	   Format.pp_close_box fmt ();
	   Format.pp_print_flush fmt ();
	   close_out chanout
	  );
	changed <- false
      with
	Sys_error s -> 
	  raise (Failure s)

    method load =
      try
	if !Rep_args.binary_input then
	  (
	   let chanin = open_file ~bin: true file in
	   (try
	     report <- input_value chanin 
	   with
	     End_of_file ->
	       report <- { rep_params = [] ; rep_header = "" ; rep_eles = [] }
	   );
	   close_in chanin
	  )
	else
	  (
	   let chanin = open_file file in
	   (try
	     let tree = IoXML.parse_xml (Stream.of_channel chanin) in
	     report <- Rep_desc.xparse_report tree
	   with
	     e ->
	       let e = 
		 match e with 
		   IoXML.ExcLoc ((bp,ep), e) ->
		     prerr_endline ("Error at location "^(string_of_int bp)^" -> "^(string_of_int ep));
		     e
		 | e ->
		     e
	       in
	       report <- { rep_params = [] ; rep_header = "" ; rep_eles = [] }
	   );
	   close_in chanin
	  );
	changed <- false
      with
	Sys_error s ->
	  raise (Failure s)

    method file = (file : string)

    method remove_from_parent ele parent_opt =
      match parent_opt with
	None ->
	  report.rep_eles <- (List.filter (fun e -> e != ele) report.rep_eles) ;
	  self#set_changed true 
      |	Some (Tag t) ->
	  t.tag_subs <- (List.filter (fun e -> e != ele) t.tag_subs) ;
	  self#set_changed true 
      |	Some (List l) ->
	  l.list_subs <- (List.filter (fun e -> e != ele) l.list_subs) ;
	  self#set_changed true 
      |	Some (Rep_desc.Then c) ->
	  c.subs_then <- (List.filter (fun e -> e != ele) c.subs_then) ;
	  self#set_changed true 
      |	Some (Rep_desc.Else c) ->
	  c.subs_else <- (List.filter (fun e -> e != ele) c.subs_else) ;
	  self#set_changed true 
      |	Some (Sub _)
      |	Some (Cond _)
      |	Some (Mark _)
      |	Some (Leaf _) -> ()

    method append_in_parent ele parent_opt =
      match parent_opt with
	None -> 
	  report.rep_eles <- report.rep_eles @ [ele] ;
	  self#set_changed true 
      |	Some (Tag t) -> 
	  t.tag_subs <- t.tag_subs @ [ele] ;
	  self#set_changed true 
      |	Some (List l) -> 
	  l.list_subs <- l.list_subs @ [ele] ;
	  self#set_changed true 
      |	Some (Rep_desc.Then c) ->
	  c.subs_then <- c.subs_then @ [ele] ;
	  self#set_changed true 
      |	Some (Rep_desc.Else c) ->
	  c.subs_else <- c.subs_else @ [ele] ;
	  self#set_changed true 
      |	Some (Sub _)
      |	Some (Cond _)
      |	Some (Mark _)
      |	Some (Leaf _) -> ()

    method set_parameters l =
      report.rep_params <- l ;
      self#set_changed true

    method set_header s =
      report.rep_header <- s;
      self#set_changed true
	
    method up_element ele parent_opt =
      let rec f = function
	  ele1 :: ele2 :: q -> 
	    if ele2 == ele then
	      ele2 :: ele1 :: q
	    else
	      ele1 :: (f (ele2 :: q))
	| l -> l
      in
      match parent_opt with
	None -> 
	  report.rep_eles <- f report.rep_eles ;
	  self#set_changed true 
      |	Some (Tag t) -> 
	  t.tag_subs <- f t.tag_subs ;
	  self#set_changed true 
      |	Some (List l) -> 
	  l.list_subs <- f l.list_subs ;
	  self#set_changed true 
      |	Some (Rep_desc.Then c) ->
	  c.subs_then <- f c.subs_then ;
	  self#set_changed true 
      |	Some (Rep_desc.Else c) ->
	  c.subs_else <- f c.subs_else ;
	  self#set_changed true 
      |	Some (Sub _)
      |	Some (Cond _)
      |	Some (Mark _)
      |	Some (Leaf _) -> ()

    initializer
      self#load
  end
