/* gnu.classpath.tools.doclets.xmldoclet.Driver
   Copyright (C) 2001 Free Software Foundation, Inc.

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.
 
GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA. */

package gnu.classpath.tools.doclets.xmldoclet;

import com.sun.javadoc.*;
import java.io.*;
import java.util.Arrays;
import java.util.Comparator;

/**
 *  A Doclet which retrieves all information presented by the Doclet
 *  API, dumping it to stdout in XML format.
 *
 *  @author Julian Scheid
 */
public class Driver {

   /**
    *  All XML output will go to this stream.
    */
   private PrintWriter out;

   /**
    *  How many spaces to indent each XML node level,
    *  i.e. Tab size for output.
    */
   private static int indentStep = 1;

   /**
    *  Won't output superfluous spaces if set to true.
    *  If set to false, output will be more legible.
    */
   private boolean compress = false;

   /**
    *  Won't output warning messages while fixing
    *  HTML code if set to true.
    */
   private boolean noHTMLWarn = false;

   /**
    *  Won't output warning messages when encountering tags
    *  that look like an email address if set to true.
    */
   private boolean noEmailWarn = false;

   /**
    *  Will fix HTML if necessary so that each comment
    *  contains valid XML code if set to true. If set
    *  to false, HTML code will not be modified and
    *  instead encapsulated in a CDATA section.
    */
   private boolean fixHTML = false;

   /**
    *  Name of the directory files will be written to.
    */
   private String targetDirectory = ".";

   /**
    *  Stores the Doclet API RootDoc we are operating on.
    */
   private RootDoc rootDoc;

   /**
    *  XML namespace prefix used for all tags, except for HTML
    *  tags copied from Javadoc comments. Excluding colon.
    */
   public static final String tagPrefix = "gjdoc";

   /**
    * The current class that is being processed.
    * Set in outputClassDoc().
    */
   private ClassDoc currentClass;

   /**
    * The current member that is being processed.
    * Set in outputMemberDoc().
    */
   private MemberDoc currentMember;

   /**
    * The current constructor/method that is being processed.
    * Set in outputExecutableMemberDoc().
    */
   private ExecutableMemberDoc currentExecMember;

   /**
    *  Official Doclet entry point.
    */
   public static boolean start(RootDoc rootDoc) {

      // Create a new XmlDoclet instance and delegate control.
      return new Driver().instanceStart(rootDoc);
   }

   /**
    *  Output an XML tag describing a com.sun.javadoc.Type object.
    *  Assumes that the tag does not have subtags.
    *
    *  @param level  Level of indentation. Will be multiplied by 
    *                <code>indentStep</code> to yield actual amount
    *                of whitespace inserted at start of line.
    *  @param tag    Identifier for the XML tag being output.
    *  @param type   The Javadoc Type to be output.
    */
   protected void outputType(int level, String tag, Type type) {
      outputType(level, tag, type, true);
   }

   protected void outputType(int level, String tag, Type type, boolean atomic) {
      println(level, "<"+tagPrefix+":"+tag+" typename=\""+type.typeName()+"\""+
	      " qualifiedtypename=\""+type.qualifiedTypeName()+"\""
	      +(type.dimension().length()==0?"":" dimension=\""+type.dimension()+"\"")
	      +(atomic?"/":"")+">");
   }

   protected void outputExecutableMemberDocBody(int level, ExecutableMemberDoc memberDoc) {

      currentExecMember = memberDoc;

      outputMemberDocBody(level, memberDoc);

      Parameter[] parameters = memberDoc.parameters();
      for (int i=0, ilim=parameters.length; i<ilim; ++i) {
	 Parameter parameter = parameters[i];
	 outputType(level, "parameter name=\""+parameter.name()+"\"", parameter.type());
      }

      ClassDoc[] exceptions = memberDoc.thrownExceptions();
      for (int i=0, ilim=exceptions.length; i<ilim; ++i) {
	 ClassDoc exception = exceptions[i];
	 outputType(level, "thrownException", exception);
       }

      printAtomTag(level, "signature full=\""+memberDoc.signature()+"\" flat=\""+memberDoc.flatSignature()+"\"");

      if (memberDoc.isNative()) {
	 printAtomTag(level, "isNative");
      }

      if (memberDoc.isSynchronized()) {
	 printAtomTag(level, "isSynchronized");
      }
   }

   protected void outputMethodDoc(int level, MethodDoc methodDoc) {
      println();
      printOpenTag(level, "methoddoc name=\""+methodDoc.name()+"\"");
      outputExecutableMemberDocBody(level+1, methodDoc);
      outputType(level+1, "returns", methodDoc.returnType());
      printCloseTag(level, "methoddoc");
   }

   protected void outputMemberDocBody(int level, MemberDoc memberDoc) {
      currentMember = memberDoc;
      outputProgramElementDocBody(level, memberDoc);
   }

   protected void outputFieldDocBody(int level, FieldDoc fieldDoc) {
      outputType(level, "type", fieldDoc.type());
      if (fieldDoc.isTransient()) {
	 printAtomTag(level, "isTransient");
      }
      if (fieldDoc.isVolatile()) {
	 printAtomTag(level, "isVolatile");
      }
   }

   private void outputFieldDoc(int level, FieldDoc fieldDoc) {
      println();
      printOpenTag(level, "fielddoc name=\""+fieldDoc.name()+"\"");
      outputMemberDocBody(level+1, fieldDoc);
      outputFieldDocBody(level+1, fieldDoc);
      printCloseTag(level, "fielddoc");
   }

   protected void outputConstructorDoc(int level, ConstructorDoc constructorDoc) {
      println();
      printOpenTag(level, "constructordoc name=\""+constructorDoc.name()+"\"");
      outputExecutableMemberDocBody(level+1, constructorDoc);
      printCloseTag(level, "constructordoc");
   }

   protected void outputSuperInterfacesRec(int level, ClassDoc classDoc) {
      if (null!=classDoc) {
	 ClassDoc[] interfaces = classDoc.interfaces();
	 for (int i=0, ilim=interfaces.length; i<ilim; ++i) {
	    outputType(level, "superimplements", interfaces[i]);
	 }
	 outputSuperInterfacesRec(level, classDoc.superclass());
      }
   }

   protected void outputClassDocSummary(ClassDoc classDoc) {
      println();
      printOpenTag(1, "classdoc name=\""+classDoc.name()+"\" qualifiedtypename=\""+classDoc.qualifiedName()+"\"");
      if (null!=classDoc.superclass()) {
	 outputType(2, "superclass", classDoc.superclass());
      }

      ClassDoc[] interfaces = classDoc.interfaces();
      for (int i=0, ilim=interfaces.length; i<ilim; ++i) {
	 outputType(2, "implements", interfaces[i]);
      }
      outputSuperInterfacesRec(2, classDoc.superclass());

      printAtomTag(2, "containingPackage name=\""+classDoc.containingPackage().name()+"\"");

      printCloseTag(1, "classdoc");
   }

   protected void outputPackageDoc(PackageDoc packageDoc) {
      println();
      printOpenTag(1, "packagedoc name=\""+packageDoc.name()+"\"");
      if (packageDoc.firstSentenceTags().length > 0) {
	 printOpenTag(2, "firstSentenceTags", false);
	 outputTags(3, packageDoc.firstSentenceTags(), true);
	 printCloseTag(0, "firstSentenceTags");
      }

      if (packageDoc.tags().length > 0) {
	 printOpenTag(2, "tags");
	 outputTags(3, packageDoc.tags(), true);
	 printCloseTag(2, "tags");
      }

      if (packageDoc.seeTags().length > 0) {
	 printOpenTag(2, "seeTags");
	 outputTags(3, packageDoc.seeTags(), true);
	 printCloseTag(2, "seeTags");
      }

      ClassDoc[] allClasses = (ClassDoc[]) packageDoc.allClasses().clone();
      Arrays.sort(allClasses, new Comparator() {
	    public int compare(Object o1, Object o2) {
	       return ((ClassDoc)o1).name().compareTo(((ClassDoc)o2).name());
	    }
	 });
      for (int i = 0, ilim = allClasses.length; i < ilim; ++ i) {
	 printAtomTag(2, "containsClass qualifiedtypename=\""+allClasses[i].qualifiedTypeName()+"\"");
      }

      printCloseTag(1, "packagedoc");
   }

   protected void outputClassDoc(ClassDoc classDoc) {

      currentClass = classDoc;

      println();
      printOpenTag(1, "classdoc xmlns=\"http://www.w3.org/TR/REC-html40\" xmlns:"+tagPrefix+"=\"http://www.gnu.org/software/cp-tools/gjdocxml\" name=\""+classDoc.name()+"\" qualifiedtypename=\""+classDoc.qualifiedName()+"\"");

      if (null!=classDoc.superclass()) {
	 outputType(2, "superclass", classDoc.superclass());
      }

      ClassDoc[] interfaces = classDoc.interfaces();
      for (int i=0, ilim=interfaces.length; i<ilim; ++i) {
	 outputType(2, "implements", interfaces[i]);
      }
      outputSuperInterfacesRec(2, classDoc.superclass());

      outputProgramElementDocBody(2, classDoc);
      if (classDoc.isAbstract())
	 printAtomTag(2, "isAbstract");
      if (classDoc.isSerializable())
	 printAtomTag(2, "isSerializable");
      if (classDoc.isExternalizable())
	 printAtomTag(2, "isExternalizable");
      if (classDoc.definesSerializableFields())
	 printAtomTag(2, "definesSerializableFields");

      ConstructorDoc[] constructors = classDoc.constructors();
      for (int i=0, ilim=constructors.length; i<ilim; ++i) {
	 outputConstructorDoc(2, constructors[i]);
      }

      MethodDoc[] methods = classDoc.methods();
      for (int i=0, ilim=methods.length; i<ilim; ++i) {
	 outputMethodDoc(2, methods[i]);
      }

      FieldDoc[] fields = classDoc.fields();
      for (int i=0, ilim=fields.length; i<ilim; ++i) {
	 outputFieldDoc(2, fields[i]);
      }

      printCloseTag(1, "classdoc");

      currentClass = null;
      currentMember = null;
      currentExecMember = null;
   }

   protected void outputDocBody(int level, Doc doc) {
      if (doc.isClass()) {
	 printAtomTag(level, "isClass");
      }
      if (doc.isConstructor()) {
	 printAtomTag(level, "isConstructor");
      }
      if (doc.isError()) {
	 printAtomTag(level, "isError");
      }
      if (doc.isException()) {
	 printAtomTag(level, "isException");
      }
      if (doc.isField()) {
	 printAtomTag(level, "isField");
      }
      if (doc.isIncluded()) {
	 printAtomTag(level, "isIncluded");
      }
      if (doc.isInterface()) {
	 printAtomTag(level, "isInterface");
      }
      if (doc.isMethod()) {
	 printAtomTag(level, "isMethod");
      }
      if (doc.isOrdinaryClass()) {
	 printAtomTag(level, "isOrdinaryClass");
      }
      if (doc.inlineTags().length > 0) {
	 printOpenTag(level, "inlineTags", false);
	 outputTags(level+1, doc.inlineTags(), true);
	 printCloseTag(0, "inlineTags");
      }

      if (doc.firstSentenceTags().length > 0) {
	 printOpenTag(level, "firstSentenceTags", false);
	 outputTags(level+1, doc.firstSentenceTags(), true);
	 printCloseTag(0, "firstSentenceTags");
      }

      if (doc.tags().length > 0) {
	 printOpenTag(level, "tags");
	 outputTags(level+1, doc.tags(), true);
	 printCloseTag(level, "tags");
      }

      if (doc.seeTags().length > 0) {
	 printOpenTag(level, "seeTags");
	 outputTags(level+1, doc.seeTags(), true);
	 printCloseTag(level, "seeTags");
      }
   }

   protected void outputProgramElementDocBody(int level, ProgramElementDoc programElementDoc) {
      outputDocBody(level, programElementDoc);
      printAtomTag(level, "containingPackage name=\""+programElementDoc.containingPackage().name()+"\"");
      if (null!=programElementDoc.containingClass()) {
	 outputType(level, "containingClass", programElementDoc.containingClass());
      }
      String access;
      if (programElementDoc.isPublic()) 
	 access="public";
      else if (programElementDoc.isProtected()) 
	 access="protected";
      else if (programElementDoc.isProtected()) 
	 access="protected";
      else if (programElementDoc.isPackagePrivate()) 
	 access="package";
      else
	 throw new RuntimeException("Huh? "+programElementDoc+" is neither public, protected, protected nor package protected.");
      printAtomTag(level, "access scope=\""+access+"\"");
      if (programElementDoc.isFinal())
	 printAtomTag(level, "isFinal");
      if (programElementDoc.isStatic())
	 printAtomTag(level, "isStatic");
   }

   protected void outputTags(int level, Tag[] tags, boolean descend) {

      HtmlWell htmlWell = new HtmlWell(rootDoc, noHTMLWarn, noEmailWarn,
				       currentClass, currentMember);

      for (int i=0, ilim=tags.length; i<ilim; ++i) {
	 Tag tag = tags[i];
	 if (!"Text".equals(tag.name())) {
	    printOpenTag(0 /* don't introduce additional whitespace */, 
			 "tag kind=\""+tag.kind()+"\" name=\""+tag.name()+"\"", false);
	 }
	 if (tag instanceof ThrowsTag) {
	    ThrowsTag throwsTag = (ThrowsTag)tag;
	    if (null!=throwsTag.exception()) {
	       outputType(level+1, "exception", throwsTag.exception());
	    }
	    else {
	       StringBuffer sb = new StringBuffer("In ThrowsTag: Exception ");
	       sb.append(throwsTag.exceptionName());
	       sb.append(" not found in ");
	       if (currentExecMember instanceof MethodDoc) {
		   MethodDoc m = (MethodDoc)currentExecMember;
		   sb.append(m.returnType().typeName());
		   sb.append(m.returnType().dimension());
		   sb.append(' ');
	       }
	       sb.append(currentClass.qualifiedName());
	       sb.append('.');
	       sb.append(currentExecMember.name());
	       sb.append('(');
	       Parameter[] params = currentExecMember.parameters();
	       for (int j=0; j < params.length; j++) {
		   sb.append(params[j].type().typeName());
		   sb.append(params[j].type().dimension());
		   sb.append(' ');
		   sb.append(params[j].name());
		   if (j != params.length-1)
			sb.append(", ");
	       }
	       sb.append(')');
	       rootDoc.printWarning(sb.toString());

	       printAtomTag(level+1, "exception typename=\""+throwsTag.exceptionName()+"\"");
	    }
	 }
	 else if (tag instanceof ParamTag) {
	    ParamTag paramTag = (ParamTag)tag;
	    printAtomTag(level+1, "parameter name=\""+paramTag.parameterName()+"\"");
	 }
	 if (null != tag.text()) {
	    //printOpenTag(level+1, "text", false);
	    if (fixHTML) {
	       print(htmlWell.wellHtml(tag.text()));
	    }
	    else {
	       print("<![CDATA["+cdata(tag.text())+"]]>");
	    }
	    //printCloseTag(0 /* don't introduce additional whitespace */, "text");
	 }
	 else {
	    rootDoc.printWarning("Tag got null text: "+tag);
	 }

	 if (descend && ("@throws".equals(tag.name()) || "@param".equals(tag.name()))) {
	    if (tag.firstSentenceTags().length>0) {
	       printOpenTag(level+1, "firstSentenceTags", false);
	       outputTags(level+2, tag.firstSentenceTags(), false);
	       printCloseTag(0, "firstSentenceTags");
	    }
	    
	    if (tag.inlineTags().length>0) {
	       printOpenTag(level+1, "inlineTags", false);
	       outputTags(level+2, tag.firstSentenceTags(), false);
	       printCloseTag(0, "inlineTags");
	    }
	 }

	 if (!"Text".equals(tag.name())) {
	    printCloseTag(0, "tag", false);
	 }
      }

      if (fixHTML) {
	 String terminateText = htmlWell.terminateText();
	 if (null != terminateText && terminateText.length() > 0) {
	    //printOpenTag(level, "tag kind=\"text\" name=\"text\"");
	    //printOpenTag(level+1, "text", false);
	    print(terminateText);
	    //printCloseTag(level+1, "text");
	    //printCloseTag(level, "tag");
	 }
      }
      
   }

   /**
    *  Inofficial entry point. We got a DebugDoclet instance here.
    */
   protected boolean instanceStart(RootDoc rootDoc) {
      
      this.rootDoc = rootDoc;

      try {

	 // Process command line options passed through to this doclet

	 for (int i=0, ilim=rootDoc.options().length; i<ilim; ++i) {
	    String[] option = rootDoc.options()[i];
	    String optionTag = option[0];

	    if ("-d".equals(optionTag)) {
	       targetDirectory = option[1];
	    }
	    else if ("-fixhtml".equals(optionTag)) {
	       fixHTML = true;
	    }
	    else if ("-compress".equals(optionTag)) {
	       compress = true;
	    }
	    else if ("-nohtmlwarn".equals(optionTag)) {
	       noHTMLWarn = true;
	    }
	    else if ("-noemailwarn".equals(optionTag)) {
	       noEmailWarn = true;
	    }
	    else if ("-indentstep".equals(optionTag)) {
	       indentStep = Integer.parseInt(option[1]);
	    }
	 }

	 // Assign output stream

	 setTargetFile("index.xml");

	 // Output XML document header

	 println(0, "<?xml version=\"1.0\"?>");
	 println("<!DOCTYPE gjdoc PUBLIC \"-//GNU//DTD Gjdoc XML V0.1.1//EN\" \"http://www.gnu.org/software/cp-tools/dtd/gjdoc.dtd\">");
	 println();
	 printOpenTag(0, "rootdoc xmlns=\"http://www.w3.org/TR/REC-html40\" xmlns:gjdoc=\"http://www.gnu.org/software/cp-tools/gjdocxml\"");

	 // Output summary of all classes specified on command line

	 println();
	 println(1, "<!-- Classes specified by user on command line -->");
	 ClassDoc[] specifiedClasses = rootDoc.specifiedClasses();
	 for (int i=0, ilim=specifiedClasses.length; i<ilim; ++i) {
	    ClassDoc sc = specifiedClasses[i];
	    printAtomTag(1, "specifiedclass fqname=\""+sc.qualifiedName()+"\" name=\""+sc.name()+"\"");
	 }

	 // Output summary of all packages specified on command line

	 println();
	 println(1, "<!-- Packages specified by user on command line -->");
	 PackageDoc[] specifiedPackages = rootDoc.specifiedPackages();
	 for (int i=0, ilim=specifiedPackages.length; i<ilim; ++i) {
	    PackageDoc sp = specifiedPackages[i];
	    printAtomTag(1, "specifiedpackage name=\""+sp.name()+"\"");
	 }
	 
	 // Output information on all packages for which documentation
	 // has been made available via the Doclet API

	 println();
	 println(1, "<!-- Documentation for all packages -->");
	 PackageDoc[] packages = rootDoc.specifiedPackages();
	 for (int i=0, ilim=packages.length; i<ilim; ++i) {
	    PackageDoc c = packages[i];
	    outputPackageDoc(c);
	 }

	 // Output brief summary on all classes for which documentation
	 // has been made available via the Doclet API.
	 //
	 // While this is redundant, it can speed up XSLT
	 // processing by orders of magnitude

	 println();
	 println(1, "<!-- Brief summary for all classes -->");
	 ClassDoc[] sumclasses = rootDoc.classes();
	 for (int i=0, ilim=sumclasses.length; i<ilim; ++i) {
	    ClassDoc c = sumclasses[i];
	    outputClassDocSummary(c);
	 }

	 // Output closing tag, finish output stream

	 println();
	 printCloseTag(0, "rootdoc");

	 closeTargetFile();


	 // Output information on all classes for which documentation
	 // has been made available via the Doclet API

	 println();
	 println(1, "<!-- Documentation for all classes -->");
	 ClassDoc[] classes = rootDoc.classes();
	 for (int i=0, ilim=classes.length; i<ilim; ++i) {
	    ClassDoc c = classes[i];

	    setTargetFile(c.qualifiedName().replace('/','.')+".xml");

	    println(0, "<?xml version=\"1.0\"?>");
	    println("<!DOCTYPE gjdoc PUBLIC \"-//GNU//DTD Gjdoc XML V0.1.1//EN\" \"http://www.gnu.org/software/cp-tools/dtd/gjdoc.dtd\">");

	    outputClassDoc(c);

	    closeTargetFile();
	 }
      
	 // Done

	 return true;
      }
      catch (Exception e) {

	 // Something went wrong. Report to stderr
	 // and pass error to Javadoc Reporter.

	 e.printStackTrace();
	 rootDoc.printError(e.toString());
	 return false;
      }
   }

   /**
    *  Prints a string to stdout and appends a newline.  Convenience
    *  method.  
    */
   protected void println(String str) {
      out.println(str);
   }

   /**
    *  Prints a string to stdout without appending a newline.
    *  Convenience method.  
    */
   protected void print(String str) {
      out.print(str);
   }

   /**
    *  In standard mode, prints an empty line to stdout.
    *  In thight mode, nothing happens.
    *  Convenience method.  
    */
   protected void println() {
      if (!compress) {
	 out.println();
      }
   }

   /**
    *  In standard mode, prints the given text indented to stdout and appends newline. 
    *  In tight mode, doesn't print indentation or newlines.
    */
   protected void print(int indentLevel, String msg) {
      if (compress) {
	 out.print(msg);
      }
      else {
	 StringBuffer indentation = new StringBuffer();
	 for (int i=0; i<indentLevel*indentStep; ++i) {
	    indentation.append(' ');
	 }
	 out.print(indentation+msg);
      }
   }
   
   /**
    *  In thight mode, prints a message at a given indentation level.
    *  In standard mode, appends a newline in addition.
    */
   protected void println(int indentLevel, String msg) {
      print(indentLevel, msg);
      if (!compress) out.println();
   }

   /**
    *  Prints an atom tag at the given indentation level.
    */
   protected void printAtomTag(int level, String tag) {
      println(level, "<"+tagPrefix+":"+replaceCharsInTag(tag)+"/>");
   }

   /**
    *  Prints an open tag at the given indentation level.
    */
   protected void printOpenTag(int level, String tag) {
      printOpenTag(level, replaceCharsInTag(tag), true);
   }

   /**
    *  Prints an open tag at the given indentation level and
    *  conditionally appends a newline (if not in tight mode).
    */
   protected void printOpenTag(int level, String tag, boolean appendNewline) {
      if (appendNewline && !compress) {
	 println(level, "<"+tagPrefix+":"+replaceCharsInTag(tag)+">");
      }
      else {
	 print(level, "<"+tagPrefix+":"+replaceCharsInTag(tag)+">");
      }
   }

   /**
    *  Prints a close tag at the given indentation level.
    */
   protected void printCloseTag(int level, String tag) {
      printCloseTag(level, tag, true);
   }

   /**
    *  Prints a close tag at the given indentation level and
    *  conditionally appends a newline (if not in tight mode).
    */
   protected void printCloseTag(int level, String tag, boolean appendNewline) {
      if (appendNewline && !compress) {
	 println(level, "</"+tagPrefix+":"+replaceCharsInTag(tag)+">");
      }
      else {
	 print(level, "</"+tagPrefix+":"+replaceCharsInTag(tag)+">");
      }
   }

   public static int optionLength(String option) {
      if ("-d".equals(option)) return 2;
      else if ("-fixhtml".equals(option)) return 1;
      else if ("-compress".equals(option)) return 1;
      else if ("-nohtmlwarn".equals(option)) return 1;
      else if ("-noemailwarn".equals(option)) return 1;
      else if ("-indentstep".equals(option)) return 2;
      else return -1;
   }

   public static boolean validOptions(String[][] options) {
      return true;
   }


   /**
    *  Workaround for non well-formed comments: fix tag contents
    *  by replacing <code>&lt;</code> with <code>&amp;lt;</code>,
    *  <code>&gt;</code> with <code>&amp;gt;</code> and
    *  <code>&amp;</code> with <code>&amp;amp;</code>.
    *
    *  @param tagContent  String to process
    *
    *  @return given String with all special characters replaced by 
    *          HTML entities.
    */
   private static String replaceCharsInTag(String tagContent) {
      return 
	 replaceString(
	    replaceString(
	       replaceString(
		  tagContent, 
		  "<", "&lt;"
		  ), 
	       ">", "&gt;"
	       ),
	    "&", "&amp;"
	    );
   }

   /**
    *  Replaces all occurences of string <code>needle</code> within string
    *  <code>haystack</code> by string <code>replacement</code>.
    *
    *  @param haystack    The string to search and replace in.
    *  @param needle      The string which is searched for.
    *  @param replacement The string by which every occurence of <code>needle</code> is replaced.
    */
   private static String replaceString(String haystack, String needle, String replacement) {
      int ndx = haystack.indexOf(needle);
      if (ndx<0)
	 return haystack;
      else
	 return haystack.substring(0, ndx) + replacement 
	    + replaceString(haystack.substring(ndx+needle.length()), needle, replacement);
   }

   protected void setTargetFile(String filename) throws IOException {

      OutputStream fileOut = new FileOutputStream(new File(new File(targetDirectory), filename));
      out = new PrintWriter(new BufferedWriter(new OutputStreamWriter(fileOut, "UTF8")));;
   }

   protected void closeTargetFile() {

      out.flush();
      out.close();
   }

   private String cdata(String str) {

      if (null==str) {
	 return str;
      } // end of if ((null==str)
	  

      StringBuffer rc = new StringBuffer();
      for (int i=0; i<str.length(); ++i) {
	 char c = str.charAt(i);
	 if (c==0x09 || c==0x0a || c==0x0d || (c>=0x20 && c<=0xd7ff) || (c>=0xe000 && c<=0xfffd) || (c>=0x10000 && c<=0x10ffff)) {
	    rc.append(c);
	 }
	 else {
	    rootDoc.printWarning("Invalid Unicode character 0x"+Integer.toString(c, 16)+" in javadoc markup has been stripped.");
	 } // end of else
	 
      }
      return rc.toString();
   }
}
