/* Listens on the EMWIN multicast address/port and reconstructs the
incoming file(s) or outputs the EMWIN QBT block to stdout.

Copyright (C) 1999 Antonio Querubin

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

--

Compiles/runs on:  Solaris, BSDI, IRIX, Linux, OS X, OS/2, Windows
95/98/NT.

The GCC compiler is required to build this on UNIX and OS/2.  To build
Microsoft Windows binaries requires either GCC-Mingw32 or LCC-Win32.

To compile for most UNIXs:  gcc -O2 emwinmcr.c -o emwinmcr

To compile for SunOS/Solaris:  gcc -O2 emwinmcr.c -lsocket -lnsl -o emwinmcr

To compile for Mac OS X:  cc -O2 emwinmcr.c -o emwinmcr

To compile for OS/2 using emx-gcc:  gcc -O2 emwinmcr.c -lsocket
Depending on how you link the binary, it may require the EMX run-time DLL
for distribution.  It can be obtained from:
ftp://ftp.leo.org/pub/comp/os/os2/leo/gnu/emx+gcc/emxrt.zip

To compile for Windows using gcc-mingw32: 
gcc -O2 emwinmcr.c -lwsock32 -o emwinmcr.exe

To compile for Windows using lcc-win32, the wsock32.lib file must be added
as an additional library in the linker configuration.

Please send patches and/or bug reports to:  tony@lava.net

Contributions/bug fixes by:
  Julian Cowley
  Maitland Bottoms (AA4HS)

Revision history:
 6 Sep 1999, v0.1:  First alpha release for Solaris/BSDI. 
18 Sep 1999, v0.2:  Added Windows 95/98/NT support.
                    Added file reconstruction.
19 Sep 1999, v0.3:  Added Linux support (from Julian Cowley).
                    Removed dependency on drive C: (Windows version).
21 Sep 1999, v0.4:  Added better error checking for bogus network packets.
                    Added command line options.
23 Sep 1999, v0.5:  Fixed up socket error reporting.
                    Added more error checking of the incoming packet.
30 Sep 1999, v0.6:  More command line options and code cleanup.
                    Added Y2K rollover fix.
                    Restored signal handlers for Windows.
                    Added SIGHUP handler for UNIX.
 1 Oct 1999, v0.7:  Added unix daemon mode and syslog support (from AA4HS).
10 Oct 1999, v0.8:  Added OS/2 support and a better move completed file
                    routine.
11 Oct 1999, v0.9:  Now truncates trailing nulls in text files.
13 Oct 1999, v0.10: Changed the move routine to a copy and delete for
                    Windows and OS/2 so that it works across different
                    drives.
14 Oct 1999, v0.11: Fixed a bug in the stderr redirection for OS/2.  OS/2
                    wants to see NUL instead of NUL:
15 Oct 1999, v0.12: Tests for BSD-type SO_REUSEPORT instead of specific OS
                    at compile time.
19 Oct 1999, v0.13: Added call to unzip for ZIS files (output filename is
                    in lower-case).
                    Set umask to plug a security hole.
                    Added directory name error checking.
20 Oct 1999, v0.14: Fixed getcwd/chdir problem in OS/2.  
                    Added option to specify unzip path.
27 Nov 1999, v0.15: Fixed Windows Ctrl-C handler.
31 Dec 1999, v0.16: Converted to new 4-digit year format in the header.
10 Jan 2000, v0.17: Increased datestamp length and added GMT option.
11 Jan 2000, v0.18: Cosmetic update to the patch/bug report address.

Known bugs:

Multiple concurrent instances of the OS/2 version can't bind to the
address/port.  However, most users will only run a single copy anyway.

--

Uniquely defined macros for various compilers and specific OS targets:
bsdi            gcc / BSDI
linux           gcc / Linux
__APPLE__       cc / OS X
__EMX__         emx-gcc / OS/2
WIN32           gcc-mingw32 or lcc-win32 / Microsoft Windows 95/98/NT

*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <ctype.h>
#include <sys/stat.h>
#include <errno.h>

#ifdef WIN32
#include <direct.h>
#include <winsock.h>
#else
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/param.h>
#ifndef __EMX__
#include <syslog.h>
#endif
#endif

#define VERSION "0.18"

/* Normally found in sys/param.h but LCC-Win32 doesn't have it. */
#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

#define TRUE 1
#define FALSE 0

/* IANA assigned address for emwin.mcast.net. */
#define EMWINMCADDR "224.0.1.143"
#define EMWINPORT 1024

#if defined(WIN32) || defined(__EMX__)
/* This assumes a WeatherNode directory structure */
#define BASEDIR "\\weather\\wxdata\\"
#define TRACKDIR BASEDIR "rxtrack"
#define TMPDIR BASEDIR "rxtmp\\tmp"
#define OUTDIR BASEDIR "newfiles"
#define NULLDEVICE "NUL"
#define COPYCMD "copy %s %s"
#define PATHSEPARATOR "\\"
#else
#ifdef linux
#define TMPDIR "/var/tmp"
#else
#define TMPDIR P_tmpdir
#endif
#define OUTDIR "."
#define BASEDIR "/weather/wxdata/"
#define TRACKDIR BASEDIR "rxtrack"
#define NULLDEVICE "/dev/null"
#define COPYCMD "/bin/cp -f %s %s"
#define PATHSEPARATOR "/"
#endif

#ifndef UNZIPPATH
#define UNZIPPATH "unzip"
#endif
#define UNZIPPARAMS " -qoj %s -d %s"

#ifndef BEEPMODE
#define BEEPMODE 0
#endif

#ifndef OUTMODE
#define OUTMODE 2
#endif

#ifndef DEBUG
#define DEBUG 0
#endif

#define MAXBLOCKS 500

/* These are defined externally since they might be referenced by
   other functions. */
unsigned long pktcount =0 ;
unsigned long badpktcount = 0;
unsigned char *outdir = OUTDIR;
unsigned char *basedir = BASEDIR;
unsigned char *trackdir = TRACKDIR;
unsigned char *tmpdir = TMPDIR;
unsigned char *unzippath = UNZIPPATH;
unsigned char *pathseparator = PATHSEPARATOR;
int outmode = OUTMODE;
int beepmode = BEEPMODE;
int debug = DEBUG;
int recompat=0;

struct QBT {	/* http://iwin.nws.noaa.gov/emwin/winpro.htm */
  unsigned char preamble[6];
  unsigned char header[80];
  unsigned char data[1024];
  unsigned char trailer[6];
} qbtpacket;

struct PKTATTR {
  unsigned char filename[13];	/* Null-terminated. */
  unsigned int blocknum, totalblocks;
  unsigned long checksum;
  unsigned char datestamp[24];	/* Null-terminated. */
  unsigned char *data;
} pktattr;

#ifdef __EMX__
/* Under OS/2 the standard chdir() and getcwd() behave differently
   so we redefine these to their UNIX equivalents. */
#define chdir _chdir2
#define getcwd _getcwd2
#endif

#ifdef WIN32
/* perror() is useless with Windows sockets. */
#define psocketerror(S) \
  fprintf(stderr,"%s:  WSA error #%d\n", S, WSAGetLastError());
#else
#define psocketerror(S) perror(S);
#endif

/* Setup the multicast listener. */
int openemwinsocketin(void) {
  struct sockaddr_in stLocal;
  int s;
  struct ip_mreq stMreq;
  int optint;
#ifdef WIN32
  int nRet;
  WSADATA stWSAData;

  /* init WinSock */
  nRet = WSAStartup(0x0202, &stWSAData);
  if (nRet) {
    fprintf(stderr,"WSAStartup failed: %d\n", nRet);
    exit(EXIT_FAILURE);
  }
#endif

  /* get a datagram socket */
  s = socket(AF_INET, SOCK_DGRAM, 0);
  if (s < 0) {
    psocketerror("socket failed");
    exit(EXIT_FAILURE);
  }

  /* avoid EADDRINUSE error on bind() */
  optint = TRUE;
#ifdef SO_REUSEPORT
  /* For BSDI and Mac OS X systems.  Other BSD-derived systems may require
     SO_REUSEPORT also. */  
  if (setsockopt(s, SOL_SOCKET, SO_REUSEPORT,
                 (void *) &optint, sizeof(optint)) < 0) {
    psocketerror("setsockopt REUSEPORT failed");
#else
  if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR,
                 (void *) &optint, sizeof(optint)) < 0) {
    psocketerror("setsockopt REUSEADDR failed");
#endif
    exit(EXIT_FAILURE);
  }

  /* name the socket */
  stLocal.sin_family = AF_INET;
  stLocal.sin_addr.s_addr = htonl(INADDR_ANY);
  stLocal.sin_port = htons(EMWINPORT);
  if (bind(s, (struct sockaddr *) &stLocal, sizeof(stLocal)) < 0) {
    psocketerror("bind failed");
    exit(EXIT_FAILURE);
  }

  /* join the multicast group. */
  stMreq.imr_multiaddr.s_addr = inet_addr(EMWINMCADDR);
  stMreq.imr_interface.s_addr = INADDR_ANY;
  if (setsockopt(s, IPPROTO_IP, IP_ADD_MEMBERSHIP,
                 (void *) &stMreq, sizeof(stMreq)) < 0) {
    psocketerror("setsockopt ADD MEMBERSHIP failed");
    exit(EXIT_FAILURE);
  }

  return(s);
} /* openemwinsocketin() */

/* Prints statistical data. */
void dumpstats(void) {
  fprintf(stderr,"\nPackets received = %lu, # of bad packets = %lu ",
          pktcount, badpktcount);
  if (pktcount) fprintf(stderr," (%f%%)", (float) badpktcount/pktcount*100);
  fprintf(stderr,"\n");
} /* dumpstats() */

/* Ctrl-C/SIGINT/SIGTERM handler. */
void timetodie(int sig) {
  dumpstats();
#ifdef WIN32
  /* Tell WinSock we're leaving */
  WSACleanup();
#endif
  /* Restore default handler and run it. */
  signal(sig,SIG_DFL);
  raise(sig);
  /* Shouldn't get this far but just in case... */
  exit(EXIT_SUCCESS);
} /* timetodie() */

/* SIGHUP handler. */
void resetstats(int sig) {
  dumpstats();
  pktcount = 0;
  badpktcount = 0;
  signal(sig,SIG_IGN); /* We want to continue. */
  signal(sig,resetstats);
} /* resetstats() */

/* Parses and checks an EMWIN QBT block.
   Return values: -1 if any check fails, 0 otherwise.
   The attr structure is filled with the parsed block values if all
   checks are successful.  Otherwise the content of the structure is
   undefined. */
int checkblock(const struct QBT *blockp, struct PKTATTR *attr) {
  unsigned char tempbuf[sizeof(blockp->header) + 1];
  int i;
  unsigned char filename[13];
  unsigned long ourchecksum;
  time_t filetime;
  struct tm timest;
  unsigned char apm[3];

  /* The header isn't null-terminated so we have to add a NULL before
     parsing with sscanf. */
  memcpy(tempbuf,blockp->header,sizeof(blockp->header));
  tempbuf[sizeof(tempbuf)-1] = '\0';
  if (sscanf(tempbuf,"/PF%12c/PN %u /PT %u /CS %lu /FD%23c",
             filename, &(attr->blocknum), &(attr->totalblocks),
             &(attr->checksum), attr->datestamp) != 5) {
    fprintf(stderr,"Bad header.\n");
    return (-1);
  }
  filename[sizeof(filename)-1] = '\0';
  attr->datestamp[sizeof(attr->datestamp)-1] = '\0';

  /* Do a sanity check of the header. */

  /* Strip any trailing white-space from the filename. */
  if (sscanf(filename,"%s",attr->filename) != 1) {
    fprintf(stderr,"Missing filename.\n");
    return (-1);
  }
  for (i=0; i<strlen(attr->filename); i++) {
    if (!(isalnum(attr->filename[i]) || attr->filename[i] == '.')) {
      fprintf(stderr,"Invalid filename.\n");
      return (-1);
    }
  }

  /* Range check the block numbers. */
  if (attr->blocknum < 1 || attr->blocknum > attr->totalblocks ||
      attr->totalblocks > MAXBLOCKS) {
    fprintf(stderr,"Bad block number(s).\n");
    return (-1);
  }

  /* Validate the datestamp by trying to convert it to a time value. */
  memset(&timest,'\0',sizeof(timest));
  if (sscanf(attr->datestamp,"%u/%u/%u %u:%u:%u %3s",
             &timest.tm_mon,&timest.tm_mday,&timest.tm_year,
             &timest.tm_hour,&timest.tm_min,&timest.tm_sec,apm) != 7) {
    fprintf(stderr,"Bad date string.\n");
    return (-1);
  }
  if (strcmp(apm,"AM") && strcmp(apm,"PM") && strcmp(apm,"GMT")) {
    fprintf(stderr,"Bad date/timezone string.\n");
    return (-1);
  }
  /* Setup the time structure. */
  timest.tm_mon--;
  /* Adjust the hour for AM/PM. */
  if (timest.tm_hour == 12 && apm[0] == 'A') timest.tm_hour = 0;
  else if (timest.tm_hour < 12 && apm[0] == 'P') timest.tm_hour += 12;
  timest.tm_year -= 1900;
  if ((filetime=mktime(&timest)) == -1) {
    fprintf(stderr,"Bad date/time.\n");
    return (-1);
  }

  fprintf(stderr,"%s, # %u of %u, %s",
          attr->filename, attr->blocknum, attr->totalblocks,
          ctime(&filetime));

  /* Checksum the data portion. */
  for (i=0, ourchecksum=0;
       i<sizeof(blockp->data);
       ourchecksum += (unsigned int) blockp->data[i++]);
  if (ourchecksum != attr->checksum) {
    fprintf(stderr,"Checksum = %lu, should be %lu.\n",
            ourchecksum, attr->checksum);
    return (-1);
  }

  attr->data = (unsigned char *)blockp->data;
  return (0);
} /* checkblock() */

/* Reads an EMWIN QBT packet from the multicast socket.
   Returns 0 if successful. */
int getmcpacket(const int s, const struct QBT *qbtpacket) {
  int bytes;

  if ((bytes=recv(s, (void *) qbtpacket, sizeof(*qbtpacket), 0)) <= 0) {
#ifdef WIN32
    /* If the socket is closed then timetodie() was probably called during the 
       blocking call to recv() so we force the exit anyway. */
    if (WSAGetLastError() == WSAENOTSOCK) exit(EXIT_SUCCESS);
#endif
    psocketerror("recv failed");
    return(-1);
  }
  if (bytes != sizeof(*qbtpacket)) {
    fprintf(stderr,"Unexpected number of bytes received = %d\n", bytes);
    return(-1);
  }

  return(0);
} /* getmcpacket() */

/* Save the EMWIN block.
   Return values:
     -1=failure
      0=block saved
      1=file completed by this block
   savedfilepath is set to point at the full path to the saved output
   file. */
int saveblock(const struct PKTATTR *pktp, unsigned char *savedfilepath) {
  FILE *trackfp, *datafp = NULL;
  int i, n, newfile, datasize;
  unsigned char trackrec[sizeof(pktp->datestamp)+MAXBLOCKS],
                trackpath[MAXPATHLEN], datapath[MAXPATHLEN];

  /* Construct the full file paths. */
  sprintf(datapath, "%s" PATHSEPARATOR "%s", tmpdir, pktp->filename);
#if defined(WIN32) || defined(__EMX__)
  sprintf(trackpath, "%s" PATHSEPARATOR "%s", TRACKDIR, pktp->filename);
#else
  if (recompat) {
    sprintf(trackpath, "%s" PATHSEPARATOR "%s", trackdir, pktp->filename);
  } else {
  /* In UNIX use the same tmpdir to store the tracking file but
     add a '.track' suffix to the filename. */
  sprintf(trackpath, "%s.track", datapath);
  }
#endif

  newfile = FALSE;
  /* Index value into the track record for this block. */
  n = sizeof(pktp->datestamp) + pktp->blocknum - 1;

  /* Open the block tracking file first. */
  if ((trackfp=fopen(trackpath,"r+")) == NULL) newfile = TRUE;
  else {
    fgets(trackrec,sizeof(trackrec),trackfp);
    /* Drop the line terminator if it exists. */
    if (trackrec[strlen(trackrec)-1] == '\n')
      trackrec[strlen(trackrec)-1] = '\0';
    /* If the datestamp doesn't match then create a new file. */
    if (strncmp(pktp->datestamp,trackrec,sizeof(pktp->datestamp)-1)) {
      newfile = TRUE;
      fclose(trackfp);
    }
    /* Make sure the block numbers and track record length make sense. */
    else if (n >= strlen(trackrec) ||
             sizeof(pktp->datestamp)+pktp->totalblocks !=
                    strlen(trackrec)) {
      fclose(trackfp);
      fprintf(stderr,"Track record length error.\n");
      return(-1);
    }
    /* Do nothing if we already have the block. */
    else if (trackrec[n] == 'Y') {
      fclose(trackfp);
      return(0);
    }
    /* If the data file can't be opened, assume it must be created. */
    else if ((datafp=fopen(datapath,"r+b")) == NULL) {
      newfile = TRUE;
      fclose(trackfp);
    }
  }

  if (newfile) {
    if ((datafp=fopen(datapath,"w+b")) == NULL) {
      perror("saveblock, fopen data file");
      return(-1);
    }
    /* Construct a new track record.  It consists of the file
       datestamp, a comma, and a series of . or Y characters where Y
       indicates the block has already been received. */
    memcpy(trackrec,pktp->datestamp,sizeof(pktp->datestamp));
    trackrec[sizeof(pktp->datestamp)-1] = ',';
    memset(trackrec+sizeof(pktp->datestamp), '.', pktp->totalblocks);
    trackrec[sizeof(pktp->datestamp)+pktp->totalblocks] = '\0';
    if ((trackfp=fopen(trackpath,"w+")) == NULL) {
      perror("saveblock, fopen track file");
      return(-1);
    }
    fputs(trackrec,trackfp);
  }

  /* Did we already save this block? */
  if (trackrec[n] != 'Y') {
    /* Write the block. */
    if (fseek(datafp, sizeof(qbtpacket.data)*(pktp->blocknum - 1),
        SEEK_SET)) {
      perror("saveblock, fseek");
      return(-1);
    }

    if ((pktp->blocknum == pktp->totalblocks) && 
        ((strstr(pktp->filename,".TXT") != NULL) || 
         (strstr(pktp->filename,".HTM") != NULL))) {
      /* If this is the last block of a text file, truncate the trailing
         null characters by changing the number of bytes to write. */
      for(datasize=sizeof(qbtpacket.data); datasize > 0; datasize--)
        if (pktp->data[datasize-1] != '\0') break;
    }
    else datasize=sizeof(qbtpacket.data);
    if (fwrite(pktp->data, datasize, 1, datafp) != 1) {
      perror("saveblock, fwrite");
      return(-1);
    }

    /* Update the track record. */
    trackrec[n] = 'Y';
    rewind(trackfp);
    fputs(trackrec,trackfp);
  }
  fclose(datafp);
  fclose(trackfp);

  /* Check the track record to see if we have all the parts. */
  for (i=sizeof(pktp->datestamp);
       i<sizeof(pktp->datestamp) + pktp->totalblocks; i++)
  if (trackrec[i] != 'Y') return(0);

  strcpy(savedfilepath,datapath);
  return(1);
} /* saveblock() */

void showusage(void) {
  fprintf(stderr,
    "Usage:  emwinmcr [-o output-directory]"
#if !defined(WIN32) && !defined(__EMX__)
    " [-t temp-directory]"
#endif
    "\n"
    "                 [-m output-mode] [-b] [-d] [-h]\n"
    "Options:\n"
    "-o output-directory  Directory to place completed files\n"
    "                     (default = %s).\n"
#if !defined(WIN32) && !defined(__EMX__)
    "-w                   Use RealEmwin compatable directory structure\n"
    "-t temp-directory    Directory to place temporary files\n"
    "                     (default = %s).\n"
#endif
    "-m output-mode       0=receive only.\n"
    "                     1=send packets to stdout only.\n"
    "                     2=save completed files (default).\n"
#ifndef WIN32
    "                     3=same as 2 but runs in daemon mode.\n"
#endif
    "-b                   Beep on bad packets.\n"
    "-d                   Toggles diagnostic mode (default = %s).\n"
    "-u                   Path to unzip command (default = %s).\n"
    "-h                   Display options.\n",
#if !defined(WIN32) && !defined(__EMX__)
    outdir, tmpdir, debug ? "on":"off", unzippath);
#else
    outdir, debug ? "on":"off", unzippath);
#endif
} /* showusage() */

int main(const int argc, const char *argv[]) {
  int emwinsocket, i, savestatus, pktisbad = FALSE, unzipstatus, copystatus;
  time_t t;
  unsigned char timebuf[26],
                tempbuf[sizeof(COPYCMD) + 2*MAXPATHLEN],
                completedfile[MAXPATHLEN];
  unsigned char *zisextp;
#ifndef WIN32
  pid_t	pid;
#endif

  /* Parse the command line arguments. */
  for (i=1; i < argc; i++) {
    if (argv[i][0] == '-') switch (argv[i][1]) {
      case 'o':  /* Output directory for completed files. */
        if (i+1 < argc) outdir = (unsigned char *)argv[i+1];
        else {
          fprintf(stderr,"Missing option value:  output directory.\n");
          exit(EXIT_FAILURE);
        }
        i++;
        break;
#ifndef WIN32
      case 'w':  /* compat enable. */
        recompat = 1;
        break;
      case 't':  /* Temp directory for incomplete files and block
                 tracking files. */
        if (i+1 < argc) tmpdir = (unsigned char *)argv[i+1];
        else {
          fprintf(stderr,"Missing option value:  tmp directory.\n");
          exit(EXIT_FAILURE);
        }
        i++;
        break;
#endif
      case 'm':  /* Output mode. */
        if (i+1 < argc) {
          if (sscanf(argv[i+1],"%u",&outmode) != 1) {
            fprintf(stderr,"Invalid mode.\n");
            exit(EXIT_FAILURE);
          }
#ifdef WIN32
          if (outmode > 2) {
#else
          if (outmode > 3) {
#endif
            fprintf(stderr,"Invalid mode.\n");
            exit(EXIT_FAILURE);
          }
        }
        else {
          fprintf(stderr,"Missing mode.\n");
          exit(EXIT_FAILURE);
        }
        i++;
        break;
      case 'b':  /* Beep enable. */
        beepmode = TRUE;
        break;
      case 'd':  /* Toggle the diagnostic output mode from the compile
                    default. */
        debug ^= 1;
        break;
      case 'u':  /* Path to unzip utility for ZIS files. */
        if (i+1 < argc) unzippath = (unsigned char *)argv[i+1];
        else {
          fprintf(stderr,"Missing option value:  unzip path.\n");
          exit(EXIT_FAILURE);
        }
        i++;
        break;
      case 'h':
  	showusage();
        exit(EXIT_SUCCESS);
        break;
      default:
        fprintf(stderr,"Unknown option %s\n", argv[i]);
        showusage();
        exit(EXIT_FAILURE);
    } /* end of switch (argv[i][1]) */
    else {
      showusage();
      exit(EXIT_FAILURE);
    }
  }

  if (recompat) { /* This may be of interest when using samba */
     basedir = strcat(outdir,"wxdata");
     trackdir = strcat(basedir,"rxtrack");
     tmpdir = strcat(basedir,"rxtmp/tmp");
     outdir = strcat(basedir,"newfiles");
  }
  /* Preliminary error checks before we redirect diagnostic output to 
     /dev/null. */
  if (outmode > 1) {
#ifndef __APPLE__
/* getcwd() coredumps on Apple OS X so we'll bypass this check for now. */
    /* Make sure the directories exist and get their full pathnames. */
    if (chdir(outdir) < 0) {
      perror("Can't access outdir");
      exit(EXIT_FAILURE);
    }
    outdir = getcwd(NULL,MAXPATHLEN);
    if (chdir(tmpdir) < 0) {
      perror("Can't access tmpdir");
      exit(EXIT_FAILURE);
    }
    tmpdir = getcwd(NULL,MAXPATHLEN);
#endif
#ifdef __EMX__
/* getcwd() on the EMX compiler insists on converting \ to / 
   so we convert them back. */
    {unsigned char *p;
     while((p=strchr(tmpdir,'/')) != NULL) *p = '\\';
     while((p=strchr(outdir,'/')) != NULL) *p = '\\';
    }
#endif
#if !defined(WIN32) && defined(__EMX__)
    /* Set the file creation mask so that downloaded files aren't
       executable nor writeable. */
    umask(umask(0) | 0133);
#endif
  }

  /* Now we can redirect diagnostic output. */
  if (!debug) {
    if (freopen(NULLDEVICE,"a",stderr) == NULL) {
      perror("Can't redirect stderr");
      exit(EXIT_FAILURE);
    }
  }

  fprintf(stderr,"emwinmcr, version " VERSION
                 ", compiled " __DATE__ " " __TIME__ "\n");

  emwinsocket = openemwinsocketin();

  /* Setup the signal handlers.  Some of these don't exist or may have
     problems under Windows, especially NT. */
#ifdef SIGINT
  signal(SIGINT,timetodie);
#endif
#ifdef SIGTERM
  signal(SIGTERM,timetodie);
#endif
#ifdef SIGHUP
  signal(SIGHUP,resetstats);
#endif

  switch (outmode) {
  case 1:
    fprintf(stderr,"Sending packets to stdout.\n");
    break;
  case 2:
    fprintf(stderr,"Completed files will be moved to %s\n"
            "Filename will be written to stdout.\n",outdir);
    break;
#ifndef WIN32
  case 3: /* Daemon invokation */

    if((pid=fork()) < 0) {
      perror("Can't fork daemon process");
      exit(EXIT_FAILURE);
    }
    else if (pid != 0) exit(EXIT_SUCCESS); /* Parent goes bye-bye */

    /* Child continues */
#ifndef __EMX__
    /* become session leader */
    if (setsid() < 0) {
      perror("main, setsid");
      exit(EXIT_FAILURE);
    }
    openlog("emwinmcr",LOG_PID,LOG_USER);
    syslog(LOG_INFO,"EMWIN multicast receiver daemon starting");
#endif
    break;
#endif
  default: ;
  }

  fprintf(stderr,"Listening for multicast packets.  Press Ctrl-C to stop.\n");

  while(TRUE) {

    if (pktisbad) {
      badpktcount++;
      if (beepmode) fprintf(stderr,"\a");
      pktisbad = FALSE;
    }

    if (getmcpacket(emwinsocket, &qbtpacket)) {
      pktisbad = TRUE;
      continue;
    }
    pktcount++;

    if (outmode < 3) { /* Timestamp the diagnostic output. */
      t = time(NULL);
      strcpy(timebuf,ctime(&t));
      /* Drop the \n at the end of the line. */
      timebuf[strlen(ctime(&t))-1] = '\0';
      fprintf(stderr,"%s:  ",timebuf);
    }

    if(checkblock(&qbtpacket,&pktattr) == -1) pktisbad = TRUE;
    else switch (outmode) {
      case 1:
        fwrite(&qbtpacket, sizeof(qbtpacket), 1, stdout);
        break;
      case 2:
      case 3:
        if ((savestatus = saveblock(&pktattr,completedfile)) == 1) {

          unzipstatus = -1;
          copystatus = -1;
 
          zisextp = strstr(pktattr.filename,".ZIS");
          if (zisextp != NULL) {
            /* Special handling for compressed text files.  If the unzip
               operation fails, ignore the error and continue on as if it
               were a normal file. */
            sprintf(tempbuf, "%s" UNZIPPARAMS, 
                    unzippath, completedfile, outdir);
            unzipstatus = system(tempbuf);             
          }
          
          if (zisextp == NULL || unzipstatus) {
            /* Copy the new file to the destination directory. */
            sprintf(tempbuf, COPYCMD, completedfile, outdir);
            if ((copystatus=system(tempbuf))) perror("Copy failed");
          }

          /* Delete the temporary file and output the file name if unzip
             or copy succeeded. */
          if (unzipstatus == 0 || copystatus == 0) {
            if (remove(completedfile)) perror("Can't remove tempfile");
            if (outmode < 3) {
              if (zisextp != NULL && unzipstatus == 0) {
                /* For some reason, the uncompressed filename is in
                   lower-case. */
                for (i=0; pktattr.filename[i] != '\0'; i++) 
                  pktattr.filename[i] = tolower(pktattr.filename[i]);
                strcpy(zisextp,".txt");
              } 
              printf("%s\n",pktattr.filename);
              fflush(stdout);
            }
          }

        }
        else if (savestatus == -1) pktisbad = TRUE;
        break;
      default: ; 
    } /* end of switch (outmode) */

    fflush(stderr);	/* For http streaming of the diagnostic output. */
  }
} /* main() */
