# -*- coding: utf-8 -*-

# Copyright (c) 2004 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a package widget.
"""

from qt import *
from qtcanvas import *

from UMLWidget import UMLWidget

class PackageModel:
    """
    Class implementing the package model.
    """
    def __init__(self, name, moduleslist=[]):
        """
        Constructor
        
        @param name package name (string)
        @param classlist list of class names (list of strings)
        """
        self.name = name
        self.moduleslist = moduleslist
        
    def addModule(self, modulename):
        """
        Method to add a module to the package model.
        
        @param modulename module name to be added (string)
        """
        self.moduleslist.append(classname)
        
    def getModules(self):
        """
        Method to retrieve the modules of the package.
        
        @return list of module names (list of strings)
        """
        return self.moduleslist[:]
        
    def getName(self):
        """
        Method to retrieve the package name.
        
        @return package name (string)
        """
        return self.name
        
class PackageWidget(UMLWidget):
    """
    Class implementing a module widget.
    """
    def __init__(self, canvas, model=None, x=0, y=0, rounded=0):
        """
        Constructor
        
        @param canvas canvas containing the class widget (QCanvas)
        @param model module model containing the module data (ModuleModel)
        @param x x-coordinate (integer)
        @param y y-coordinate (integer)
        @param rounded flag indicating a rounded corner (boolean)
        """
        UMLWidget.__init__(self, canvas, x, y, rounded)
        self.model = model
        
        if self.model:
            self.calculateSize()
        
        self.update()
        
    def calculateSize(self):
        """
        Method to calculate the size of the package widget.
        """
        if self.model is None:
            return
            
        modules = self.model.getModules()
        
        font = QFont(self.font)
        font.setBold(1)
        font.setItalic(1)
        fm = QFontMetrics(font)
        
        # calculate height
        fontHeight = fm.lineSpacing()
        lines = 1
        if modules:
            lines += len(modules)
        else:
            lines += 1
        height = lines * fontHeight
        
        # calculate width
        width = fm.width(self.model.getName())
        
        for module in modules:
            w = fm.width(module)
            width = max(w, width)
        
        width += self.margin * 2
        width += 4  # a little bit more for strange fonts
        height += self.margin * 4
        
        height += min(15, int(width/3))
        self.setSize(width, height)
    
    def setModel(self, model):
        """
        Method to set the package model.
        
        @param model package model containing the package data (PackageModel)
        """
        self.model = model
        self.calculateSize()
        self.update()
        
    def drawShape(self, painter):
        """
        Overriden method to draw the shape.
        
        @param painter painter the shape is drawn to (QPainter)
        """
        painter.setPen(Qt.black)
        offsetX = self.x()
        offsetY = self.y()
        w = self.width()
        latchW = int(w/3)
        latchH = min(15, latchW)
        h = self.height() - latchH + 1
        fm = QFontMetrics(self.font)
        fontHeight = fm.lineSpacing() + 1 # one for the baseline itself
        
        painter.drawRect(offsetX, offsetY, latchW, latchH)
        painter.drawRect(offsetX, offsetY + latchH - 1, w, h)
        
        y = self.margin + latchH - 1
        self.font.setBold(1)
        painter.setFont(self.font)
        painter.drawText(offsetX + self.margin,
                         offsetY + y,
                         w - self.margin * 2,
                         fontHeight,
                         Qt.AlignHCenter,
                         self.model.getName())
        y += fontHeight + self.margin
        self.font.setBold(0)
        painter.setFont(self.font)
        painter.drawLine(offsetX, offsetY + y,
                         offsetX + w - 1, offsetY + y)
        y += self.margin
        
        modules = self.model.getModules()
        if modules:
            for module in modules:
                painter.drawText(offsetX + self.margin,
                                 offsetY + y,
                                 w - self.margin * 2,
                                 fontHeight,
                                 Qt.AlignHCenter,
                                 module)
                y += fontHeight
        else:
            painter.drawText(offsetX + self.margin,
                             offsetY + y,
                             w - self.margin * 2,
                             fontHeight,
                             Qt.AlignHCenter,
                             "")
            y += fontHeight
        
        if self.isSelected():
            self.drawSelected(painter, offsetX, offsetY)
