//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: ConnectivityBuilder.hpp,v 1.6 2003/12/26 18:38:02 uid68082 Exp $

#ifndef CONNECTIVITY_BUILDER_HPP
#define CONNECTIVITY_BUILDER_HPP

#include <set>
#include <list>
#include <map>

#include <TinyVector.hpp>
#include <Connectivity.hpp>
#include <ReferenceCounting.hpp>

#include <Mesh.hpp>

template <unsigned meshFamily>
struct SurfaceLinker;

template <typename MeshType>
class ConnectivityBuilder
{
private:
  friend
  struct SurfaceLinker<MeshType::family>;

  typedef typename MeshType::ElementGeometry Elements;
  typedef typename Elements::BorderType BorderType;

  typedef TinyVector<BorderType::NumberOfVertices,size_t> VerticesList;
  typedef std::pair<const Elements*,size_t> ElementFace;

  struct compareTinyVectors
  {
    bool operator()(const VerticesList& t1,
		    const VerticesList& t2) const
    {
      for (size_t i=0; i<t1.size(); ++i) {
	if(t1[i]!=t2[i]) {
	  return (t1[i]<t2[i]);
	}
      }
      return false;
    }
  };

  typedef std::map<VerticesList,std::list<ElementFace>,
		   compareTinyVectors> FaceCorrespondance;

  TinyVector<Elements::NumberOfNeighbours,
	     TinyVector<BorderType::NumberOfVertices,size_t> > __face;

  void setFaces();

public:
  ConnectivityBuilder(const MeshType& m,
		      Connectivity<Elements>& connectivity,
		      ReferenceCounting<typename MeshType::BorderMeshType> surfaceMesh = 0)
  {
    this->setFaces();

    FaceCorrespondance faces;

    for (typename MeshType::const_iterator i(m); not(i.end()); ++i) {
      for (size_t l=0; l<Elements::NumberOfNeighbours; ++l) {
	std::set<size_t> verticesSet;
	for (size_t k=0; k<BorderType::NumberOfVertices; ++k) {
	  verticesSet.insert(m.vertexNumber((*i)(__face[l][k])));
	}
	size_t k=0; 
	VerticesList vertices;
	for (std::set<size_t>::iterator n=verticesSet.begin();
	     n !=  verticesSet.end(); ++n) {
	  vertices[k++]=*n;
	}
	faces[vertices].push_back(ElementFace(i.pointer(),l));
      }
    }

    ffout(3) << "- Storing connectivity\n";

    for (typename FaceCorrespondance::iterator i = faces.begin();
	 i != faces.end(); ++i) {
      typename FaceCorrespondance::mapped_type& faceList
	= (*i).second;

      switch (faceList.size()) {
      case 1: {
	typename FaceCorrespondance::mapped_type::iterator j
	  = (*i).second.begin();
	// Ensure that the face is 0
	connectivity(m.cellNumber(*(*j).first))[(*j).second] = 0;
	break;
      }
      case 2: {
	typename FaceCorrespondance::mapped_type::iterator j1 = (*i).second.begin();
	typename FaceCorrespondance::mapped_type::iterator j2 = j1;
	j2++;

	connectivity(m.cellNumber(*(*j1).first))[(*j1).second] = (*j2).first;
	connectivity(m.cellNumber(*(*j2).first))[(*j2).second] = (*j1).first;

	break;
      }
      default: {
	fferr(0) << "\noops: Something strange happened: incorrect mesh:\n";
	fferr(0) << "        One of the faces of your mesh is related to "
		 << faceList.size() << " cells (vertices: " << (*i).first << ")\n"; ;
#ifndef NDEBUG
	std::exit(1);
#endif // NDEBUG

      }
      }
    }
    SurfaceLinker<MeshType::family>(m, faces, surfaceMesh);
  }

  ~ConnectivityBuilder()
  {
    ;
  }
};

template<unsigned meshFamily>
struct SurfaceLinker
{
  template <typename MeshType>
  SurfaceLinker(const MeshType& m,
		const typename ConnectivityBuilder<MeshType>::FaceCorrespondance& faces,
		ReferenceCounting<typename MeshType::BorderMeshType>)
  {
    ;
  } 
};

template<>
struct SurfaceLinker<Mesh::volume>
{
  template <typename MeshType>
  SurfaceLinker(const MeshType& m,
		const typename ConnectivityBuilder<MeshType>::FaceCorrespondance& faces,
		ReferenceCounting<typename MeshType::BorderMeshType> surfaceMesh)
  {
    if (surfaceMesh != 0) {
      ffout(3) << "- Connecting surface mesh\n";
      for (typename MeshType::BorderMeshType::iterator i(*surfaceMesh);
	   not(i.end()); ++i) {
	std::set<size_t> verticesSet;
	for (size_t k=0; k<ConnectivityBuilder<MeshType>::BorderType::NumberOfVertices; ++k) {
	  verticesSet.insert(m.vertexNumber((*i)(k)));
	}
	size_t k=0; 
	typename ConnectivityBuilder<MeshType>::VerticesList vertices;
	for (std::set<size_t>::iterator n=verticesSet.begin();
	     n !=  verticesSet.end(); ++n) {
	  vertices[k++]=*n;
	}
	typename ConnectivityBuilder<MeshType>::FaceCorrespondance
	  ::const_iterator icell = faces.find(vertices);
	if (icell != faces.end()) {
	  (*i).setMother(icell->second.begin()->first);
	} else {
	  fferr(0) << __FILE__ << ':' << __LINE__ << ":Oops!\n"
		   << "Surface element number " << (*surfaceMesh).cellNumber(*i)
		   << " is not related to any cell of the mesh\n";
	  std::exit(1);
	}
      }
    }
  }
};


#endif // CONNECTIVITY_BUILDER_HPP
