//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: PDEOperator.hpp,v 1.3 2003/09/20 21:31:09 delpinux Exp $


#ifndef PDEOPERATOR_HPP
#define PDEOPERATOR_HPP

#include <ReferenceCounting.hpp>
#include <UserFunction.hpp>

#include <string>

/**
 * @file   PDEOperator.hpp
 * @author Stphane Del Pino
 * @date   Mon Jun  9 23:22:45 2003
 * 
 * @brief  base PDE operators class
 * 
 * The class PDEOperator is the base class for PDE Operators.
 */

class UserFunction;
class PDEOperator
{
public:
  enum Type {
    firstorderop,
    firstorderopTransposed,
    divmugrad,
    secondorderop,
    massop
  };

protected:
  //! The type of the Operator.
  const PDEOperator::Type __type;

  /*!  This is used for "composed" operators to provied an efficient
  way to know how many elementary matrices are needed to discretize
  it.
   */
  size_t __numberOfSubOperators;
public:

  virtual const UserFunction& coefficient(const size_t i) = 0;

  const size_t& numberOfSubOperators() const
  {
    return __numberOfSubOperators;
  }

  /*!
    Return the name of the PDEOperator
    \remark An overloaded virtual function would provide better implemenation.
   */
  virtual std::string TypeName() const = 0;

  //! Returns the PDEOperator type.
  const PDEOperator::Type& type() const
  {
    return __type;
  }

  //! Each operator can be "multiplied" by a function
  virtual ReferenceCounting<PDEOperator> operator * (ConstReferenceCounting<UserFunction> u) const = 0; 

  //! Each operator should now to compute its oposit
  virtual ReferenceCounting<PDEOperator> operator - () const = 0; 

  friend std::ostream& operator << (std::ostream& os,
				    const PDEOperator& pdeOp)
  {
    os << pdeOp.TypeName();
    return os;
  }

  //! Constructs a PDEOperator of type \a T.
  PDEOperator(const PDEOperator::Type& aType,
	      const size_t theNumberOfSubOperators)
    : __type(aType),
      __numberOfSubOperators(theNumberOfSubOperators)
  {
    ;
  }

  //! Copies the PDEOperator \a P.
  PDEOperator(const PDEOperator& P)
    : __type(P.__type),
      __numberOfSubOperators(P.__numberOfSubOperators)
  {
    ;
  }


  //! Virtual destructor.
  virtual ~PDEOperator()
  {
    ;
  }
};

#endif // PDEOPERATOR_HPP
