/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "gdis.h"
#include "coords.h"
#include "matrix.h"
#include "numeric.h"
#include "opengl.h"
#include <GL/gl.h>

/* externals */
extern struct sysenv_pak sysenv;


struct varray_pak
{
/* config */
gint gl_method;
gint periodic;
gint size;

/* data */
GLuint *indices;
GLdouble *vertices;
GLdouble *normals;
GLdouble *colours;
};

/**********************/
/* creation primitive */
/**********************/
gpointer va_init(void)
{
struct varray_pak *va;

va = g_malloc(sizeof(struct varray_pak));

va->gl_method = 0;
va->periodic = FALSE;
va->size = 0;

va->indices = NULL;
va->vertices = NULL;
va->normals = NULL;
va->colours = NULL;

return(va);
}

/*************************/
/* destruction primitive */
/*************************/
void va_free(gpointer ptr_varray)
{
struct varray_pak *va = ptr_varray;

g_free(va->indices);
g_free(va->vertices);
g_free(va->normals);
g_free(va->colours);
g_free(va);
}

GLuint sphere_indices[] = {0, 2, 1, 0, 3, 2, 0, 4, 3, 0, 5, 4, 0, 6, 5, 0, 1, 6};

/* CURRENT */
void va_generate_indices(gint depth)
{
gint d, i, m, n;
gint div, mstart, nstart;

n=1;
mstart = 0;
for (d=0 ; d<depth ; d++)
  {
  nstart = n;
  div = d*6;
  m = mstart;

  for (i=0 ; i<div ; i++)
    {
    printf("m,n = %d,%d\n", m,n);
    }
  }
}

/*******************************/
/* sphere array initialization */
/*******************************/
#define DEBUG_MAKE_SPHERE 1
void va_make_sphere(gpointer ptr_varray)
{
gint i, j, k, n;
gdouble layers, divisions, alpha, beta, phi, theta, sp, cp, st, ct;
struct varray_pak *va = ptr_varray;

va->gl_method = GL_TRIANGLES;

/* number of layers - azimuthal sweep */
layers = sysenv.render.sphere_quality+1.0;
alpha = 0.5*PI/layers;

/* alloc */
n=0;
for (i=0 ; i<(gint) 1.0+layers ; i++)
  n += i;
n *= 6;
n++;

/* TODO - index & vertices depend on quality */
/*
sysenv.render.sphere_quality;
*/


/* setup indices */
/* TODO - compute/more general */
va->indices = g_malloc(18 * sizeof(GLuint));
memcpy(va->indices, sphere_indices, 18*sizeof(GLuint));


/* setup vertices and normals */
va->size = n;
va->vertices = g_malloc(3 * n * sizeof(GLdouble));
va->normals = g_malloc(3 * n * sizeof(GLdouble));

/* initial vertex */
VEC3SET(va->vertices, 0.0, 0.0, -1.0);

/* compute vertices */
k = 1;
for (i=1 ; i<(gint) 1.0+layers ; i++)
  {
#if DEBUG_MAKE_SPHERE
printf("[%d/%d]\n", i, (gint) layers);
#endif

  phi = i*alpha;
  sp = tbl_sin(phi);
  cp = tbl_cos(phi);

/* divisions at this level - hexagonal sweep */
  divisions = i*6.0;
  beta = 2.0*PI/divisions;
  for (j=0 ; j<(gint) divisions ; j++)
    {
    theta = j*beta;
    st = tbl_sin(theta);
    ct = tbl_cos(theta);
 
    VEC3SET((va->vertices+3*k), ct*sp, st*sp, -cp);
    k++;

#if DEBUG_MAKE_SPHERE
printf("%7.4f %7.4f %7.4f\n", ct*sp, st*sp, -cp);
#endif
    }
  }

#if DEBUG_MAKE_SPHERE
printf("k = %d, n = %d\n", k, n);
#endif

g_assert(k == n);

memcpy(va->normals, va->vertices, 3*n*sizeof(GLdouble));
}

/*********************/
/* drawing primitive */
/*********************/
void va_draw_sphere(gpointer ptr_varray, gdouble *x, gdouble r)
{
gint i;
GLdouble *vertices;
struct varray_pak *va = ptr_varray;

/* checks */
g_assert(va != NULL);
g_assert(va->indices != NULL);
g_assert(va->vertices != NULL);

/* setup vertex/normal arrays */
/* FIXME - inefficient - prealloc a scratch array? */
vertices = g_malloc(3*va->size*sizeof(GL_DOUBLE));
memcpy(vertices, va->vertices, 3*va->size*sizeof(GL_DOUBLE));

for (i=0 ; i<18 ; i++)
  {
  VEC3MUL((vertices+3*i), r);
  ARR3ADD((vertices+3*i), x);
  }

/* draw vertex array */
glVertexPointer(3, GL_DOUBLE, 0, vertices);
if (va->normals)
  glNormalPointer(GL_DOUBLE, 0, va->normals);

/*
glDrawElements(va->gl_method, va->size, GL_UNSIGNED_INT, va->indices);
*/

glDrawElements(va->gl_method, 18, GL_UNSIGNED_INT, va->indices);

g_free(vertices);
}

