/* find -- search for files in a directory hierarchy
   Copyright (C) 1990, 91, 92, 93, 94 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* GNU find was written by Eric Decker <cire@cisco.com>,
   with enhancements by David MacKenzie <djm@gnu.ai.mit.edu>,
   Jay Plett <jay@silence.princeton.nj.us>,
   and Tim Wood <axolotl!tim@toad.com>.
   The idea for -print0 and xargs -0 came from
   Dan Bernstein <brnstnd@kramden.acf.nyu.edu>.  */

#include <config.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <unistd.h>
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#else
#include <sys/file.h>
#endif
#include "defs.h"
#include "modetype.h"
#include "misc.h"
#include "win.h"
#include "pred.h"
#include "error.h"

#ifndef S_IFLNK
#define lstat stat
#endif

int lstat ();
int stat ();

#define apply_predicate(w, pathname, stat_buf_ptr, node)	\
  (*(node)->pred_func)((w), (pathname), (stat_buf_ptr), (node))

static void process_top_path P_((win_t *w, char *pathname, gboolean *));
static int process_path P_((win_t *w, char *pathname, char *name, boolean leaf, char *parent, gboolean *));
static void process_dir P_((win_t *w, char *pathname, char *name, int pathlen, struct stat *statp, char *parent, gboolean *));
static boolean no_side_effects P_((struct predicate *pred));


#ifdef DEBUG_STAT
static int
debug_stat (file, bufp)
     char *file;
     struct stat *bufp;
{
  fprintf (stderr, "debug_stat (%s)\n", file);
  return lstat (file, bufp);
}
#endif /* DEBUG_STAT */

extern int main2 (win_t *w, int argc, char *argv[]);
int
main2 (w, argc, argv)
     win_t *w;
     int argc;
     char *argv[];
{
  int i;
  PFB parse_function;		/* Pointer to who is to do the parsing. */
  struct predicate *cur_pred;
  char *predicate_name;		/* Name of predicate being parsed. */
  gboolean stopped;

  w->program_name = argv[0];

#ifdef DEBUG_STAT
  w->xstat = debug_stat;
#else /* !DEBUG_STAT */
  w->xstat = lstat;
#endif /* !DEBUG_STAT */

#ifdef DEBUG
  printf ("cur_day_start = %s", ctime (&cur_day_start));
#endif /* DEBUG */

  /* Find where in ARGV the predicates begin. */
  for (i = 1; i < argc && strchr ("-!(),", argv[i][0]) == NULL; i++)
    /* Do nothing. */ ;

  /* Enclose the expression in `( ... )' so a default -print will
     apply to the whole expression. */
  parse_open (w, argv, &argc);
  /* Build the input order list. */
  while (i < argc)
    {
      if (strchr ("-!(),", argv[i][0]) == NULL)
	usage (w, "paths must precede expression");
      predicate_name = argv[i];
      parse_function = find_parser (predicate_name);
      if (parse_function == NULL) {
	error (w, -1, 0, "invalid predicate `%s'", predicate_name);
	return -1;
      }
      i++;
      if (!(*parse_function) (w, argv, &i))
	{
	  if (argv[i] == NULL)
	    error (w, -1, 0, "missing argument to `%s'", predicate_name);
	  else
	    error (w, -1, 0, "invalid argument `%s' to `%s'",
		   argv[i], predicate_name);
	  return -1;
	}
    }
  if (w->predicates->pred_next == NULL)
    {
      /* No predicates that do something other than set a global variable
	 were given; remove the unneeded initial `(' and add `-print'. */
      /*
       * XXX - use parse_wprint for gui
       */
      cur_pred = w->predicates;
      w->predicates = w->last_pred = w->predicates->pred_next;
      g_free ((char *) cur_pred);
      parse_wprint (w, argv, &argc);
    }
  else if (!no_side_effects (w->predicates->pred_next))
    {
      /* One or more predicates that produce output were given;
	 remove the unneeded initial `('. */
      cur_pred = w->predicates;
      w->predicates = w->predicates->pred_next;
      g_free ((char *) cur_pred);
    }
  else
    {
      /* `( user-supplied-expression ) -print'. */
      parse_close (w, argv, &argc);
      /*
       * XXX - use parse_wprint for gui
       */
      parse_wprint (w, argv, &argc);
    }

#ifdef	DEBUG
  printf ("Predicate List:\n");
  print_list (w->predicates);
#endif /* DEBUG */

  /* Done parsing the predicates.  Build the evaluation tree. */
  cur_pred = w->predicates;
  w->eval_tree = get_expr (w, &cur_pred, NO_PREC);
#ifdef	DEBUG
  printf ("Eval Tree:\n");
  print_tree (w->eval_tree, 0);
#endif /* DEBUG */

  /* Rearrange the eval tree in optimal-predicate order. */
  opt_expr (w, &w->eval_tree);

  /* Determine the point, if any, at which to stat the file. */
  mark_stat (w, w->eval_tree);

#ifdef DEBUG
  printf ("Optimized Eval Tree:\n");
  print_tree (w->eval_tree, 0);
#endif /* DEBUG */

#ifndef HAVE_FCHDIR
  starting_dir = g_get_current_dir ();
  if (starting_dir == NULL) {
    error (w, -1, errno, "cannot get current directory");
    return -1;
  }
#else
  w->starting_desc = open (".", O_RDONLY);
  if (w->starting_desc < 0) {
    error (w, -1, errno, "cannot open current directory");
    return -1;
  }
#endif

  /* If no paths are given, default to ".".  */
  for (i = 1; i < argc && strchr ("-!(),", argv[i][0]) == NULL; i++)
    process_top_path (w, argv[i], &stopped);
  if (i == 1)
    process_top_path (w, ".", &stopped);

  return (w->exit_status);
}

/* Descend PATHNAME, which is a command-line argument.  */

static void
process_top_path (w, pathname, stopped)
     win_t *w;
     char *pathname;
     gboolean *stopped;
{
  struct stat stat_buf;

  w->curdepth = 0;
  w->path_length = strlen (pathname);

  /* We stat each pathname given on the command-line twice --
     once here and once in process_path.  It's not too bad, though,
     since the kernel can read the stat information out of its inode
     cache the second time.  */
  if ((*(w->xstat)) (pathname, &stat_buf) == 0 && S_ISDIR (stat_buf.st_mode))
    {
      if (chdir (pathname) < 0)
	{
	  error (w, 0, errno, "%s", pathname);
	  w->exit_status = 1;
	  return;
	}
      process_path (w, pathname, ".", false, ".", stopped);
#ifndef HAVE_FCHDIR
      if (chdir (starting_dir) < 0) {
	error (w, -1, errno, "%s", starting_dir);
	return;
      }
#else
      if (fchdir (w->starting_desc)) {
	error (w, -1, errno, "cannot return to starting directory");
	return;
      }
#endif
    }
  else
    process_path (w, pathname, pathname, false, ".", stopped);
}

/* (Arbitrary) number of entries to grow `dir_ids' by.  */
#define DIR_ALLOC_STEP 32

/* Recursively descend path PATHNAME, applying the predicates.
   LEAF is true if PATHNAME is known to be in a directory that has no
   more unexamined subdirectories, and therefore it is not a directory.
   Knowing this allows us to avoid calling stat as long as possible for
   leaf files.

   NAME is PATHNAME relative to the current directory.  We access NAME
   but print PATHNAME.

   PARENT is the path of the parent of NAME, relative to find's
   starting directory.

   Return nonzero iff PATHNAME is a directory. */

static int
process_path (w, pathname, name, leaf, parent, stopped)
     win_t *w;
     char *pathname;
     char *name;
     boolean leaf;
     char *parent;
     gboolean *stopped;
{
  struct stat stat_buf;
  int i;

  *stopped = w->stop_search;

  /* Assume it is a non-directory initially. */
  stat_buf.st_mode = 0;

  w->rel_pathname = name;

  if (leaf)
    w->have_stat = false;
  else
    {
      if ((*(w->xstat)) (name, &stat_buf) != 0)
	{
	  error (w, 0, errno, "%s", pathname);
	  w->exit_status = 1;
	  return 0;
	}
      w->have_stat = true;
    }

  if (!S_ISDIR (stat_buf.st_mode))
    {
      win_progbar_update(w);
      my_gtk_events_flush();
      if (w->curdepth >= w->mindepth)
	apply_predicate (w, pathname, &stat_buf, w->eval_tree);
      return 0;
    }

  /* From here on, we're working on a directory.  */

  w->stop_at_current_level = w->maxdepth >= 0 && w->curdepth >= w->maxdepth;

  /* If we've already seen this directory on this branch,
     don't descend it again.  */
  for (i = 0; i <= w->dir_curr; i++)
    if (stat_buf.st_ino == w->dir_ids[i].ino &&
	stat_buf.st_dev == w->dir_ids[i].dev)
      w->stop_at_current_level = true;

  if (w->dir_alloc <= ++(w->dir_curr))
    {
      w->dir_alloc += DIR_ALLOC_STEP;
      w->dir_ids = (struct dir_id *)
	g_realloc ((char *) w->dir_ids, w->dir_alloc * sizeof (struct dir_id));
    }
  w->dir_ids[w->dir_curr].ino = stat_buf.st_ino;
  w->dir_ids[w->dir_curr].dev = stat_buf.st_dev;

  if (w->stay_on_filesystem)
    {
      if (w->curdepth == 0)
	w->root_dev = stat_buf.st_dev;
      else if (stat_buf.st_dev != w->root_dev)
	w->stop_at_current_level = true;
    }

  if (w->do_dir_first && w->curdepth >= w->mindepth)
    apply_predicate (w, pathname, &stat_buf, w->eval_tree);

  if (w->stop_at_current_level == false)
    /* Scan directory on disk. */
    process_dir (w, pathname, name, strlen (pathname), &stat_buf, parent, stopped);

  if (w->do_dir_first == false && w->curdepth >= w->mindepth)
    apply_predicate (w, pathname, &stat_buf, w->eval_tree);

  (w->dir_curr)--;

  return 1;
}

/* Scan directory PATHNAME and recurse through process_path for each entry.

   PATHLEN is the length of PATHNAME.

   NAME is PATHNAME relative to the current directory.

   STATP is the results of *xstat on it.

   PARENT is the path of the parent of NAME, relative to find's
   starting directory.  */

static void
process_dir (w, pathname, name, pathlen, statp, parent, stopped)
     win_t *w;
     char *pathname;
     char *name;
     int pathlen;
     struct stat *statp;
     char *parent;
     gboolean *stopped;
{
  char *name_space;		/* Names of files in PATHNAME. */
  int subdirs_left;		/* Number of unexamined subdirs in PATHNAME. */

  subdirs_left = statp->st_nlink - 2; /* Account for name and ".". */

  errno = 0;
  /* On some systems (VAX 4.3BSD+NFS), NFS mount points have st_size < 0.  */
  name_space = savedir (name, statp->st_size > 0 ? statp->st_size : 512);
  if (name_space == NULL)
    {
      if (errno)
	{
	  error (w, 0, errno, "%s", pathname);
	  w->exit_status = 1;
	}
      else {
	error (w, -1, 0, "virtual memory exhausted");
	return;
      }
    }
  else
    {
      register char *namep;	/* Current point in `name_space'. */
      char *cur_path;		/* Full path of each file to process. */
      char *cur_name;		/* Base name of each file to process. */
      unsigned cur_path_size;	/* Bytes allocated for `cur_path'. */
      register unsigned file_len; /* Length of each path to process. */
      register unsigned pathname_len; /* PATHLEN plus trailing '/'. */

      if (pathname[pathlen - 1] == '/')
	pathname_len = pathlen + 1; /* For '\0'; already have '/'. */
      else
	pathname_len = pathlen + 2; /* For '/' and '\0'. */
      cur_path_size = 0;
      cur_path = NULL;

      if (strcmp (name, ".") && chdir (name) < 0)
	{
	  error (w, 0, errno, "%s", pathname);
	  w->exit_status = 1;
	  return;
	}

      for (namep = name_space; *namep; namep += file_len - pathname_len + 1)
	{
	  /* Append this directory entry's name to the path being searched. */
	  file_len = pathname_len + strlen (namep);
	  if (file_len > cur_path_size)
	    {
	      while (file_len > cur_path_size)
		cur_path_size += 1024;
	      if (cur_path)
		g_free (cur_path);
	      cur_path = g_new(char, cur_path_size);
	      strcpy (cur_path, pathname);
	      cur_path[pathname_len - 2] = '/';
	    }
	  cur_name = cur_path + pathname_len - 1;
	  strcpy (cur_name, namep);

	  w->curdepth++;
	  if (!w->no_leaf_check)
	    /* Normal case optimization.
	       On normal Unix filesystems, a directory that has no
	       subdirectories has two links: its name, and ".".  Any
	       additional links are to the ".." entries of its
	       subdirectories.  Once we have processed as many
	       subdirectories as there are additional links, we know
	       that the rest of the entries are non-directories --
	       in other words, leaf files. */
	    subdirs_left -= process_path (w, cur_path, cur_name,
					  subdirs_left == 0, pathname, stopped);
	  else
	    /* There might be weird (e.g., CD-ROM or MS-DOS) filesystems
	       mounted, which don't have Unix-like directory link counts. */
	    process_path (w, cur_path, cur_name, false, pathname, stopped);
	  w->curdepth--;
	  if (*stopped)
	      break;
	}

      if (strcmp (name, "."))
	{
	  if (!w->dereference)
	    {
	      if (chdir ("..") < 0) {
		/* We could go back and do the next command-line arg instead,
		   maybe using longjmp.  */
		error (w, -1, errno, "%s", parent);
		return;
	      }
	    }
	  else
	    {
#ifndef HAVE_FCHDIR
	      if (chdir (starting_dir) || chdir (parent)) {
		error (w, -1, errno, "%s", parent);
		return;
	      }
#else
	      if (fchdir (w->starting_desc) || chdir (parent)) {
		error (w, -1, errno, "%s", parent);
		return;
	      }
#endif
	    }
	}

      if (cur_path)
	g_free (cur_path);
      g_free (name_space);
    }
}

/* Return true if there are no side effects in any of the predicates in
   predicate list PRED, false if there are any. */

static boolean
no_side_effects (pred)
     struct predicate *pred;
{
  while (pred != NULL)
    {
      if (pred->side_effects)
	return (false);
      pred = pred->pred_next;
    }
  return (true);
}
