/*
**  ExtendedTextView.m
**
**  Copyright (c) 2002, 2003
**
**  Author: Ujwal S. Sathyam <ujwal@setlurgroup.com>
**          Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "ExtendedTextView.h"

#include "ExtendedAttachmentCell.h"
#include "GNUMail.h"
#include "GNUMail/GNUMailBundle.h"
#include "Constants.h"
#include "MimeTypeManager.h"
#include "MimeType.h"
#include "Utilities.h"

#include <Pantomime/MimeUtility.h>
#include <Pantomime/NSString+Extensions.h>

//
//
//
@implementation ExtendedTextView

- (id) init
{
  self = [super init];
  
  if (cursor == nil)
    {
      cursor = [[NSCursor alloc] initWithImage: [NSImage imageNamed: @"hand"]
				 hotSpot: NSZeroPoint];
    }
  
  return self;
}


- (id) initWithFrame: (NSRect) theFrame
{
  self = [super initWithFrame: theFrame];
  
  if (cursor == nil)
    {
      cursor = [[NSCursor alloc] initWithImage: [NSImage imageNamed: @"hand"]
				 hotSpot: NSZeroPoint];
    }
  
  return (self);
}


- (id) initWithFrame: (NSRect) theRect  textContainer: (NSTextContainer *) theTextContainer
{
  self = [super initWithFrame: theRect  textContainer: theTextContainer];
  
  if (cursor == nil)
    {
      cursor = [[NSCursor alloc] initWithImage: [NSImage imageNamed: @"hand"]
				 hotSpot: NSZeroPoint];
      
    }
  
  return (self);
}


//
//
//
- (void) dealloc
{
  RELEASE(cursor);
  [super dealloc];
}


//
// NSTextView
//
- (void) pasteAsQuoted: (id) sender
{
  NSString *aString, *aQuotedString;
  NSPasteboard *pboard;
  
  pboard = [NSPasteboard generalPasteboard];
  
  aString = [pboard stringForType:NSStringPboardType];
  
  if (aString != nil)
    {
      aQuotedString = [MimeUtility unwrapPlainTextString: aString
				   usingQuoteWrappingLimit: 78];
      [self insertText: [MimeUtility quotePlainTextString: aQuotedString
				     quoteLevel: 1
				     wrappingLimit: 80]];
    }
}


//
// Drag and Drop methods for our text view
//

//
// Called when our drop area is entered
//
- (unsigned int) draggingEntered:(id <NSDraggingInfo>)sender
{
  unsigned int iResult;
  
  // Reset our handler flag in case the the parent NSTextView can handle this drag'n'drop.
  bExtendedDragNDrop = NO;
  
  // Let's see if our parent NSTextView knows what to do
  iResult = [super draggingEntered: sender];
  
  // If parent class does not know how to deal with this drag type, check if we do.
  if (iResult == (NSDragOperationCopy|NSDragOperationGeneric))
    {
      bExtendedDragNDrop = YES;
    }

  if (iResult == NSDragOperationNone)
    {
      // Set our handler flag to notify our other methods that we handling this drag'n'drop
      bExtendedDragNDrop = YES;
      return [self _checkForSupportedDragTypes: sender];
    }
  
  return iResult;
}


//
// Called when the dragged object is moved within our drop area
//
- (unsigned int) draggingUpdated:(id <NSDraggingInfo>)sender
{
  unsigned int iResult;
   
  // Let's see if our parent NSTextView knows what to do
  iResult = [super draggingUpdated: sender];
  
  // If parent class does not know how to deal with this drag type, check if we do.
  if (iResult == NSDragOperationNone) // Parent NSTextView does not support this drag type.
    {
      return [self _checkForSupportedDragTypes: sender];
    }
  
  return iResult;
}


//
// Called when the dragged object leaves our drop area
//
- (void) draggingExited:(id <NSDraggingInfo>)sender
{
  // We don't do anything special, so let the parent NSTextView handle this.
  [super draggingExited: sender];
  
  // Reset our handler flag
  bExtendedDragNDrop = NO;
}


//
// Called when the dragged item is about to be released in our drop area.
//
- (BOOL) prepareForDragOperation:(id <NSDraggingInfo>)sender
{
  BOOL bResult;
  
  // Check if parent NSTextView knows how to handle this.
  bResult = [super prepareForDragOperation: sender];
  
  // If parent class does not know how to deal with this drag type, check if we do.
  if ( bResult != YES && 
       [self _checkForSupportedDragTypes: sender] != NSDragOperationNone )
    {
      bResult = YES;
    }

  return bResult;
}


//
// Called when the dragged item is released in our drop area.
//
- (BOOL) performDragOperation:(id <NSDraggingInfo>)sender
{
  unsigned int dragOperation;
  BOOL bResult;
  
  // Check if parent NSTextView knows how to handle this.
  // This always seems to return "YES" even if the parent can't handle this drag type.
  // Is this a Cocoa bug?
  // To get around this, we use a flag to let us know if the extended child class
  // handling the drag'n'drop.
  bResult = [super performDragOperation: sender];
  
  // If parent class does not know how to deal with this drag type, check if we do.
  if (bExtendedDragNDrop)
    {
      NSPasteboard *pb = [sender draggingPasteboard];
      NSArray *propertyList;
      int i;
      
      dragOperation = [self _checkForSupportedDragTypes: sender];
      
      switch (dragOperation)
	{
	case NSDragOperationCopy:
	  propertyList = [pb propertyListForType: NSFilenamesPboardType];

	  for (i = 0; i < [propertyList count]; i++)
	    {
	      [self insertFile: (NSString*)[propertyList objectAtIndex: i]];
	    }

	  bResult = YES;
	  break;				
	}
      
    }
  
  return bResult;
}


//
//
//
- (void) concludeDragOperation:(id <NSDraggingInfo>)sender
{
  // If we did no handle the drag'n'drop, ask our parent to clean up
  // I really wish the concludeDragOperation would have a useful exit value.
  if (!bExtendedDragNDrop)
    {
      [super concludeDragOperation: sender];
    }
  
  bExtendedDragNDrop = NO;
}


//
// Inserts an individual file.
//
- (BOOL) insertFile: (NSString *) theFilename
{
  NSAttributedString *aAttributedString;
  NSTextAttachment *aTextAttachment;
  ExtendedAttachmentCell *cell;
  NSFileWrapper *aFileWrapper;
  MimeType *aMimeType;
  
  aFileWrapper = [[NSFileWrapper alloc] initWithPath: theFilename];
  AUTORELEASE(aFileWrapper);
  
  // We save the path of the last file
  [GNUMail setCurrentWorkingPath: [theFilename stringByDeletingLastPathComponent]];
  
  // We set the icon of the attachment if the mime-type is found
  aMimeType = [[MimeTypeManager singleInstance] bestMimeTypeForFileExtension: [[theFilename lastPathComponent]
										pathExtension]];
  
  // If we found a MIME-type for this attachment...
  if ( (aMimeType && [aMimeType icon]) ||
       (aMimeType && [[aMimeType primaryType] caseInsensitiveCompare: @"image"] == NSOrderedSame) )
    {
      // If we got an image, we resize our attachment to fix the text view, if we need to.
      if ( [[aMimeType primaryType] caseInsensitiveCompare: @"image"] == NSOrderedSame )
	{
	  NSImage *anImage;
	  
	  NSRect rectOfTextView;
	  NSSize imageSize;

	  anImage = [[NSImage alloc] initWithContentsOfFile: theFilename];
	  rectOfTextView = [self frame];
	  imageSize = [anImage size];
	  
	  if ( imageSize.width > rectOfTextView.size.width )
	    {
	      double delta =  1.0 / ( imageSize.width / rectOfTextView.size.width );
	      double dy = 15*delta;
	      
	      [anImage setScalesWhenResized: YES];
	      [anImage setSize: NSMakeSize( ((imageSize.width-15) * delta), 
							(imageSize.height-dy) * delta)];
	    }

	  [aFileWrapper setIcon: anImage];
	  RELEASE(anImage);
	}
      else
	{
	  [aFileWrapper setIcon: [aMimeType icon]];
	}
    }

  // We now create our attachment from our filewrapper
  aTextAttachment = [[NSTextAttachment alloc] initWithFileWrapper: aFileWrapper];
  
  cell = [[ExtendedAttachmentCell alloc] initWithFilename: [theFilename lastPathComponent]
					 size: [[aFileWrapper regularFileContents] length]];
  
  [aTextAttachment setAttachmentCell: cell];
  
  // Cocoa bug
#ifdef MACOSX
  [cell setAttachment: aTextAttachment];
  [cell setImage: [aFileWrapper icon]];
#endif
  RELEASE(cell);
  
  aAttributedString = [NSAttributedString attributedStringWithAttachment: aTextAttachment];
  RELEASE(aTextAttachment);
  
  if ( aAttributedString )
    {
      [self insertText: (id)aAttributedString];
    }
  
  return YES;
}


//
//  
//
- (void) updateCursorForLinks
{
  NSTextStorage *aTextStorage;
  
  NSRange visibleGlyphRange, visibleCharRange, attrsRange;
  NSPoint aPoint;
  NSRect aRect;
  
  // We get the attributed text
  aTextStorage = [self textStorage];
  
  // We found out what part is visible
  aPoint = [self textContainerOrigin];
  aRect = NSOffsetRect ([self visibleRect], -aPoint.x, -aPoint.y);
  
  // We found out what characters are visible
  visibleGlyphRange = [[self layoutManager] glyphRangeForBoundingRect: aRect
					    inTextContainer: [self textContainer]];

  visibleCharRange = [[self layoutManager] characterRangeForGlyphRange: visibleGlyphRange
					   actualGlyphRange: NULL];
  
  attrsRange = NSMakeRange(visibleCharRange.location, 0);
  
  // Loop whitin the visible range of characters
  while (NSMaxRange(attrsRange) < NSMaxRange(visibleCharRange))
    {
      if ( [aTextStorage attribute: NSLinkAttributeName
			 atIndex: NSMaxRange(attrsRange)
			 effectiveRange: &attrsRange] )
	{
	  unsigned int rectCount, rectIndex;
	  NSRect *rects;
	 
	  // Find the rectangle(s) associated to the link
	  // NB: there may be multiple rects if the link uses many lines
	  rects = [[self layoutManager] rectArrayForCharacterRange: attrsRange
#ifdef MACOSX
					withinSelectedCharacterRange: NSMakeRange(NSNotFound, 0)
#else
					withinSelectedCharacterRange: attrsRange
#endif
					inTextContainer: [self textContainer]
					rectCount: &rectCount];
	  
	  // For the visible part of each rectangle, make the cursor visible
	  for (rectIndex = 0; rectIndex < rectCount; rectIndex++)
	    {
	      aRect = NSIntersectionRect(rects[rectIndex], [self visibleRect]);
	      [self addCursorRect: aRect
		    cursor: cursor];
	    }
	}
    }
}


//
//
//
- (void) resetCursorRects
{
  [super resetCursorRects];
  [self updateCursorForLinks];
} 

@end


//
// private methods
//
@implementation ExtendedTextView (Private)

- (unsigned int) _checkForSupportedDragTypes: (id<NSDraggingInfo>) sender
{
  NSString *sourceType;
  BOOL iResult;
  
  iResult = NSDragOperationNone;

  // We support the FileName drag type for attching files
  sourceType = [[sender draggingPasteboard] availableTypeFromArray: [NSArray arrayWithObjects: 
									       NSFilenamesPboardType, 
									     NSStringPboardType, 
									     nil]];
  
  if (sourceType)
    {
      iResult = NSDragOperationCopy;
    }
  
  return iResult;
}

@end
