/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 * Licensed under the GNU GPL.
 * Absolutely no warranty.
 */

#include <math.h>
#include <audiofile.h>
#include "mem.h"
#include "cache.h"
#include "block.h"
#include "snd.h"
/*
block *
block_insert(block *b,
             frame_bits_t frame_bits,
             AFframecount frame_offset,
             AFframecount frame_count) {
    block *b_head, *b_tail;

    b_head = b;
    if(b_head->frame_count >= DEF_BLOCK_SIZE) {
        DEBUG("block too big (%ld), not inserting\n",
              b_head->frame_count);
        return NULL;
    }

    DEBUG("trying to fit %ld new frames into block of size %ld\n",
          frame_count, b_head->frame_count + frame_count);

    if(frame_offset && b_head->frame_count > frame_offset) {
        b_tail = block_split(b_head, frame_offset);
        if(!b_tail) {
            FAIL("cannot split block\n");
            return NULL;
        }
    }
    
    frames_before_resize = b_head->frame_count;
    b = block_resize(b_head, frames_before_resize + frame_count);
    if(!b) {
        FAIL("cannot resize block from %ld to %ld, recovering...\n",
             frames_before_resize, frames_before_resize + frame_count);
        if(b_tail) {
            b_head = block_join(b_head, b_tail);
            if(!b_head) 
                FAIL("recovery failed, %ld frames lost!\n", frame_count);
            block_unref(b_tail);
        }
        return NULL;
    }
    
    cache_fill(b_head->frame_cache,
               frame_bits,
               frames_before_resize * b_head->frame_width,
               frame_count * b_head->frame_width);
    snd_frames_buffer_to_graph_buffer(b_head->graph_cache->data + 
                                      (int)floor(frames_before_resize / b_head->graph_hres) * sizeof(_graph_bits_t),
                                      frame_bits,
                                      frame_count,
                                      b_head->frame_width,
                                      b_head->graph_hres);
    b_head->graph_cache->high = b_head->graph_cache->sz;
}
*/

block *
block_resize(block *block,
             AFframecount frame_count) {
    int err;
    AFframecount frames_before_resize = block->frame_count;
    block->frame_count = frame_count;
    err = cache_resize(block->graph_cache_low, 
                       ceil(frame_count / block->graph_hres) * sizeof(graph_bits_unit_t));
    if(err) {
        FAIL("could not resize low graph cache.\n");
        return NULL;
    }
    err = cache_resize(block->graph_cache_high, 
                       ceil(frame_count / block->graph_hres) * sizeof(graph_bits_unit_t));
    if(err) {
        FAIL("could not resize high graph cache, trying to recover low graph cache...\n");
        err = cache_resize(block->graph_cache_low, 
                           ceil(frames_before_resize / 
                                block->graph_hres) * sizeof(graph_bits_unit_t));
        if(err) {
            FAIL("could not recover low graph cache, flushing graph cache\n");
            cache_clear(block->graph_cache_low);
            cache_clear(block->graph_cache_high);
        }
        return NULL;
    }


    err = cache_resize(block->frame_cache, 
                       frame_count * block->frame_width);

    if(err) {

        FAIL("could not resize frame cache, trying to recover low & high graph cache...\n");
        err = cache_resize(block->graph_cache_low, 
                           ceil(frames_before_resize / 
                                block->graph_hres) * sizeof(graph_bits_unit_t));
        if(err) {
            FAIL("could not recover low graph cache, flushing graph cache\n");
            cache_clear(block->graph_cache_low);
            cache_clear(block->graph_cache_high);
        }
        return NULL;

        err = cache_resize(block->graph_cache_high, 
                           ceil(frames_before_resize / 
                                block->graph_hres) * sizeof(graph_bits_unit_t));
        if(err) {
            FAIL("could not recover high graph cache, flushing graph cache\n");
            cache_clear(block->graph_cache_low);
            cache_clear(block->graph_cache_high);
        }
        return NULL;
    }
    return block;
}

void
block_move(block *block,
           AFframecount new_frame_offset,
           AFframecount old_frame_offset,
           AFframecount frame_count) {
    cache_move(block->frame_cache, 
               new_frame_offset * block->frame_width,
               old_frame_offset * block->frame_width,
               frame_count * block->frame_width);
    cache_move(block->graph_cache_low, 
               (size_t) (new_frame_offset / block->graph_hres) * sizeof(graph_bits_unit_t),
               (size_t) (old_frame_offset / block->graph_hres) * sizeof(graph_bits_unit_t),
               (size_t) (ceil(frame_count / block->graph_hres)) * sizeof(graph_bits_unit_t));
    cache_move(block->graph_cache_high, 
               (size_t) (new_frame_offset / block->graph_hres) * sizeof(graph_bits_unit_t),
               (size_t) (old_frame_offset / block->graph_hres) * sizeof(graph_bits_unit_t),
               (size_t) (ceil(frame_count / block->graph_hres)) * sizeof(graph_bits_unit_t));
}

block *
block_construct(int type,
                int frame_width,
                float graph_hres,
                AFframecount frame_count,
                cache *fc,
                cache *gc_low,
                cache *gc_high) {
    block *b;
    //    DEBUG("frame_count: %ld\n", frame_count);
    b = mem_calloc(1, sizeof(block));
    if(!b)
        return NULL;
    b->type = type;
    b->ref = 1;
    b->frame_width = frame_width;
    b->graph_hres = graph_hres;
    b->frame_count = frame_count;
    b->frame_cache = fc;
    b->graph_cache_low = gc_low;
    b->graph_cache_high = gc_high;
    if(frame_count == 0)
        abort();

    if(b->type == CACHE_NULL) 
        DEBUG("created %ld frame NULL block.\n", frame_count);

    if(!b->frame_cache) {
        FAIL("failed to allocate frame cache, block allocation failed\n");
        if(b->graph_cache_low) 
            cache_destroy(b->graph_cache_low);
        if(b->graph_cache_high) 
            cache_destroy(b->graph_cache_high);
        free(b);
        return NULL;
    }
    b->graph_cache_low = gc_low;
    if(!b->graph_cache_low) {
        FAIL("failed to allocate low graph cache, block allocation failed\n");
        cache_destroy(b->frame_cache);
        if(b->graph_cache_high)
            cache_destroy(b->graph_cache_high);
        free(b);
        return NULL;
    }
    b->graph_cache_high = gc_high;
    if(!b->graph_cache_high) {
        FAIL("failed to allocate high graph cache, block allocation failed\n");
        cache_destroy(b->graph_cache_low);
        cache_destroy(b->frame_cache);
        free(b);
        return NULL;
    }
    return b;
}

block *
block_new(cache_type type,
          int frame_width,
          float graph_hres,
          AFframecount frame_count) {
    return block_construct(type,
                           frame_width,
                           graph_hres,
                           frame_count,
                           cache_new(type, frame_count * frame_width),
                           cache_new(type, ceil(frame_count / graph_hres) * 
                                     sizeof(graph_bits_unit_t)),
                           cache_new(type, ceil(frame_count / graph_hres) * 
                                     sizeof(graph_bits_unit_t)));

}

/* Clones a block (partially). The frame_offset and frame_count parameters
   specify what part of the block should be cloned. */

block *
block_clone(block *block,
            AFframecount frame_offset,
            AFframecount frame_count) {
    size_t foff, fc, goff, gc;
    struct _block *blk_split;

    foff = frame_offset * block->frame_width;
    fc = frame_count * block->frame_width;
    goff = (size_t) (frame_offset / block->graph_hres) * sizeof(graph_bits_unit_t);
    gc = ceil(frame_count / block->graph_hres) * sizeof(graph_bits_unit_t);
    
    blk_split = block_construct(block->type,
                                block->frame_width,
                                block->graph_hres, 
                                frame_count,
                                cache_clone(block->frame_cache, foff, fc),
                                cache_clone(block->graph_cache_low, goff, gc),
                                cache_clone(block->graph_cache_high, goff, gc));
    
    if(!blk_split) {
        FAIL("cannot create new block for clone (%ld frames)\n", frame_count);
        return NULL;
    }

    return blk_split;
}

/* FIXME: does not work correctly for blocks where b1->frame_count <
   GRAPH_HRES or b2->frame_count < GRAPH_HRES. */

/*
block *
block_join(block *b1,
           block *b2) {
    cache *c1, *c2;

    //    DEBUG("block join\n");

    if(!(b1->type == b2->type &&
         b1->frame_width == b2->frame_width &&
         b1->graph_hres == b2->graph_hres)) {
        FAIL("cannot join two different kinds of blocks\n");
        return NULL;
    }

    // If the result will fit in a single peak element then
    // don't bother with reallocating and moving memory. 
    
    if(b1->type == CACHE_REAL &&
       b1->frame_count + b2->frame_count <= b1->graph_hres) {
        ((struct graph_bits_t *)b1->graph_cache->data)->l = 
            MIN(((struct graph_bits_t *)b1->graph_cache->data)->l,
                ((struct graph_bits_t *)b2->graph_cache->data)->l);
        ((struct graph_bits_t *)b1->graph_cache->data)->h = 
            MAX(((struct graph_bits_t *)b1->graph_cache->data)->h,
                ((struct graph_bits_t *)b2->graph_cache->data)->h);
        return b1;
    }
    
    c1 = cache_join(b1->frame_cache, b2->frame_cache);
    c2 = cache_join(b1->graph_cache, b2->graph_cache);

    /// Shrink the caches back on failure. 

    if(!c1) {
        FAIL("could not join frame cache\n");
        if(c2) 
            cache_resize(c2, 
                         ceil(b1->frame_count / b1->graph_hres) *
                         sizeof(struct graph_bits_t));
        return NULL;
    }
    if(!c2) {
        FAIL("could not join graph cache\n");
        if(c1)
            cache_resize(c1, b1->frame_count * b1->frame_width);
        return NULL;
    }

    cache_resize(b1->graph_cache, 
                 ceil((b1->frame_count + b2->frame_count) / b1->graph_hres) * 
                 sizeof(struct graph_bits_t));

    snd_frames_buffer_to_graph_buffer(((struct graph_bits_t *)(b1->graph_cache->data)) + 
                                      (int)floor(b1->frame_count / b1->graph_hres),
                                      (frame_bits_t)(b1->frame_cache->data) + 
                                      (b1->frame_count * b1->frame_width),
                                      b2->frame_count,
                                      b1->frame_width,
                                      b1->graph_hres);

    //    DEBUG("joined b1: %p; b1->frame_count: %ld, b2->frame_count: %ld\n",
    //          b1, b1->frame_count, b2->frame_count);
    b1->frame_count += b2->frame_count;

    return b1;
}
*/

block *
block_split(block *block,
            AFframecount frame_offset) {
    struct _block *blk_split;

    if(frame_offset == 0) {
        FAIL("frame_offset == 0\n");
        abort();
    }

    //    DEBUG("block->frame_count: %ld, frame_offset: %ld (difference: %ld)\n",
    //          block->frame_count, frame_offset, block->frame_count - frame_offset);

    /* FIXME: the size of the two new graph caches size does not
       always equal the size of the graph cache before it got
       split! */

    blk_split = block_clone(block,
                            frame_offset,
                            block->frame_count - frame_offset);

    if(!blk_split) {
        FAIL("could not copy block (%ld-%ld)\n", frame_offset, block->frame_count - frame_offset);
        return NULL;
    }

    block_resize(block, frame_offset);
    /*

    DEBUG("return block 1 size: %ld, block 2 size: %ld\n",
          block->frame_count, blk_split->frame_count);
    */

    return blk_split;
}

void
block_destroy(block *block) {
    //    DEBUG("destroying block, %ld frames\n", block->frame_count);
    cache_destroy(block->frame_cache);
    cache_destroy(block->graph_cache_low);
    cache_destroy(block->graph_cache_high);
    free(block);
}

void
block_unref(block *block) {
    block->ref--;
    if(block->ref == 0)
        block_destroy(block);
}

void
block_addref(block *block) {
    block->ref++;
}
