/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 * Licensed under the GNU GPL.
 * Absolutely no warranty.
 */

#include <math.h>
#include <audiofile.h>
#include <config.h>
#include "mem.h"
#include "shell.h"
#include "module.h"
#include "action.h"
#include "gui.h"

#ifdef HAVE_SAMPLERATE
#include <samplerate.h>
#endif

char *curdir = ".";
char *pathpart = NULL;

const char *
dirname(const char *s) {
    char *s2;
    s2 = rindex(s, '/');
    if(!s2)
        return curdir;
    if(pathpart) 
        mem_free(pathpart);
    pathpart = mem_alloc(s2 - s + 1);
    if(!pathpart) {
        FAIL("cannot create dirname\n");
        return curdir;
    }
    memcpy(pathpart, s, s2 - s);
    pathpart[s2 -s] = '\0';
    return pathpart;
}

void
mix(shell *target_shell,
    int target_channel,
    AFframecount target_offset,
    snd *source_sr,
    int source_channel,
    AFframecount source_offset,
    AFframecount frame_count) {
    int i;
    int32_t *out_buffer;
    double marker_value = 0;
    AFframecount source_read = 0;
    ITERATOR_INIT(target_offset, frame_count);

    out_buffer = mem_calloc(1, EFFECT_BUF_SIZE * sizeof(int32_t));
    if(!out_buffer) {
        FAIL("not enough memory for mix buffer (%"CONVSPEC_SIZE_T" bytes)\n",
             EFFECT_BUF_SIZE * sizeof(int32_t));
        ITERATOR_EXIT();
        return;
    }

    ITERATOR(target_shell, target_shell->sr->tracks[target_channel], 
             for(i = 0; i < iter_read; i++) {
                 marker_value = 
                     marker_list_slope_value(target_shell->sr->markers[target_channel],
                                             iter_frame_offset + i,
                                             MARKER_SLOPE); 
                 out_buffer[i] = int32_frame_bits[i] + 
                     (int32_frame_bits[i] * marker_value);
             }
             source_read = track_int32_frames_get(source_sr->tracks[source_channel],
                                                  int32_frame_bits,
                                                  source_offset,
                                                  iter_read);
             for(i = 0; i < source_read; i++) {
                 marker_value = 
                     marker_list_slope_value(source_sr->markers[source_channel],
                                             source_offset + i,
                                             MARKER_SLOPE); 
                 out_buffer[i] += int32_frame_bits[i] + 
                     (int32_frame_bits[i] * marker_value);
             }
             DEBUG("iter_frame_offset: %ld\n", iter_frame_offset);
             blocklist_blocks_destroy(track_delete(target_shell->sr->tracks[target_channel],
                                                   iter_frame_offset,
                                                   iter_read));
             track_int32_frames_insert(target_shell->sr->tracks[target_channel],
                                       out_buffer,
                                       iter_frame_offset,
                                       iter_read);
             source_offset += iter_read;
             memset(out_buffer, '\0', iter_read * sizeof(int32_t)));
    
    free(out_buffer);
    ITERATOR_EXIT();
}

#define RESAMPLE_INT_ZOH_DESCRIPTION "A very bad (but very fast) resampling algorithm. You are better off installing SRC: http://www.mega-nerd.com/SRC/";
const char *
resample_algo_name(int algo) {
#ifdef HAVE_SAMPLERATE
    if(src_get_name(algo) == NULL && src_get_name(algo-1))
        return "Integer ZOH";
    return src_get_name(algo);
#else
    if(algo == 0) 
        return "Integer ZOH";
    return NULL;
#endif
}

const char *
resample_algo_description(int algo) {
#ifdef HAVE_SAMPLERATE
    if(src_get_description(algo) == NULL && src_get_description(algo-1))
        return RESAMPLE_INT_ZOH_DESCRIPTION;
    return src_get_description(algo);
#else
    if(algo == 0) 
        return RESAMPLE_INT_ZOH_DESCRIPTION;
    return NULL;
#endif
}

AFframecount
#ifdef HAVE_SAMPLERATE
resample_int_zoh(shell *shl,
#else
resample(shell *shl,
#endif
         int track,
         AFframecount start_offset,
         AFframecount end_offset,
         AFframecount new_frame_count,
         int algo,
         int honor_envelopes,
         int guarantee_new_frame_count) {
    int32_t *out_buffer;
    double j = 0, factor = (double)(end_offset - start_offset) / (double)new_frame_count,
        adjusted_factor = factor, slope_value;
    AFframecount out_buf_count = ceil((1.0f / (factor / 2)) * EFFECT_BUF_SIZE);
    AFframecount out_buf_count_iteration = 0, total_frames_written = 0;
    
    ITERATOR_INIT(start_offset, end_offset - start_offset);
    
    DEBUG("start_offset: %ld, end_offset: %ld, new_frame_count: %ld, factor: %f\n",
          start_offset, end_offset, new_frame_count, factor);
    out_buffer = mem_calloc(1, out_buf_count * sizeof(int32_t));
    if(!out_buffer) {
        FAIL("not enough memory for resample buffer (%ld bytes)\n",
             out_buf_count * sizeof(int32_t));
        ITERATOR_EXIT();
        return end_offset - start_offset;
    }
    
    ITERATOR(shl, shl->sr->tracks[track], 
             for(out_buf_count_iteration = 0, j = 0; floor(j) < iter_read;
                 out_buf_count_iteration++, j += adjusted_factor) {
                 out_buffer[out_buf_count_iteration] = int32_frame_bits[(int)floor(j)];
                 if(honor_envelopes &&
                    (shl->sr->markers[track]->marker_types_enabled & MARKER_SLOPE)) {
                     slope_value = marker_list_slope_value(shl->sr->markers[track],
                                                           iter_frame_offset + out_buf_count_iteration,
                                                           MARKER_SLOPE); 
                     adjusted_factor = (slope_value < 0 ? 
                                        factor - ((factor * -slope_value) / 2) :
                                        factor + (factor * slope_value));
                     
                 }
             }
             blocklist_blocks_destroy(track_delete(shl->sr->tracks[track],
                                                   iter_frame_offset,
                                                   iter_read));
             
             if(guarantee_new_frame_count && 
                total_frames_written + out_buf_count_iteration > new_frame_count) {
                 DEBUG("clamping, %ld frames lost\n", 
                       (total_frames_written + out_buf_count_iteration) - new_frame_count);
                 out_buf_count_iteration -= 
                     (total_frames_written + out_buf_count_iteration) - new_frame_count;
             }
             track_int32_frames_insert(shl->sr->tracks[track],
                                       out_buffer,
                                       iter_frame_offset,
                                       out_buf_count_iteration);
             iter_written = out_buf_count_iteration;
             total_frames_written += iter_written);
    
    if(guarantee_new_frame_count && total_frames_written < new_frame_count) {
        DEBUG("padding, %ld frames inserted\n",
              new_frame_count - total_frames_written);
        track_null_insert(shl->sr->tracks[track],
                          iter_frame_offset,
                          new_frame_count - total_frames_written);
        total_frames_written += (new_frame_count - total_frames_written);
    }
    free(out_buffer);
    ITERATOR_EXIT();
    return total_frames_written;
}

#ifdef HAVE_SAMPLERATE
AFframecount
resample(shell *shl,
         int track,
         AFframecount start_offset,
         AFframecount end_offset,
         AFframecount new_frame_count,
         int algo,
         int honor_envelopes,
         int guarantee_new_frame_count) {
    float *out_buffer;
    double factor = (double)(end_offset - start_offset) / (double)new_frame_count,
        adjusted_factor = factor, slope_value;
    AFframecount out_buf_count = ceil((1.0f / (factor / 2)) * EFFECT_BUF_SIZE);
    AFframecount out_buf_count_iteration = 0, in_buf_count_iteration = 0, total_frames_written = 0;
    struct marker *marker;
    int err;
    SRC_STATE *src_state;
    SRC_DATA src_data;
    ITERATORF_INIT(start_offset, end_offset - start_offset);

    if(src_get_name(algo) == NULL && src_get_name(algo-1)) {
        ITERATORF_EXIT();
        return resample_int_zoh(shl, 
                                track,
                                start_offset,
                                end_offset,
                                new_frame_count, 
                                algo,
                                honor_envelopes, 
                                guarantee_new_frame_count);
    }

    DEBUG("start_offset: %ld, end_offset: %ld, new_frame_count: %ld, factor: %f\n",
          start_offset, end_offset, new_frame_count, factor);

    src_state = src_new(algo, 1, &err);
    if(!src_state) {
        FAIL("cannot initialize sample rate converter (algorithm: %d): %s\n",
             algo, src_strerror(err));
        ITERATORF_EXIT();
        return end_offset - start_offset;
    }

    out_buffer = mem_calloc(1, out_buf_count * sizeof(float));
    if(!out_buffer) {
        FAIL("not enough memory for resample buffer (%ld bytes)\n",
             out_buf_count * sizeof(float));
        src_delete(src_state) ;
        ITERATORF_EXIT();
        return end_offset - start_offset;
    }
    src_data.end_of_input = 0;
    ITERATORF(shl, shl->sr->tracks[track], 
              if(!honor_envelopes ||
                 !(shl->sr->markers[track]->marker_types_enabled & MARKER_SLOPE)) {

                  /* Envelopes are disabled, that makes the entire
                     process very simple. */

                  DEBUG("envelope disabled or not honored\n");
                  src_data.end_of_input = iter_is_last_iteration;
                  src_data.input_frames = iter_read;
                  src_data.output_frames = out_buf_count;
                  src_data.data_in = float_frame_bits;
                  src_data.data_out = out_buffer;
                  src_data.src_ratio = 1 / adjusted_factor;
                  if((err = src_process(src_state, &src_data))) {
                      FAIL("samplerate returned error: %s\n",
                           src_strerror(err));
                      ITERATOR_ESCAPE();
                      break;
                  }
                  out_buf_count_iteration = src_data.output_frames_gen;
                  DEBUG("input_frames_used: %ld, output_frames_gen: %ld\n", 
                        src_data.input_frames_used, src_data.output_frames_gen);
              } else {

                  /* Envelopes are enabled, so we need to split up the
                     job along envelope slope change boundaries. We do
                     this by splitting it into a head, a body, and a
                     tail. */

                  in_buf_count_iteration = out_buf_count_iteration = 0;

                  /* Head. */

                  slope_value = marker_list_slope_value(shl->sr->markers[track],
                                                        iter_frame_offset,
                                                        MARKER_SLOPE);
                  src_data.input_frames = 1;
                  src_data.output_frames = out_buf_count;
                  src_data.data_in = &float_frame_bits[in_buf_count_iteration];
                  src_data.data_out = &out_buffer[out_buf_count_iteration];
                  adjusted_factor = (slope_value < 0 ? 
                                     factor - ((factor * -slope_value) / 2) :
                                     factor + (factor * slope_value));
                  src_data.src_ratio = 1 / adjusted_factor;
                  if((err = src_process(src_state, &src_data))) {
                      FAIL("samplerate returned error: %s\n",
                           src_strerror(err));
                      ITERATOR_ESCAPE();
                      break;
                  }
                  out_buf_count_iteration += src_data.output_frames_gen;
                  //                  DEBUG("head: input_frames_used: %ld, output_frames_gen: %ld\n", 
                  //                  src_data.input_frames_used, src_data.output_frames_gen);
                  in_buf_count_iteration = 1;

                  /* Body. */
                  
                  for(marker = marker_list_next(shl->sr->markers[track],
                                                iter_frame_offset,
                                                MARKER_SLOPE);
                      marker && marker->frame_offset < iter_frame_offset + iter_read;
                      marker = marker_list_next(shl->sr->markers[track],
                                                iter_frame_offset + in_buf_count_iteration + 1,
                                                MARKER_SLOPE)) {
                      src_data.input_frames = marker->frame_offset - 
                          (iter_frame_offset + in_buf_count_iteration);
                      src_data.output_frames = out_buf_count - out_buf_count_iteration;
                      src_data.data_in = &float_frame_bits[in_buf_count_iteration];
                      src_data.data_out = &out_buffer[out_buf_count_iteration];
                      slope_value = marker->multiplier;
                      adjusted_factor = (slope_value < 0 ? 
                                         factor - ((factor * -slope_value) / 2) :
                                         factor + (factor * slope_value));
                      
                      src_data.src_ratio = 1 / adjusted_factor;
                      if((err = src_process(src_state, &src_data))) {
                          FAIL("samplerate returned error: %s\n",
                               src_strerror(err));
                          ITERATOR_ESCAPE();
                          break;
                      }
                      out_buf_count_iteration += src_data.output_frames_gen;
                      //                      DEBUG("body: input_frames_used: %ld, output_frames_gen: %ld\n", 
                      //                            src_data.input_frames_used, src_data.output_frames_gen);
                      in_buf_count_iteration = marker->frame_offset - iter_frame_offset;
                  }
                  
                  if(iter_escape) 
                      break;
                  
                  /* Tail. */
                  
                  slope_value = marker_list_slope_value(shl->sr->markers[track],
                                                        iter_frame_offset + iter_read,
                                                        MARKER_SLOPE);
                  src_data.input_frames = iter_read - in_buf_count_iteration;
                  src_data.output_frames = out_buf_count - out_buf_count_iteration;
                  src_data.data_in = &float_frame_bits[in_buf_count_iteration];
                  src_data.data_out = &out_buffer[out_buf_count_iteration];
                  adjusted_factor = (slope_value < 0 ? 
                                     factor - ((factor * -slope_value) / 2) :
                                     factor + (factor * slope_value));
                  src_data.src_ratio = 1 / adjusted_factor;
                  if((err = src_process(src_state, &src_data))) {
                      FAIL("samplerate returned error: %s\n",
                           src_strerror(err));
                      ITERATOR_ESCAPE();
                      break;
                  }
                  out_buf_count_iteration += src_data.output_frames_gen;
                  //                  DEBUG("tail: input_frames_used: %ld, output_frames_gen: %ld\n", 
                  //                        src_data.input_frames_used, src_data.output_frames_gen);
                  in_buf_count_iteration = marker->frame_offset - iter_frame_offset;
              }
              blocklist_blocks_destroy(track_delete(shl->sr->tracks[track],
                                                    iter_frame_offset,
                                                    iter_read));
              
              if(guarantee_new_frame_count && 
                 total_frames_written + out_buf_count_iteration > new_frame_count) {
                  DEBUG("clamping, %ld frames lost\n", 
                        (total_frames_written + out_buf_count_iteration) - new_frame_count);
                  out_buf_count_iteration -= 
                      (total_frames_written + out_buf_count_iteration) - new_frame_count;
              }
              track_float_frames_insert(shl->sr->tracks[track],
                                        out_buffer,
                                        iter_frame_offset,
                                        out_buf_count_iteration);
             iter_written = out_buf_count_iteration;
             total_frames_written += iter_written);
    
    if(guarantee_new_frame_count && total_frames_written < new_frame_count) {
        DEBUG("padding, %ld frames inserted\n",
              new_frame_count - total_frames_written);
        track_null_insert(shl->sr->tracks[track],
                          iter_frame_offset,
                          new_frame_count - total_frames_written);
        total_frames_written += new_frame_count - total_frames_written;
    }
    src_delete(src_state);
    free(out_buffer);
    ITERATORF_EXIT();
    return total_frames_written;
}

#endif

void
reverse(shell *shl,
        int track,
        AFframecount start_offset,
        AFframecount end_offset) {
    int i;
    int32_t f;
    AFframecount insert_offset = start_offset;
    ITERATOR_INIT(start_offset, end_offset - start_offset);
    ITERATOR(shl, shl->sr->tracks[track], 
             for(i = 0; i < (iter_read / 2); i++) {
                 f = int32_frame_bits[iter_read - (i + 1)];
                 int32_frame_bits[iter_read - (i + 1)] = int32_frame_bits[i];
                 int32_frame_bits[i] = f;
             }
             blocklist_blocks_destroy(track_delete(shl->sr->tracks[track],
                                                   iter_frame_offset,
                                                   iter_read));
             
             track_int32_frames_insert(shl->sr->tracks[track],
                                       int32_frame_bits,
                                       insert_offset,
                                       iter_read));
    ITERATOR_EXIT();
}

void
amplify(shell *shl,
        int track,
        AFframecount start_offset,
        AFframecount end_offset,
        double factor,
        double slope) {
    int i;
    ITERATORF_INIT(start_offset, end_offset - start_offset);
    ITERATORF(shl, shl->sr->tracks[track],
              for(i = 0; i < iter_read; i++) {
                      float_frame_bits[i] = float_frame_bits[i] * 
                      (factor + 
                       (factor * marker_list_slope_value(shl->sr->markers[track],
                                                         iter_frame_offset + i,
                                                         MARKER_SLOPE)) - 
                       (slope * ((float)iter_frames_processed + i)));
              }
              track_float_frames_replace(shl->sr->tracks[track],
                                         float_frame_bits,
                                         iter_frame_offset,
                                         iter_read));
    ITERATORF_EXIT();
}

int32_t
find_peak(shell *shl,
          int track,
          AFframecount start_offset,
          AFframecount end_offset) {
    int i;
    int32_t peak = 1, val = 0;
    ITERATOR_INIT(start_offset, end_offset - start_offset);
    ITERATOR(shl, shl->sr->tracks[track],
             for(i = 0; i < iter_read; i++) {
                 val = ABS(int32_frame_bits[i]);
                 if(val > peak)
                     peak = val;
             });
    ITERATOR_EXIT();
    return peak;
}

action_group *
fade(shell *shl, 
     int undo,
     double factor,
     double slope) {
    int t, map = shl->select_channel_map;
    AFframecount start = shl->select_start, end = shl->select_end;
    action_group *undo_ag = NULL;
    
    DEBUG("locking shl->sr->rwl\n");
    rwlock_rlock(&shl->sr->rwl);
    
    DEBUG("creating undo\n");
    if(undo) 
        undo_ag = action_group_undo_create(shl,
                                           map, 
                                           start,
                                           end - start,
                                           start,
                                           end - start);
    
    DEBUG("slope: %f\n", slope);
    DEBUG("amplifying...\n");
    
    for(t = 0; t < snd_track_count(shl->sr); t++) {
        if((1 << t) & map) {
            DEBUG("fading track %d\n", t);
            amplify(shl,
                    t,
                    start,
                    end,
                    factor,
                    slope);
        }
    }
    rwlock_runlock(&shl->sr->rwl);

    return undo_ag;
}

AFframecount
find_zero(shell *shl,
          int track,
          AFframecount start_offset,
          AFframecount end_offset) {
    AFframecount r = 0, abs_offset = start_offset, frame_count;
    int i, cur_sign = 0, prev_sign = -1, delta = 1, count_down = 0;
    frame_bits_t frame_bits = mem_alloc(EFFECT_BUF_SIZE * 4);

    if(!frame_bits)
        return start_offset;

    if(start_offset > end_offset)
        count_down = 1;

    frame_count = ABS(end_offset - start_offset);

    if(count_down) {
        delta = -1;
        start_offset = MAX(0, start_offset - MIN(EFFECT_BUF_SIZE, frame_count));
    }

    do {
        r = track_int32_frames_get(shl->sr->tracks[track],
                                   frame_bits,
                                   start_offset,
                                   MIN(EFFECT_BUF_SIZE, frame_count));
        if(r <= 0)
            break;
        
        for(i = count_down ? r - 1 : 0; 
            count_down ? i >= 0 : i < r; 
            i += delta, abs_offset += delta) {

            cur_sign = (((int32_t *)frame_bits)[i] > 0 ? 1 : 0);
            
            /* First iteration, fix start point. */
            
            if(prev_sign == -1)
                prev_sign = cur_sign;

            /* Sign change. */
            
            if(cur_sign != prev_sign && prev_sign != -1) {
                DEBUG("found nul: abs_offset: %ld\n", abs_offset);
                mem_free(frame_bits);
                return abs_offset;
            }
            prev_sign = cur_sign;
        }

        start_offset += (delta * r);
        frame_count -= r;
        
    } while(r > 0);

    mem_free(frame_bits);
    return start_offset;
}

