/*
 * Copyright (C) 1997, 1998 Free Software Foundation
 * Copyright (C) 1994, 1995 Eric M. Ludlam
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 *
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 *
 * Please send bug reports, etc. to zappo@gnu.org
 * 
 * Purpose:
 *   This file contains the functions used in managing the list of
 * hosts used at run time.  Hosts are managed to remember the talk
 * daemon type, and addresses of thier talk daemons.  Eventually,
 * place some in/out routines for handling a .gtalk_hostsrc type file. 
 *
 * $Log: gtl_host.c,v $
 * Revision 1.17  1998/09/22 13:33:03  zappo
 * Update printing of the Internet address to display tuples, not a
 * straight long.
 *
 * Revision 1.16  1998/01/04 13:34:11  zappo
 * Fixed warnings
 *
 * Revision 1.15  1997/12/14 19:21:17  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.14  1997/12/02 22:56:30  zappo
 * If local host does not resolve, then use loopback.
 *
 * Revision 1.13  1997/10/18 03:06:48  zappo
 * Changed to deal w/ new verified flag.
 * Recycle routine now compares the internet address.
 * Name is grabbed from the longest alias.
 *
 * Revision 1.12  1997/07/23 01:22:26  zappo
 * Changed to use the list library.
 *
 * Revision 1.11  1997/02/21 04:05:19  zappo
 * Added in new feature when getting a host from an addr struct.  If the
 * family is a UNIX socket, then return the local host.
 *
 * Revision 1.10  1995/12/10 03:42:52  zappo
 * Changed all error messages to use DISP_message
 *
 * Revision 1.9  1995/09/20  23:30:51  zappo
 * Added a function to return the number of hosts loaded in
 *
 * Revision 1.8  1995/04/08  20:13:16  zappo
 * Default new systems to GTALKD.  This may irritate some, but will
 * always let you get the most out of a system.
 *
 * Revision 1.7  1995/04/08  20:04:13  zappo
 * Replaced malloc/strcpy with strdup
 *
 * Revision 1.6  1995/03/30  02:32:45  zappo
 * Modified to use utility case insensitive string compare
 *
 * Revision 1.5  1995/03/25  04:13:27  zappo
 * Updated copyright
 *
 * Revision 1.4  1995/03/12  20:29:33  zappo
 * Added an option to always lowercase the text name of a host so strcmp
 * works better.
 *
 * Revision 1.3  1995/03/04  14:46:58  zappo
 * Added use of syslog to report errors when linked to daemon.
 *
 * Revision 1.2  1994/11/16  23:15:45  zappo
 * Added check for malloc failure
 *
 * Revision 1.1  1994/08/29  23:42:51  zappo
 * Initial revision
 *
 * ::Header:: gtalklib.h
 */

#include "gtalklib.h"

extern int h_errno;

static MakeList(list);

/*
 * Print h_errno in human readable fassion...
 */
static void print_hosterr(machine)
     char *machine;
{
  char buff[150];
  switch(h_errno) {
  case HOST_NOT_FOUND:
    sprintf(buff, "Host %s unknown", machine);
    break;
  case NO_ADDRESS:
    sprintf(buff, "Host %s exists but has no address.", machine);
    break;
  case NO_RECOVERY:
    sprintf(buff, "Name server error while looking for %s", machine);
    break;
  case TRY_AGAIN:
    sprintf(buff, "Name server error looking for %s, try again later.", machine);
    break;
  default:
    sprintf(buff, "Unknown error looking for %s", machine);
    }
  DISP_message(NULL, buff, LOG_ERR);
}

/*
 * Function: append_host
 *
 * Take the host structure and tac it onto the end of the local
 * static list.  Be sure to recycle duplicate definitions of hosts
 * 
 * Parameters: host - the host to append
 *
 * History:
 * eml 3/1/94
 */
static int host_compare(host, sin_addr)
     struct hostent *host;
     char *sin_addr;
{
  int i;

  for(i = 0; i < host->h_length; i++)
    if(host->h_addr[i] != sin_addr[i])
      return 1;

  return 0;
}

static struct HostObject *append_host(host)
     struct hostent *host;
{
  struct HostObject *new;
  const char *name_to_show;
  int i;

  /*
   * first, quick search to see if we know him or not (case insensitive)
   */
  new = FirstElement(list);
  while(new && (host_compare(host, (char *)&new->addr.sin_addr) != 0))
    new = NextElement(new);
  
  if(new) {
    if(verbose)
      printf("Recycling host %s\n", new->name);
    return new;
  }

  new = (struct HostObject *)LIST_alloc(&list, sizeof(struct HostObject));
  
  if(!new)
    {
      DISP_message(NULL, "append_host: Error allocating new host struct.",
		   LOG_ERR);
      return NULL;
    }

#if OTALK_ONLY == 1
  new->type     = OTALKD;
#else
  new->type     = GTALKD;	/* assume this first */
#endif
  new->rc       = RC_new;	/* assume a new host */

  new->verified = 0;

  for(i=0, name_to_show = host->h_name; host->h_aliases[i]; i++) {
    /* Some systems use the short name as the official name.
     * I'm not sure the best way to do this, but I'm going to guess
     * that the best way to find the long name is to take the longest
     * string that is available.
     * Try to get a better way next time the dialup isn't busy.
     */
    if(strlen(name_to_show) < strlen(host->h_aliases[i]))
      name_to_show = host->h_aliases[i];
  }
  new->name     = strdup(name_to_show);
  new->addr.sin_family = AF_INET;
  new->addr.sin_port = 0;
  bcopy(host->h_addr, (char *)&new->addr.sin_addr, host->h_length);
  new->addr_len = host->h_length;

  return new;
}

/*
 * Function: HOST_gen_host
 *
 * Create a host object based of some machine name.  Important part here
 * is the address field that is filled in, and keeping track of OTALK hosts.
 * 
 * Parameters: machine - character name of machine
 *
 * History:
 * eml 3/1/94
 */
struct HostObject *HOST_gen_host( machine )
     char *machine ;
{
  struct hostent *host;

  host = gethostbyname( machine );

  if(!host) 
    {
      print_hosterr(machine);
      return NULL;
    }

  return append_host(host);
}     

/*
 * Function: HOST_gen_host_by_addr
 *
 * Create a host object based of some machine name.  Important part here
 * is the address field that is filled in, and keeping track of OTALK hosts.
 * 
 * Parameters: machine - character name of machine
 *             size    - size of the address
 *
 * History:
 * eml 3/1/94
 */
struct HostObject *HOST_gen_host_by_addr(addr, size)
     struct sockaddr *addr;
     int size ;
{
  struct hostent *host;
  struct in_addr  addr_in;

  if(addr->sa_family == AF_UNIX)
    {
      /* Unix native sockets are sometimes used instead of TCP sockets
       * to speed things up.  These are always local.
       */
      return HOST_gen_local_host();
    }
  else
    {

      addr_in = ((struct sockaddr_in *)addr)->sin_addr;

      /* Potential bug in linux:
       *  Technically, addr_in is always 4 bytes (long) however,
       * for expandability, INET makes their fns pass it about.  The size
       * on linux  0.99 didn't give back the right number, and as a result,
       * this call always failed until the constant was put in.
       */
      host = gethostbyaddr( (const char *)&addr_in, sizeof(addr_in), AF_INET );

      if(!host) 
	{
	  DISP_message(NULL, 
		       "HOST_gen_host_by_addr: sockaddr struct lookup failed.",
		       LOG_ERR);
	  return NULL;
	}

      return append_host(host);
    }
}     

/*
 * Function: HOST_gen_local_host
 *
 * Special function to return a host struct of the machine we are running
 * on.  Use this with short circut to access information about ourselves.
 * 
 * Parameters:
 *
 * History:
 * eml 3/1/94
 */

struct HostObject *HOST_gen_local_host()
{
  static struct HostObject *new = NULL;
  char                      buffer[100];
  int                       result;

  if(!new)
    {
      result = gethostname(buffer, 100);

      if(result < 0) {
	DISP_message(NULL, "GT_gen_local_host: gethostbyname failed.",
		     LOG_CRIT);
	return 0;
      }
      new = HOST_gen_host(buffer);

      if(new == NULL) {
	/* Ooops!  For somereason the system administrator failed to
	 * setup this machine correctly.  Lets use loopback.
	 */
	new = HOST_gen_host("127.0.0.1");
      }

    }
  return new;
}

/*
 * Function: HOST_print, HOST_print_node
 *
 * Print a formatted list of all hosts currently in our table.
 * 
 * Parameters:
 *
 * History:
 * eml 4/21/94
 */
static void HOST_print_node(host, data)
     struct HostObject *host;
     void *data;
{
  static char *types[] = { "None", "OTALK", "NTALK", "GTALK" };
  static char *rcf[] = { "System", "Local", "New", "Change" };
  char   buff[180];
  unsigned char *tuple = (unsigned char *)&host->addr.sin_addr.s_addr;

  sprintf(buff, "%-23.23s\t%s%c\t%s\tF(%d) P(%d) IN(%d.%d.%d.%d)",
	  host->name?host->name:"Unknown",
	  types[host->type+1],
	  host->verified?'*':' ',
	  rcf[host->rc],
	  host->addr.sin_family,
	  ntohs(host->addr.sin_port),
	  tuple[0], tuple[1], tuple[2], tuple[3]);
  DISP_message(NULL, buff, LOG_DEBUG);
}

void HOST_print()
{
  DISP_message(NULL, "Name\t\t\tType(*)\tRC\tAddress", LOG_DEBUG);

  LIST_map(&list, HOST_print_node, NULL);
}

/*
 * Function: HOST_number
 *
 *   returns the number of hosts currently in the host table
 *
 * Returns:     int  - 
 * Parameters:  None
 *
 * History:
 * zappo   8/25/95    Created
 */
int HOST_number()
{
  return LIST_map(&list, NULL, NULL);
}
