/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
//
// -proc.c
//
// specialized markup processors
//
// Copyright (c) 1995-96 Jim Nelson.  Permission to distribute
// granted by the author.  No warranties are made on the fitness of this
// source code.
//
*/

#include "defs.h"
#include "macro.h"

#include "htp-files.h"
#include "def-proc.h"
#include "file-proc.h"

#ifdef HAVE_PIPE
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#endif

uint ExternalFileProcessor(TASK *task, HTML_MARKUP *htmlMarkup,
    const char *externalName, char **newPlaintext)
{
    struct stat fileStat;
    struct tm *fileTime;
    uint precision;
    const char *attribValue;
    const char *precisionString;
    const char *value;

    assert(externalName != NULL);
    assert(htmlMarkup != NULL);

    /* get information on the file itself */
    if(stat(externalName, &fileStat) != 0)
    {
        HtpMsg(MSG_ERROR, task->infile, "unable to retrieve file information on \"%s\"",
            externalName);
        return MARKUP_ERROR;
    }

    /* get the precision attribute value, if present */
    /* (this is only valid for SIZE attribute, but not checking for simplicity */
    /* ignored for other types of FILE attributes) */
    precision = DEFAULT_PRECISION;
    if(IsAttributeInMarkup(htmlMarkup, "PRECISION"))
    {
        precisionString = MarkupAttributeValue(htmlMarkup, "PRECISION");
        if(precisionString != NULL)
        {
            precision = atoi(precisionString);
        }
        else
        {
            HtpMsg(MSG_WARNING, task->infile, "precision attribute needs a value");
        }
    }

    /* allocate room for the replacment plaintext */
    if((*newPlaintext = AllocMemory(MAX_TIME_DATE_SIZE)) == NULL)
    {
        HtpMsg(MSG_ERROR, task->infile, "unable to allocate memory for expansion");
        return MARKUP_ERROR;
    }

    /* create new plaintext depending on what extra information is specified */
    /* !! this is technically not correct ... SIZE, TIME, and DATE are */
    /* not allowed in the same markup but not checking that only one is */
    /* present and not any others */
    if(IsAttributeInMarkup(htmlMarkup, "SIZE"))
    {
        attribValue = MarkupAttributeValue(htmlMarkup, "SIZE");

        /* expand markup depending on how SIZE should be represented */
        if((attribValue == NULL) || (stricmp(attribValue, "BYTE") == 0))
        {
            /* byte representation is default */
            sprintf(*newPlaintext, "%lu", fileStat.st_size);
        }
        else if(stricmp(attribValue, "KBYTE") == 0)
        {
            sprintf(*newPlaintext, "%.*f", (int) precision,
                (double) ((double) fileStat.st_size / (double) KBYTE));
        }
        else if(stricmp(attribValue, "MBYTE") == 0)
        {
            sprintf(*newPlaintext, "%.*f", (int) precision,
                (double) ((double) fileStat.st_size / (double) MBYTE));
        }
        else if(stricmp(attribValue, "GBYTE") == 0)
        {
            sprintf(*newPlaintext, "%.*f", (int) precision,
                (double) ((double) fileStat.st_size / (double) GBYTE));
        }
        else
        {
            /* free the plaintext memory before returning */
            HtpMsg(MSG_ERROR, task->infile, "unknown SIZE specifier");
            FreeMemory(*newPlaintext);
            *newPlaintext = NULL;
            return MARKUP_ERROR;
        }
    }
    else if(IsAttributeInMarkup(htmlMarkup, "TIME"))
    {
        const char *value;

        /* convert into an ANSI time structure */
        fileTime = localtime(&fileStat.st_mtime);

        /* see if the attribute has a value ... if so, let it be the */
        /* strftime() formatter */
        if((value = MarkupAttributeValue(htmlMarkup, "TIME")) != NULL)
        {
            strftime(*newPlaintext, MAX_TIME_DATE_SIZE, value, fileTime);
        }
        else
        {
            strftime(*newPlaintext, MAX_TIME_DATE_SIZE, "%I:%M:%S %p", fileTime);
        }
    }
    else if(IsAttributeInMarkup(htmlMarkup, "DATE"))
    {
        /* convert into an ANSI time structure */
        fileTime = localtime(&fileStat.st_mtime);

        /* see if the attribute has a value ... if so, let it be the */
        /* strftime() formatter */
        if((value = MarkupAttributeValue(htmlMarkup, "DATE")) != NULL)
        {
            strftime(*newPlaintext, MAX_TIME_DATE_SIZE, value, fileTime);
        }
        else
        {
            strftime(*newPlaintext, MAX_TIME_DATE_SIZE, "%a %b %d, %Y", fileTime);
        }
    }
    else
    {
        /* free the plaintext, unused */
        FreeMemory(*newPlaintext);
        *newPlaintext = NULL;

        HtpMsg(MSG_ERROR, task->infile, "bad file information specifier");
        return MARKUP_ERROR;
    }

    /* the new plaintext was created successfully */
    return NEW_MARKUP;
}

uint FileExecuteProcessor(TASK *task, HTML_MARKUP *htmlMarkup)
{
    const char *cmdline;
    const char *output;
#ifndef HAVE_PIPE
    char newCmdline[MAX_CMDLINE_LEN];
    char tempFilename[MAX_PATHNAME_LEN];
#endif
    BOOL redirect;
    STREAM infile;
    TASK newTask;
    BOOL result = TRUE;
    uint exitCode;

    assert(task != NULL);
    assert(htmlMarkup != NULL);

    if((cmdline = MarkupAttributeValue(htmlMarkup, "EXECUTE")) == NULL)
    {
        HtpMsg(MSG_ERROR, task->infile,"FILE EXECUTE must specify a command-line");
        return MARKUP_ERROR;
    }

    output = MarkupAttributeValue(htmlMarkup, "OUTPUT");
    redirect = IsAttributeInMarkup(htmlMarkup, "REDIRECT");

    /* either output or redirect, but not both, must be specified */
    if((output == NULL) && (redirect == FALSE))
    {
        HtpMsg(MSG_ERROR, task->infile, "Either REDIRECT or OUTPUT must be specified for FILE EXECUTE");
        return MARKUP_ERROR;
    }

    if((output != NULL) && (redirect == TRUE))
    {
        HtpMsg(MSG_ERROR, task->infile, "REDIRECT and OUTPUT cannot both be specified for FILE EXECUTE");
        return MARKUP_ERROR;
    }

    HtpMsg(MSG_INFO, task->infile, "Executing command \"%s\" ...", cmdline);

#ifndef HAVE_PIPE
    /* if redirection required, append to the command-line a redirector to */
    /* a temporary file */
    if(redirect)
    {
        if(CreateTempFilename(tempFilename, MAX_PATHNAME_LEN) == FALSE)
        {
            HtpMsg(MSG_ERROR, task->infile, "unable to create a temporary file for redirection");
            return MARKUP_ERROR;
        }

        strncat(newCmdline, " > ", MAX_PATHNAME_LEN);
        strncat(newCmdline, tempFilename, MAX_PATHNAME_LEN);
        cmdline = newCmdline;
        output = tempFilename;
    }

    /* execute the command */
    exitCode = system(cmdline);

#else
    /* if redirection required, create a pipe for our child process */
    if(redirect) {
        pid_t pid;
        int pipefds[2];
        extern char **environ;

        pipe(pipefds);
        pid = fork();
        if (pid == 0) {
            char *argv[4];

            if (redirect) {
                dup2(pipefds[1], 1);
                close(pipefds[0]);
                close(pipefds[1]);
            }
            
            argv[0] = "sh";
            argv[1] = "-c";
            argv[2] = (char*)cmdline;
            argv[3] = NULL;
            execve("/bin/sh", argv, environ);
            exit(127);
        } else
            close(pipefds[1]);

        if (pid == -1) 
            exitCode = -1;
        else {
            if (CreateFDReader(&infile, "stdout", pipefds[0]) == FALSE)
            {
                HtpMsg(MSG_ERROR, task->infile, "unable to open execute result file");
                return MARKUP_ERROR;
            }
            
            /* build a new task */
            newTask.infile = &infile;
            newTask.outfile = task->outfile;
            newTask.varstore = task->varstore;
            newTask.sourceFilename = task->sourceFilename;
            
            /* process the file */
            result = ProcessTask(&newTask);

            if (!result) {
                /* Error message was already spitted out.  However, we
                 * should give a hint where the meta-tag was called.  
                 */
                HtpMsg(MSG_ERROR, task->infile,
                       "... in output from '%s'", cmdline);
            }

            CloseStream(&infile);

            /* wait for termination of the command */
            while (waitpid(pid, &exitCode, 0) == -1) {
                if (errno != EINTR) {
                    exitCode = -1;
                    break;
                }
            }
        }
    } else {
        /* execute the command */
        exitCode = system(cmdline);
    }

#endif

    if(exitCode != 0)
    {
        if(IsAttributeInMarkup(htmlMarkup, "NOERROR") == FALSE)
        {
            /* the program has exited with an error condition */
            HtpMsg(MSG_ERROR, task->infile, "Command \"%s\" exited with an error code of %u",
                   cmdline, exitCode);
            
            /* remove the temporary file, in case it was partially created */
            if (output != NULL)
                remove(output);
            
            return MARKUP_ERROR;
        }
    }

    if (output != NULL)
    {
        /* include the output file like it was anything else */
        /* first, open it */
        if(CreateFileReader(&infile, output) == FALSE)
        {
            HtpMsg(MSG_ERROR, task->infile, "unable to open execute result file");
            return MARKUP_ERROR;
        }
        
        /* build a new task */
        newTask.infile = &infile;
        newTask.outfile = task->outfile;
        newTask.varstore = task->varstore;
        newTask.sourceFilename = task->sourceFilename;
        
        /* process the file */
        result = ProcessTask(&newTask);

        if (!result) {
            /* Error message was already spitted out.  However, we
             * should give a hint where the meta-tag was called.  
             */
            HtpMsg(MSG_ERROR, task->infile,
                   "... in output from '%s'", cmdline);
        }

        /* close and destroy the output file */
        CloseStream(&infile);
        remove(output);
    }
    return (result == TRUE) ? DISCARD_MARKUP : MARKUP_ERROR;
}

uint FileTemplateProcessor(TASK *task, HTML_MARKUP *htmlMarkup)
{
    const char *templateFile;

    assert(task != NULL);
    assert(htmlMarkup != NULL);

    if((templateFile = MarkupAttributeValue(htmlMarkup, "TEMPLATE")) == NULL)
    {
        HtpMsg(MSG_ERROR, task->infile, "a template file must be specified");
        return MARKUP_ERROR;
    }

    /* the template file is not actually processed now, but rather when the */
    /* rest of the file is completed ... to postpone processing, the template */
    /* name is kept in the variable store under a special name and retrieved */
    /* later */
    if(StoreVariable(task->varstore, VAR_TEMPLATE_NAME, 
                     DuplicateString(templateFile),
                     VAR_TYPE_INTERNAL, VAR_FLAG_DEALLOC_VALUE, 
                     NULL, NULL) == FALSE)
    {
        HtpMsg(MSG_ERROR, task->infile,
            "unable to store template filename for post-processing (out of memory?)");
        return MARKUP_ERROR;
    }

    return DISCARD_MARKUP;
}

uint FileIncludeProcessor(TASK *task, HTML_MARKUP *htmlMarkup)
{
    STREAM incfile;
    BOOL result;
    TASK newTask;
    char fullPathname[MAX_PATHNAME_LEN];
    const char *attribValue;
    VARSTORE varstore;
    VARSTORE *topVarstore;
    uint ctr;
    uint flag;

    /* get the filename to include */
    attribValue = MarkupAttributeValue(htmlMarkup, "INCLUDE");
    if(attribValue == NULL)
    {
        HtpMsg(MSG_ERROR, task->infile, "include filename not specified");
        return MARKUP_ERROR;
    }

    /* open the include file as input */
    if(CreateFileReader(&incfile, attribValue) == FALSE)
    {
        /* use the search path to find the file */
        if(SearchForFile(attribValue, fullPathname, MAX_PATHNAME_LEN) == FALSE)
        {
            /* could not find the file in the search path either */
            HtpMsg(MSG_ERROR, task->infile, "unable to open include file \"%s\"",
                attribValue);
            return MARKUP_ERROR;
        }

        if(CreateFileReader(&incfile, fullPathname) == FALSE)
        {
            HtpMsg(MSG_ERROR, task->infile, "unable to open include file \"%s\"",
                fullPathname);
            return MARKUP_ERROR;
        }
    }
    else
    {
        StringCopy(fullPathname, attribValue, MAX_PATHNAME_LEN);
    }

    /* if additional parameters exist in the tag, build a local varstore */
    /* and push it onto the context */
    if(htmlMarkup->attribCount > 1)
    {
        if(InitializeVariableStore(&varstore) == FALSE)
        {
            HtpMsg(MSG_ERROR, task->infile, "unable to initialize context for include file");
            CloseStream(&incfile);
            return MARKUP_ERROR;
        }

        /* start including the parameters as local macros */
        for(ctr = 1; ctr < htmlMarkup->attribCount; ctr++)
        {
            flag = (htmlMarkup->attrib[ctr].quoted == TRUE) ? VAR_FLAG_QUOTED
                : VAR_FLAG_NONE;
            /* No need to duplicate values, as varstore is destroyed before
             * this function returns.
             */
            if(StoreVariable(&varstore, htmlMarkup->attrib[ctr].name,
                htmlMarkup->attrib[ctr].value, VAR_TYPE_SET_MACRO,
                flag, NULL, NULL) == FALSE)
            {
                HtpMsg(MSG_ERROR, task->infile, "unable to add variable to context for include file");
                CloseStream(&incfile);
                DestroyVariableStore(&varstore);
                return MARKUP_ERROR;
            }
        }

        /* push this onto the context and use it for the include file */
        PushVariableStoreContext(task->varstore, &varstore);
        topVarstore = &varstore;
    }
    else
    {
        topVarstore = task->varstore;
    }

    /* build a new task structure */
    newTask.infile = &incfile;
    newTask.outfile = task->outfile;
    newTask.varstore = topVarstore;
    newTask.sourceFilename = task->sourceFilename;

    /* informational message for the user */
    HtpMsg(MSG_INFO, task->infile, "including file \"%s\"", fullPathname);

    /* process the new input file */
    result = ProcessTask(&newTask);
    if (!result) {
        /* Error message was already spitted out.  However, we
         * should give a hint where the meta-tag was called.  
         */
        HtpMsg(MSG_ERROR, task->infile, "... in file included here");
    }

    /* pop the local context */
    if(topVarstore == &varstore)
    {
        assert(PeekVariableStoreContext(topVarstore) == topVarstore);
        PopVariableStoreContext(topVarstore);
        DestroyVariableStore(&varstore);
    }

    CloseStream(&incfile);

    /* if the new file did not process, return an error, otherwise discard */
    /* the markup */
    return (result == TRUE) ? DISCARD_MARKUP : MARKUP_ERROR;
}

uint FileProcessor(TASK *task, HTML_MARKUP *htmlMarkup, char **newPlaintext)
{
    const char *attribName;
    const char *attribValue;
    const char *externalName;
    struct tm *timeNow;
    struct stat fileStat;

    /* if a NAME attribute is found, and it contains a value, use the */
    /* ExternalFileProcessor to create the plaintext (this function only */
    /* reports output file's time, date, name) */
    if(IsAttributeInMarkup(htmlMarkup, "NAME"))
    {
        if((externalName = MarkupAttributeValue(htmlMarkup, "NAME")) != NULL)
        {
            return ExternalFileProcessor(task, htmlMarkup, externalName,
                newPlaintext);
        }
    }

    /* if NAME attribute not in markup, or no external filename specified, */
    /* only one attribute can be used: NAME, SIZE, TIME, DATE , or INCLUDE */
    /* (the exception being EXECUTE and TEMPLATE) */
    if(IsAttributeInMarkup(htmlMarkup, "EXECUTE") == TRUE)
    {
        return FileExecuteProcessor(task, htmlMarkup);
    }

    if(IsAttributeInMarkup(htmlMarkup, "TEMPLATE") == TRUE)
    {
        return FileTemplateProcessor(task, htmlMarkup);
    }

    if(IsAttributeInMarkup(htmlMarkup, "INCLUDE") == TRUE)
    {
        return FileIncludeProcessor(task, htmlMarkup);
    }

    if(htmlMarkup->attribCount != 1)
    {
        HtpMsg(MSG_ERROR, task->infile, "improper FILE syntax");
        return MARKUP_ERROR;
    }

    /* get the attribute */
    attribName = htmlMarkup->attrib[0].name;
    attribValue = htmlMarkup->attrib[0].value;

    /* act on the attribute */
    if(stricmp(attribName, "SEARCH") == 0)
    {
        /* set the include search path to what was specified */
        if(attribValue != NULL)
        {
            StringCopy(searchPath, attribValue, SEARCH_PATH_SIZE);
        }
        else
        {
            /* search path is cleared */
            searchPath[0] = NUL;
        }

        return DISCARD_MARKUP;
    }
    else
    {
        /* NAME, TIME, DATE or bad tag */
        /* first, allocate some space for the (possibly) new markup */
        if((*newPlaintext = AllocMemory(MAX_TIME_DATE_SIZE)) == NULL)
        {
            HtpMsg(MSG_ERROR, task->infile, "unable to allocate memory for expansion");
            return MARKUP_ERROR;
        }

        /* get the input files time, in case the attribute is TIME or DATE */
        if(stat(task->sourceFilename, &fileStat) != 0)
        {
            HtpMsg(MSG_ERROR, task->infile, 
                   "unable to get information for file \"%s\"\n",
                   task->infile->name);
            return MARKUP_ERROR;
        }

        timeNow = localtime(&fileStat.st_mtime);

        if(stricmp(attribName, "TIME") == 0)
        {
            if(attribValue != NULL)
            {
                strftime(*newPlaintext, MAX_TIME_DATE_SIZE, attribValue, timeNow);
            }
            else
            {
                strftime(*newPlaintext, MAX_TIME_DATE_SIZE, "%I:%M:%S %p", timeNow);
            }

            HtpMsg(MSG_INFO, task->outfile, "adding local time");
        }
        else if(stricmp(attribName, "DATE") == 0)
        {
            if(attribValue != NULL)
            {
                strftime(*newPlaintext, MAX_TIME_DATE_SIZE, attribValue, timeNow);
            }
            else
            {
                strftime(*newPlaintext, MAX_TIME_DATE_SIZE, "%a %b %d, %Y", timeNow);
            }

            HtpMsg(MSG_INFO, task->outfile, "adding local date");
        }
        else if(stricmp(attribName, "NAME") == 0)
        {
            StringCopy(*newPlaintext, task->outfile->name, MAX_TIME_DATE_SIZE);

            HtpMsg(MSG_INFO, task->outfile, "adding output filename");
        }
        else
        {
            /* no appropriate tags found */
            HtpMsg(MSG_ERROR, task->infile, "invalid FILE tag attribute \"%s\"",
                attribName);

            /* free the allocated plaintext buffer */
            FreeMemory(*newPlaintext);
            *newPlaintext = NULL;

            return MARKUP_ERROR;
        }
    }

    /* the new plaintext has been created */
    return NEW_MARKUP;
}   

uint OutputProcessor(TASK *task, HTML_MARKUP *htmlMarkup, char **newPlaintext)
{
    STREAM outputfile;
    BOOL result;
    const char *attribValue;
    BOOL append;

    UNREF_PARAM(newPlaintext);

    /* get the filename to include */
    attribValue = MarkupAttributeValue(htmlMarkup, "FILE");
    if(attribValue == NULL)
    {
        HtpMsg(MSG_ERROR, task->infile, "output filename not specified");
        return MARKUP_ERROR;
    }

    append = IsAttributeInMarkup(htmlMarkup, "APPEND");

    /* open the output file */
    if(CreateFileWriter(&outputfile, attribValue, append) == FALSE)
    {
        HtpMsg(MSG_ERROR, task->infile, "unable to open output file \"%s\"",
               attribValue);
        return MARKUP_ERROR;
    }

    result = ReadBlockToFile(task, htmlMarkup, &outputfile);
    CloseStream(&outputfile);

    return result == FALSE ? MARKUP_ERROR : DISCARD_MARKUP;
}
