#ifndef IAC_TYPES_H_
#define IAC_TYPES_H_

#include "config.h"

#include <limits.h>
#include <sys/types.h>
#include <time.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>

#include "util.h"

#define IAC_DNS_NONE 0
#define IAC_DNS_BUSY 1
#define IAC_DNS_DONE 2
#define IAC_DNS_FAIL 3

/*
 * client connection flags
 */
#define IAC_NICKSENT      (1<<0)         /* 00000001 nickname sent          */
#define IAC_USERSENT      (1<<1)         /* 00000010 username sent          */
#define IAC_OK            (1<<2)         /* 00000100 connection is OK       */
#define IAC_LOCAL         (1<<3)         /* 00001000 local connection       */
#define IAC_PONG          (1<<5)         /* 00100000 i sent a ping reply    */
#define IAC_TOBEREMOVED   (1<<6)         /* 01000000 to be removed mark     */
#define IAC_OPERATOR      (1<<7)         /* 10000000 operator               */

#define IAC_WALLOPS       (1<<8)         /* 00000001 wants wallop messages  */
#define IAC_FORTUNE       (1<<9)         /* 00000010 wants to have phun     */
#define IAC_INVISIBLE     (1<<10)        /* 00000100 user is invisible      */
#define IAC_DEBUGMSG      (1<<11)        /* 00001000 wants to see dbg msgs  */
#define IAC_KILLED        (1<<12)        /* 00010000 user was killed        */
#define IAC_SUICIDE       (1<<13)        /* 00100000 suicide                */
#define IAC_AWAY_INFORMS  (1<<14)        /* 01000000 inform about aways     */

/*
 * server connection flags
 */
#define IAC_PASSSENT      (1<<0)         /* 00000001 PASS line sent         */
#define IAC_AUTH          (1<<1)         /* 00000010 AUTH was sent          */
/* #define IAC_OK         (1<<2) */      /* 00000100 already defined above  */
#define IAC_AUTH_OK       (1<<3)         /* 00001000 authentication OK      */
#define IAC_MASTER        (1<<4)         /* 00010000 master (i connected)   */
/* #define IAC_PONG       (1<<5) */      /* 00100000 already defined above  */
/* #define IAC_TOBEREMOVED (1<<6) */     /* 01000000 to be removed mark     */
#define IAC_USEMSGID      (1<<7)         /* 10000000 Use message id system  */

#define IAC_COMPRESSED    (1<<8)         /* 00000001 Compressed link        */

/*
 * server line flags
 */
#define IAC_AUTOCONNECT   (1<<0)         /* 00000001 autoconnect on startup */
#define IAC_CONNECTED     (1<<1)         /* 00000010 link is connected      */
#define IAC_FAILED        (1<<2)         /* 00000100 last try failed        */
#define IAC_RETRY         (1<<3)         /* 00001000 retrying               */
/* #define IAC_MASTER        (1<<4) */   /* 00010000 master                 */
#define IAC_DONT_RETRY    (1<<5)         /* 00100000 do not retry           */
#define IAC_SLOWRETRY     (1<<6)         /* 01000000 slow retry             */


/*
 * server configuration
 */
#define IAC_DAEMON        (1<<0)         /* 00000001 run as daemon          */
#define IAC_REHASH        (1<<1)         /* 00000010 rehash configuration   */
#define IAC_PARANOIA      (1<<2)         /* 00000100 paranoia mode          */
#define IAC_LASTLOG       (1<<4)         /* 00010000 lastlog                */
#define IAC_TIMESTAMPS    (1<<5)         /* 00100000 use timestamps         */

/*
 * channel flags
 */
#define IAC_ENCRYPTED     (1<<0)         /* 00000001 channel is encrypted   */


#define IAC_GENHOSTMASK(src) { \
    if (src->hostmask) \
        RL_FREE(src->hostmask); \
	asprintf(&(src->hostmask), "%s!%s@%s", \
        (src->nick) ? src->nick : "(null)", \
        (src->user) ? src->user : "(null)",  \
        (src->host) ? src->host : "(null)"); \
}

typedef struct iac_channel_link_s
{
    struct iac_channel_s *link;
    struct iac_channel_link_s *next;

} iac_channel_link_t;

typedef struct iac_script_s
{
    char *nick;
    char *cmd;
    char *char_regex;

    struct iac_script_s *next;

} iac_script_t;

/*
 * client connection
 */
typedef struct iac_client_conn_s
{
    /*
     * socket stuff
     */
	int sock;                          /* socket handle                     */
#ifdef HAVE_IPV6
    struct sockaddr_in6 addr;
#else
	struct sockaddr_in addr;           /* socket address                    */
#endif

	socklen_t addrlen;                 /* socket adress length              */

    char *msg;                         /* current message being sent by the
                                          client                            */

    /*
     * a child process will look up the clients hostname async
     */
	pid_t child_pid;                   /* child pid of process              */
	int pipe[2];                       /* pipe used to communicate          */

#ifdef HAVE_SCRIPTS
    iac_script_t *script;              /* ptr to script being run           */
#endif

    /*
     * irc values
     */
	char *nick;                        /* nickname                          */
	char *user;                        /* username                          */
	char *host;                        /* host (ip or hostname)             */
    char *ip;                          /* ip address                        */
    int port;                          /* port                              */
    int hopcount;                      /* hop count                         */
	char *hostmask;                    /* hostmask nick!user@host           */
	char *ircname;                     /* real name                         */
    char *server;                      /* server being connected to         */
    char *server_info;                 /* server info of that server        */
    char *away;                        /* away message NULL if none         */
    char *quitmsg;                     /* quit message NULL if none         */

#ifdef IAC_AUTOAWAY
    short autoaway;                    /* auto away in seconds (0=disabled) */
#endif

#ifdef IAC_COUNT_IDLETIME
    time_t signon;                     /* signon time                       */
    time_t last_action;                /* timestamp of last action done     */
#endif

    iac_channel_link_t *channels;      /* list of channels joined           */

	int flags;                         /* flags                             */
	int lookup;                        /* lookup state                      */
	struct dns_resolver *resolver;
} iac_client_conn_t;

typedef struct iac_server_msg_s
{
    int numeric;
    int msgid;
    char *nick;
    char *msg;

} iac_server_msg_t;

/*
 * server connnection
 */
typedef struct iac_server_conn_s
{
    /*
     * socket stuff
     */
	int sock;                          /* socket handle                     */
	struct sockaddr_in addr;           /* socket address                    */

	socklen_t addrlen;                 /* socket address length             */
   
    char *msg;                         /* current message being sent by the
                                          other peer                        */
    iac_server_msg_t smsg;             /* current message (splitted up)     */
    char *omsg;                        /* copy of original message          */

    char *ip;                          /* ip address                        */
    char *hostname;                    /* hostname (not checked)            */
    int port;                          /* port                              */
    int numeric;                       /* server numeric (server token)     */
    char *info;                        /* info string (description)         */
    int  hops;                         /* hops to server (0 is local peer)  */
    char *password;                    /* password userd on this conn.      */

    unsigned int queued;               /* queued messages (no queue yet)    */
    unsigned int bytes_sent;           /* total bytes sent                  */
    unsigned int bytes_recvd;          /* total bytes received              */
    unsigned int msgs_sent;            /* total messages sent               */
    unsigned int msgs_recvd;           /* total messages sent               */
    time_t time_init;                  /* time the connection was opened    */

    int flags;                         /* flags                             */

    int pingtime;                      /* ping time                         */

} iac_server_conn_t;

typedef struct iac_user_link_s
{
    iac_client_conn_t *link;
    struct iac_user_link_s *next;

} iac_user_link_t;


/*
 * channel
 */
typedef struct iac_channel_s
{
	char *name;                        /* channel name                      */

    /*
     * topic stuff
     */
    char *topic[IAC_TLIST_MAX];        /* list of last topics               */
#ifndef IAC_RFC_STRICT
    char *topic_owner[IAC_TLIST_MAX];  /* owner of topic (same index)       */
    time_t topic_time[IAC_TLIST_MAX];  /* time the topic changed            */
#endif
    int   itopic;                      /* topic index, indexes current      */

	int flags;                         /* flags                             */
    char *owner;                       /* owner of channel                  */

    iac_user_link_t *users;            /* list of clients joined            */

} iac_channel_t;


/*
 * server (not a connection)
 */
typedef struct iac_server_s
{
    char *name;                        /* server name                       */
    char *info;                        /* server info                       */
    int numeric;                       /* server numeric                    */
    int hopcount;                      /* hopcount to this server           */

    int flags;                         /* flags                             */

    struct iac_server_s *next;         /* pointer to next server entry      */

} iac_server_t;


typedef struct iac_link_line_s
{
    int numeric;                       /* server numeric                    */
    char *host;                        /* hostname                          */
    char *ip;                          /* ip                                */
    char *addr;                        /* real address (dynamic)            */
    int port;                          /* port                              */
    char *password;                    /* password                          */
    char *options;                     /* server options                    */
    char *mask;                        /* allowed mask                      */
    int flags;                         /* flags                             */
    int fd;                            /* needed to link to server_conn_t   */

    struct iac_link_line_s *next;      /* pointer to next line              */

} iac_link_line_t;

typedef struct iac_operator_line_s
{
    char *nick;                        /* nick (must match)                 */
    char *mask;                        /* allowed mask                      */
    char *password;                    /* password (probably encrypted)     */

    struct iac_operator_line_s *next;  /* pointer to next operator line     */

} iac_operator_line_t;

typedef struct iac_confpair_s
{
    char *key;
    char *data;

    struct iac_confpair_s *next;

} iac_confpair_t;


/*
 * gotta block some suckers
 */
typedef struct iac_block_s
{
    char *mask;                        /* mask to block (e.g. *!*@62.2.*.*) */
    char *reason;                      /* reason                            */
    unsigned int matches;              /* nr of matches                     */

    struct iac_block_s *next;          /* pointer to next block item        */

} iac_block_t;


typedef struct iac_lastlog_s
{
    char *nick;
    char *username;
    char *hostname;
    char *ip;
    int port;
    char *ircname;
    int flags;
    time_t signon;

    struct iac_lastlog_s *next;
    struct iac_lastlog_s *prev;

} iac_lastlog_t;


typedef struct iac_server_configuration_s
{
    int flags;                              /* flags                        */
    int debug;                              /* debug level                  */

    int client_port;                        /* client port                  */
    int server_port;                        /* server port                  */

    int numeric;                            /* server numeric               */
    char servername[IAC_HOSTLEN];           /* server name                  */
    char info[IAC_SERVERINFOLEN];           /* server info                  */

    char logprefix[FILENAME_MAX];           /* logfile prefix               */
    char configfile[FILENAME_MAX];          /* path to configifile          */

    char user[256];                         /* username to change to        */
    char group[256];                        /* group to change to           */

    char chroot[FILENAME_MAX];              /* chroot path                  */

    char motdfile[FILENAME_MAX];            /* path to MOTD file            */
    char interface[256];                    /* interface to bind on         */
    char pid_file[FILENAME_MAX];            /* pid file                     */

    iac_lastlog_t *lastlog;                 /* head lastlog                 */
    iac_lastlog_t *lastlog_tail;            /* tail of lastlog              */
    int lastlog_entries;                    /* nr of lastlog entries        */

    iac_block_t *blocked;                   /* list of blocked masks        */
    iac_operator_line_t *operators;         /* list of allowed operators    */
    iac_link_line_t *links;                 /* list of allowed links        */
    rl_htable_t *channels;                  /* list of allowed channels     */
#ifdef HAVE_SCRIPTS
    iac_script_t *scripts;                  /* list of scripts              */
#endif
    
} iac_server_configuration_t;

typedef struct iac_server_status_s
{
    unsigned int nr_channels;               /* nr of channels created       */
    unsigned int local_connections;         /* current local users          */
    unsigned int total_connections;         /* current users in the network */
    unsigned int server_connections;        /* nr of server connections     */
    unsigned int waiting;                   /* clients waiting not IAC_OK   */

    iac_server_t *servers;                  /* list of servers              */

} iac_server_status_t;

typedef struct iac_cmd_usage_s
{
    unsigned int nick;
    unsigned int user;
    unsigned int privmsg;
    unsigned int notice;         /* will be 0, because NOTICE counts as
                                    privmsg for now */
    unsigned int join;
    unsigned int part;
    unsigned int topic;
    unsigned int mode;
    unsigned int quit;
    unsigned int whois;
    unsigned int whowas;
    unsigned int who;
    unsigned int names;
    unsigned int kill;
    unsigned int die;
    unsigned int rehash;
    unsigned int ison;
    unsigned int userhost;
    unsigned int away;
    unsigned int admin;
    unsigned int list;
    unsigned int users;
    unsigned int summon;
    unsigned int version;
    unsigned int ping;
    unsigned int oper;
    unsigned int wallops;
    unsigned int squit;
    unsigned int connect;
    unsigned int sync;
    unsigned int block;
    unsigned int unblock;
    unsigned int dump;
    unsigned int stats;
    unsigned int motd;
    unsigned int links;
    unsigned int set;
    unsigned int thistory;
    unsigned int unknown;

} iac_cmd_usage_t;



#endif
