<?php

require_once 'Horde/MIME/Viewer/images.php';

/**
 * The IMP_MIME_Viewer_images class allows images to be displayed
 * inline in a message.
 *
 * $Horde: imp/lib/MIME/Viewer/images.php,v 1.43.10.2 2005/01/18 23:32:22 jan Exp $
 *
 * Copyright 2002-2005 Michael Slusarz <slusarz@bigworm.colorado.edu>
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/gpl.html.
 *
 * @author  Michael Slusarz <slusarz@bigworm.colorado.edu>
 * @version $Revision: 1.43.10.2 $
 * @since   IMP 3.2
 * @package Horde_MIME_Viewer
 */
class IMP_MIME_Viewer_images extends MIME_Viewer_images {

    /**
     * The content-type of the generated data.
     *
     * @var string $_contentType
     */
    var $_contentType;

    /**
     * Render out the currently set contents.
     *
     * @access public
     *
     * @param array $params  An array with a reference to a MIME_Contents
     *                       object.
     *
     * @return string  The rendered information.
     */
    function render($params)
    {
        $contents = $params[0];

        global $browser;

        if ($contents->viewAsAttachment()) {
            if (Util::getFormData('images_view_convert')) {
                /* Convert the image to browser-viewable format and
                 * display. */
                return $this->_viewConvert();
            } elseif (Util::getFormData('images_view_thumbnail')) {
                /* Create the thumbnail and display. */
                return $this->_viewConvert(true);
            } elseif (!$browser->hasFeature('javascript') ||
                      !$contents->viewAsAttachment(true) ||
                      Util::getFormData('img_data')) {
                /* If either:
                   + The browser doesn't support javascript
                   + We are not viewing in a popup window
                   + The calling URL wants to force us to output straight
                     data ('img_data' parameter is set)
                   Then we simply render the image data. */
                return parent::render();
            } elseif (Util::getFormData('view_thumbnail') ||
                      Util::getFormData('images_load_convert') ||
                      $browser->isViewable(parent::getType())) {
                /* If either:
                   + This page has been called with the thumbnail/convert
                     parameter
                   + The browser can display the image type directly
                   Than we output the javascript code to render the auto
                   resize popup image window. */
                return $this->_popupImageWindow();
            }
        }

        if ($browser->isViewable($this->mime_part->getType())) {
            /* If we are viewing inline, and the browser can handle the image
               type directly, output an <img> tag to load the image. */
            $alt = $this->mime_part->getName(false, true);
            return Horde::img($contents->urlView($this->mime_part, 'view_attach'), $alt, null, '');
        } else {
            /* If we have made it this far, than the browser can not view this
               image inline.  Inform the user of this and, possibly, ask user
               if we should convert to another image type. */
            $msg = _("Your browser does not support inline display of this image type.");

            /* See if we can convert to an inline browser viewable form. */
            $img = $this->_getHordeImageOb();
            if ($img && $browser->isViewable($img->getContentType())) {
                if ($contents->viewAsAttachment()) {
                    $selfurl = Horde::selfUrl(true);
                    $msg .= '<br />' . sprintf(_("Click %s to download the image."), $contents->linkView($this->mime_part, 'download_attach', _("HERE"), array('viewparams' => array('img_data' => 1)), true));
                    $convert_link = Horde::link($contents->urlView($this->mime_part, 'view_attach', array('images_load_convert' => 1, 'popup_view' => 1))) . _("HERE") . '</a>';
                } else {
                    $convert_link = $contents->linkViewJS($this->mime_part, 'view_attach', _("HERE"), null, null, array('images_load_convert' => 1));
                }
                $msg .= '<br />' . sprintf(_("Click %s to convert the image file into a format your browser can view."), $convert_link);
            }

            return $contents->formatStatusMsg($msg);
        }
    }

    /**
     * Return the content-type
     *
     * @access public
     *
     * @return string  The content-type of the output.
     */
    function getType()
    {
        global $browser;

        if ($this->_contentType) {
            return $this->_contentType;
        } elseif (MIME_Contents::viewAsAttachment() &&
            (!$browser->hasFeature('javascript') ||
             !MIME_Contents::viewAsAttachment(true) ||
             Util::getFormData('img_data') ||
             Util::getFormData('images_view_thumbnail'))) {
            return parent::getType();
        }

        return 'text/html; charset=' . NLS::getCharset();
    }

    /**
     * Render out attachment information.
     *
     * @access public
     *
     * @param array $params  An array with a reference to a MIME_Contents
     *                       object.
     *
     * @return string  The rendered text in HTML.
     */
    function renderAttachmentInfo($params)
    {
        $contents = &$params[0];

        /* Display the thumbnail link only if size is greater than 50 KB and
           there is an image conversion utility available. */
        if (($this->mime_part->getBytes() > 51200) &&
            $this->_getHordeImageOb()) {
            $status = array(
                sprintf(_("A large image named %s is attached to this message."), $this->mime_part->getName(true)),
                sprintf(_("Click %s to view a thumbnail of this image."), $contents->linkViewJS($this->mime_part, 'view_attach', _("HERE"), _("View Thumbnail"), null, array('view_thumbnail' => 1)))
            );
            return $contents->formatStatusMsg($status, Horde::img('mime/image.png', _("View Thumbnail"), null, $GLOBALS['registry']->getImageDir('horde')));
        }

        return '';
    }

    /**
     * Generate the HTML output for the JS auto-resize view window.
     *
     * @access private
     *
     * @return string  The HTML output.
     */
    function _popupImageWindow()
    {
        $self_url = Util::addParameter(Horde::selfUrl(true), array('img_data' => 1, 'images_view_thumbnail' => Util::getFormData('view_thumbnail'), 'images_view_convert' => Util::getFormData('images_load_convert')));
        $title = MIME::decode($this->mime_part->getName(false, true));
        return parent::_popupImageWindow($self_url, $title);
    }

    /**
     * View thumbnail sized image.
     *
     * @access private
     *
     * @param optional boolean $thumb  View thumbnail size?
     *
     * @return string  The image data.
     */
    function _viewConvert($thumb = false)
    {
        $mime = $this->mime_part;
        $img = &$this->_getHordeImageOb();

        $img->loadString(1, $mime->getContents());
        if ($thumb) {
            $img->resize(96, 96, true);
        }
        $mime->setType($img->getContentType());
        $mime->setContents($img->raw());

        $this->_contentType = $img->getContentType();

        return $mime->getContents();
    }

    /**
     * Return a Horde_Image object.
     *
     * @access private
     *
     * @return object Horde_Image  The requested object.
     */
    function &_getHordeImageOb()
    {
        global $conf;

        include_once 'Horde/Image.php';
        $params = array('temp' => Horde::getTempdir());
        if (!empty($conf['image']['convert'])) {
            return Horde_Image::singleton('im', $params);
        } elseif (Util::extensionExists('gd')) {
            return Horde_Image::singleton('gd', $params);
        } else {
            return false;
        }
    }

}
