/*****************************************************************************
 *                              HashTable.h
 * Author: Matthew Ballance 
 * Desc:   Hash-table class
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 *****************************************************************************/
#ifndef INCLUDED_HASH_TABLE_H
#define INCLUDED_HASH_TABLE_H

#include "types.h"
#include "HashEntry.h"
#include <stdio.h>
#include <string.h>

template <class eType> class HashTable {
   
    /******************************************************
     * Public methods
     ******************************************************/
public:
    
    HashTable(Uint32    hashEntries);
    HashTable(void);
   ~HashTable(void);

    /****************************************************************
     * addEntry()
     ****************************************************************/
     void addEntry(eType    *newEntry);

    /****************************************************************
     * findEntry()
     ****************************************************************/
     eType *findEntry(Char    *key);

    /****************************************************************
     * removeEntry()
     ****************************************************************/
     eType *removeEntry(Char    *key);

private:
    eType        **syms;
    Uint32         numEntries;

};

#define DEFAULT_SIZE    65519

/****************************************************************
 * HashTable(void)
 ****************************************************************/
template <class eType> HashTable<eType>::HashTable(void)
{
    syms = new eType*[DEFAULT_SIZE];
    numEntries = 65519;

    memset(syms, 0, sizeof(eType *)*numEntries);
}

/****************************************************************
 * HashTable(size)
 ****************************************************************/
template <class eType> HashTable<eType>::HashTable(
    Uint32    size
    )
{
    entries = new eType*[size];
    numEntries = size;

    memset(syms, 0, sizeof(eType *)*numEntries);
}

/****************************************************************
 * addEntry()
 ****************************************************************/
template <class eType> void HashTable<eType>::addEntry(
    eType    *newEntry
    )
{
    Uint32    hashV;

    hashV = newEntry->hash(numEntries);
    newEntry->nextEntry = syms[hashV];
    syms[hashV] = newEntry;
}

/****************************************************************
 * findEntry()
 ****************************************************************/
template <class eType> eType *HashTable<eType>::findEntry(
    Char    *key
    )
{
    Uint32     hashV;
    HashEntry *tmp2;

    hashV = HashEntry::hash(key, numEntries);
    if (syms[hashV] == 0) {
        return 0;
    } else {
        tmp2 = syms[hashV];
    }

    while (tmp2) {
        if (!HashEntry::compare(tmp2, key)) {
            return (eType *)tmp2;
        }
        if (!tmp2->nextEntry) break;
        tmp2 = tmp2->nextEntry;
    }

    return 0;
}

/****************************************************************
 * removeEntry()
 ****************************************************************/
template <class eType> eType *HashTable<eType>::removeEntry(
    Char    *key
    )
{
    Uint32     hashV;
    eType      tmp(key);
    HashEntry *tmp2, *tmpHead, *tmpTail, *ret;

    hashV = tmp.hash(numEntries);
    if (syms[hashV] == 0) {
        return 0;
    } else {
        tmp2 = syms[hashV];
    }

    tmpHead = tmpTail = 0;

    while (tmp2) {
        /**** If not equal, append to the new root
         ****/
        if (tmp2->compare(&tmp)) {
            if (!tmpHead) {
                tmpHead = tmpTail = tmp2;
            } else {
                tmpTail->nextEntry = tmp2;
                tmpTail = tmp2;
            }
        } else {
            ret = tmp2;
        }
            /**** If this is the last entry, be sure that the
             **** new list is terminated properly...
             ****/
        if (!tmp2->nextEntry) {
            if (tmpTail) {
                tmpTail->nextEntry = 0;
            }
        }

        tmp2 = tmp2->nextEntry;
    }

    syms[hashV] = (eType *)tmpTail;

    return (eType *)ret;
}

/****************************************************************
 * HashCompare()
 * 
 * NOTE: Inputs are of type HashEntry, the base hash-entry type.
 ****************************************************************/
static Int32 HashCompare(const void *v1, const void *v2)
{
    HashEntry *s1 = *(HashEntry **)v1;
    HashEntry *s2 = *(HashEntry **)v2;

    int rc = s1->compare(s2);

    return rc;
}


#endif /*INCLUDED_HASH_TABLE_H*/
