/****************************************************************************
 *                             PsPixmapObj.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements a wrapper around a pixmap... This allows us to 
 *         implement various back-ends and use the same drawing code...
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "PsPixmapObj.h"
#include "tclConfig.h"
#include "PsGC.h"
#include "PsTextLayout.h"
#include "ivi_version.h"
#include <stdio.h>
#include <string.h>

#define FP      stderr
#undef  DEBUG_PIXMAP

#ifdef DEBUG_PIXMAP
#define DBG_MSG(x)   fprintf x
#else
#define DBG_MSG(x)
#endif

struct PsPmClip {
    Uint32    x;
    Uint32    y;
    Uint32    w;
    Uint32    h;
};

/********************************************************************
 * PsPixmapObj()
 ********************************************************************/
PsPixmapObj::PsPixmapObj(
        Tcl_Interp        *interp,
        Uint32             objc,
        Tcl_Obj           *const objv[]) : PixmapObj()
{
    ok = 0;

    d_interp = interp;

    d_fileAcc = 0;
    d_paperSize = 0;
    d_paperConfigured = false;
    d_currPage = 1;
    d_pageDone = true;
    d_originX      = 0;
    d_originY      = 0;
    d_headMargin   = 0;
    d_footMargin   = 0;
    d_lsideMargin  = 0;
    d_rsideMargin  = 0;
    d_currGC       = 0;
    d_headerOutput = 0;

    Tcl_DStringInit(&dStr);

    memset(&d_config, 0, sizeof(PsPixmapConfig));

    d_instName = Tcl_GetString(objv[1]);

    Tcl_CreateObjCommand(d_interp, d_instName.value(), &PsPixmapObj::InstCmd,
            this, 0);
    Tcl_AppendResult(d_interp, d_instName.value(), 0);

    if (Tcl_InitOptions(d_interp, (char *)&d_config, d_OptTab) != TCL_OK) {
        fprintf(stderr, "InitOptions failed\n");
        return;
    }

    if (ConfigFile() != TCL_OK) {
        fprintf(stderr, "ConfigFile failed\n");
        return;
    }

    ConfigPaper();

    if (Configure(objc-2, &objv[2]) != TCL_OK) {
        fprintf(stderr, "Configure failed\n");
        return;
    }

    WidgetMgr_AddInst(d_interp, "PsPixmapObj", d_instName.value(), this);

    ok = 1;
}

/********************************************************************
 * ~PsPixmapObj()
 ********************************************************************/
PsPixmapObj::~PsPixmapObj()
{
    if (d_fileAcc) {
        if (!d_pageDone) {
            EndPage();
        }

        /**** Causes file to be flushed... ****/
        delete d_fileAcc;
    }

    WidgetMgr_DelInst(d_interp, "PsPixmapObj", d_instName.value());
}

/********************************************************************
 * CreateInst()
 ********************************************************************/
int PsPixmapObj::CreateInst(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               objc,
        Tcl_Obj          *const objv[])
{
    PsPixmapObj *obj = new PsPixmapObj(interp, objc, objv);
    return (obj->ok)?TCL_OK:TCL_ERROR;
}

enum {
    PPC_Config = 1,
    PPC_Cget,
    PPC_Delete,
    PPC_PaperSize,
    PPC_PaperSizes,
    PPC_Margins,
    PPC_NumCmds
};

/********************************************************************
 * getCmdStructs()
 ********************************************************************/
ObjWidgetBase::CmdStruct *PsPixmapObj::getCmdStructs()
{
    static ObjWidgetBase::CmdStruct cmds[] = {
        {"configure",         PPC_Config    },
        {"config",            PPC_Config    },
        {"cget",              PPC_Cget      },
        {"delete",            PPC_Delete    },
        {"paper_size",        PPC_PaperSize },
        {"paper_sizes",       PPC_PaperSizes},
        {"margins",           PPC_Margins   },
        { (char *)0,          PPC_NumCmds   },
    };

    return cmds;
}

/********************************************************************
 * setPaperSize()
 ********************************************************************/
int PsPixmapObj::setPaperSize(const char *size)
{
    PaperSize *psize;

    if (!(psize = FindPaperSize(size))) {
        Tcl_AppendResult(d_interp, "No paper size \"", size, "\"", 0);
        return TCL_ERROR;
    }

    return TCL_OK;
}

/********************************************************************
 * newPage()
 ********************************************************************/
void PsPixmapObj::newPage()
{

}

/********************************************************************
 * InstCmd()
 ********************************************************************/
int PsPixmapObj::InstCmd(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               objc,
        Tcl_Obj          *const objv[])
{
    return ((PsPixmapObj *)clientData)->InstCmd(objc, objv);
}

/********************************************************************
 * InstCmd()
 ********************************************************************/
int PsPixmapObj::InstCmd(int objc, Tcl_Obj *const objv[])
{
    Int32 cmd;

    cmd = ObjWidgetBase::CmdSwitch(getCmdStructs(), Tcl_GetString(objv[1]));

    switch (cmd) {
        case PPC_Config:
            return Configure(objc-2, &objv[2]);
            break;

        case PPC_Cget:
            Tcl_SetObjResult(d_interp,
                    Tcl_GetOptionValue(d_interp, (char *)&d_config,
                        d_OptTab, objv[2]));
            break;

        case PPC_Delete:
            delete this;
            break;

        case PPC_PaperSize:
            if (objc > 2) {
                return setPaperSize(Tcl_GetString(objv[2]));
            } else {
            }

        case PPC_PaperSizes:
            {
                Tcl_Obj    *ret = Tcl_NewListObj(0, 0);
                Tcl_Obj    *sub;
                Uint32      idx=0;

                while (d_paperSizes[idx].name) {
                    sub = Tcl_NewListObj(0, 0);
                    Tcl_ListObjAppendElement(d_interp, sub, 
                            Tcl_NewStringObj(d_paperSizes[idx].name, -1));
                    Tcl_ListObjAppendElement(d_interp, sub,
                            Tcl_NewIntObj(d_paperSizes[idx].x));
                    Tcl_ListObjAppendElement(d_interp, sub,
                            Tcl_NewIntObj(d_paperSizes[idx].y));
                    Tcl_ListObjAppendElement(d_interp, ret, sub);
                    idx++;
                }

                Tcl_SetObjResult(d_interp, ret);
            }
            break;

        /************************************************************
         * PPC_Margins
         ************************************************************/
        case PPC_Margins:
            if (objc == 6) {
                Int32    head, foot, lside, rside;

                if (Tcl_GetIntFromObj(d_interp, objv[2], &lside) != TCL_OK) {
                    return TCL_ERROR;
                }

                if (Tcl_GetIntFromObj(d_interp, objv[2], &head) != TCL_OK) {
                    return TCL_ERROR;
                }

                if (Tcl_GetIntFromObj(d_interp, objv[2], &rside) != TCL_OK) {
                    return TCL_ERROR;
                }

                if (Tcl_GetIntFromObj(d_interp, objv[2], &foot) != TCL_OK) {
                    return TCL_ERROR;
                }

                SetMargins(lside, head, rside, foot);
            } else if (objc == 2) {
                Tcl_Obj *ret = Tcl_NewListObj(0, 0);

                Tcl_ListObjAppendElement(d_interp, ret, 
                        Tcl_NewIntObj(d_lsideMargin));
                Tcl_ListObjAppendElement(d_interp, ret, 
                        Tcl_NewIntObj(d_rsideMargin));
                Tcl_ListObjAppendElement(d_interp, ret, 
                        Tcl_NewIntObj(d_headMargin));
                Tcl_ListObjAppendElement(d_interp, ret, 
                        Tcl_NewIntObj(d_footMargin));

                Tcl_SetObjResult(d_interp, ret);
            } else {
                Tcl_AppendResult(d_interp, "wrong # args", 0);
                return TCL_ERROR;
            }
            break;

        default:
            return TCL_ERROR;
            break;
    }

    return TCL_OK;
}

#if 0
/********************************************************************
 * operator =
 ********************************************************************/
void PsPixmapObj::operator = (const PixmapRange &rng)
{
    PixmapObj *src = rng.d_pixmap;

    copy(src, rng.d_x, rng.d_y, rng.d_width, rng.d_height);
}
#endif

/********************************************************************
 * setBgColor()
 ********************************************************************/
void PsPixmapObj::setBgColor(Tk_3DBorder background) 
{
    d_background = background;
}

/********************************************************************
 * emit()
 ********************************************************************/
void PsPixmapObj::emit(const char *fmt, ...)
{
    va_list   ap;
    va_start(ap, fmt);

    if (!d_fileAcc) {
        return;
    }

    vsprintf(d_prvBuf, fmt, ap);
    d_fileAcc->write_bytes((Uint8 *)d_prvBuf, strlen(d_prvBuf));

    va_end(ap);
}

/********************************************************************
 * translate()
 ********************************************************************/
void PsPixmapObj::translate(Uint32 &x, Uint32 &y)
{
    Uint32 t;

    y += d_originY;
    x += d_originX;

    /**** Portrait:  (y, x)
     **** Landscape: (x, y')
     **** y' = width(paper)-y
     ****/
    if (d_isLandscape) {
        t = x;
        x = y;
        y = t;
    } else {

    }
}

/********************************************************************
 * SetGC()
 ********************************************************************/
void PsPixmapObj::SetGC(GC gc)
{
    PsGC *pgc = (PsGC *)gc;

    if (!d_headerOutput) {
        OutputHeader();
        d_headerOutput = true;
    }

    if (pgc != d_currGC) {
        if (d_fileAcc) {
            pgc->SetGC(d_fileAcc);
        }
        d_currGC = pgc;
    }
}

/********************************************************************
 * width()
 ********************************************************************/
Uint32 PsPixmapObj::width() const
{
    return d_winW;
}

/********************************************************************
 * height()
 ********************************************************************/
Uint32 PsPixmapObj::height() const
{
    return d_winH;
}

/********************************************************************
 * getRealHeight()
 ********************************************************************/
Uint32 PsPixmapObj::getRealHeight() const
{
    if (!d_paperSize) {
        return 0;
    }

    if (d_isLandscape) {
        return d_paperSize->x;
    } else {
        return d_paperSize->y;
    }
}

/********************************************************************
 * getRealWidth()
 ********************************************************************/
Uint32 PsPixmapObj::getRealWidth() const
{
    if (!d_paperSize) {
        return 0;
    }

    if (d_isLandscape) {
        return d_paperSize->y;
    } else {
        return d_paperSize->x;
    }
}

/*******************************************************************
 * prvSetWinRect()
 *******************************************************************/
void PsPixmapObj::prvSetWinRect(Uint32 x, Uint32 y, Uint32 w, Uint32 h)
{
    if (d_isLandscape) {
        d_winH    = w;
        d_winW    = h;
        d_originX = y;
        d_originY = x;
    } else {
        d_winW    = w;
        d_winH    = h;
        d_originX = x;
        d_originY = y;
    }
}

/*******************************************************************
 * setWinRect()
 *
 * x, y are offsets from the current clip
 ********************************************************************/
void PsPixmapObj::setWinRect(Uint32 x, Uint32 y, Uint32 w, Uint32 h)
{
    PsPmClip    *clip = new PsPmClip;

    /**** Save current clip ****/
    getWinRect(clip->x, clip->y, clip->w, clip->h);
    d_clipStack.push(clip);

    x      += d_originX;
    y      += d_originY;
    d_winW  = w;
    d_winH  = h;

    d_originX = x;
    d_originY = y;

    emit("save %d %d %d %d rectclip\n", 
            (d_isLandscape)?y:x,
            (d_isLandscape)?x:y,
            (d_isLandscape)?h:w,
            (d_isLandscape)?w:h);
    /*
    emit("save %d %d %d %d rectclip\n", 
            y,x, h,w);
     */             
}

/********************************************************************
 * getWinRect()
 ********************************************************************/
void PsPixmapObj::getWinRect(Uint32 &x, Uint32 &y, Uint32 &w, Uint32 &h)
{
    w = d_winW;
    h = d_winH;
    x = d_originX;
    y = d_originY;
}

/********************************************************************
 * clrWinRect()
 ********************************************************************/
void PsPixmapObj::clrWinRect()
{
    PsPmClip   *clip;

    if (d_clipStack.depth()) {
        emit("restore\n");
        clip = d_clipStack.pop();
        /*
         * set clip
         */
        d_originX = clip->x;
        d_originY = clip->y;
        d_winW    = clip->w;
        d_winH    = clip->h;
        delete clip;
    }
}

/********************************************************************
 * SetMargins()
 ********************************************************************/
void PsPixmapObj::SetMargins(Uint32 lside, Uint32 head, Uint32 rside,
        Uint32 foot)
{
    d_headMargin = head;
    d_footMargin = foot;
    d_lsideMargin = lside;
    d_rsideMargin = rside;

    setWinRect(lside, head, width()-(lside+rside), height()-(foot+head));
}

/********************************************************************
 * line()
 ********************************************************************/
void PsPixmapObj::line(GC gc, Uint32 x1, Uint32 y1, 
        Uint32 x2, Uint32 y2)
{
    Uint32 x1_p, y1_p, x2_p, y2_p;
    SetGC(gc);

    translate(x1, y1);
    translate(x2, y2);

    emit("%d %d %d %d l\n", 
            x1, y1, x2, y2);
}

/********************************************************************
 * rect()
 ********************************************************************/
void PsPixmapObj::rect(GC gc, Uint32 x, Uint32 y, 
        Uint32 width, Uint32 height)
{
    SetGC(gc);

    translate(x, y);

    if (d_isLandscape) { 
        Uint32 t;
        t = width;
        width = height;
        height = t;
    } 

    emit("%d %d %d %d rectstroke\n", x, y, width, height);
}

/********************************************************************
 * fill_rect()
 ********************************************************************/
void PsPixmapObj::fill_rect(GC gc, Uint32 x, Uint32 y, 
        Uint32 width, Uint32 height)
{
    SetGC(gc);

    translate(x, y);

    if (!d_isLandscape) {
        Uint32 t;
        t = width;
        width = height;
        height = t;
    }

    emit("%d %d %d %d rectfill\n", x, y, width, height);
}

/********************************************************************
 * createGC()
 ********************************************************************/
GC PsPixmapObj::createGC(GCObj &gcObj)
{
    PsGC *gc = new PsGC(&gcObj);

    return (GC)gc;
}

/********************************************************************
 * freeGC()
 ********************************************************************/
void PsPixmapObj::freeGC(GCObj &gcObj)
{
    PsGC *gc = (PsGC *)gcObj.getGC();

    if (gc == d_currGC) {
        d_currGC = 0;
    }

    delete gc;
}

/********************************************************************
 * copy()
 ********************************************************************/
void PsPixmapObj::copy(
        PixmapObj *src, Uint32 x, Uint32 y, Uint32 width,
        Uint32 height, Uint32 x1, Uint32 y1)
{
#ifdef UNDEFINED
    DBG_MSG((FP, "----> copy(%x, %d, %d, %d, %d, %d, %d) %x = dest\n",
                src, x, y, width, height, x1, y1, this));

    /**** Copy data from obj to this pixmap... ****/
    XCopyArea(d_display, ((PsPixmapObj *)src)->getPixmap(), 
            getPixmap(), getCopyGC(), x, y, width, height, x1, y1);

    DBG_MSG((FP, "<---- copy(%x, %d, %d, %d, %d, %d, %d) %x = dest\n",
                src, x, y, width, height, x1, y1, this));
#endif /* UNDEFINED */
}

/********************************************************************
 * copy()
 ********************************************************************/
void PsPixmapObj::copy(
        PixmapObj *obj, Uint32 x, Uint32 y, Uint32 width, Uint32 height)
{
    copy(obj, x, y, width, height, 0, 0);
}

/********************************************************************
 * width_height()
 ********************************************************************/
void PsPixmapObj::width_height(Uint32 width, Uint32 height)
{
#ifdef UNDEFINED
    DBG_MSG((FP, "----> width_height( %d, %d )\n", width, height));

    if (width != d_width || height != d_height) {
        if (d_pixmap) {
            DBG_MSG((FP, "\tFree pixmap...\n"));
            Tk_FreePixmap(d_display, d_pixmap);
        }

        DBG_MSG((FP, "\tAlloc pixmap\n"));
        d_pixmap = Tk_GetPixmap(d_display, Tk_WindowId(d_window), 
                width, height, Tk_Depth(d_window));
    }

    d_width  = width;
    d_height = height;

    DBG_MSG((FP, "<---- width_height( %d, %d )\n", width, height));
#endif /* UNDEFINED */
}

/********************************************************************
 * ComputeTextLayout()
 ********************************************************************/
Tk_TextLayout PsPixmapObj::ComputeTextLayout(
                Tk_Font font, Char *str, Int32 len, Int32 wraplen,
                Tk_Justify justify, Int32 flags, Int32 *width, Int32 *height)
{
    PsTextLayout *ps_tl = new PsTextLayout(font, str, len, wraplen,
            justify, flags, width, height);

    return (Tk_TextLayout)ps_tl;
}

/********************************************************************
 * DrawTextLayout()
 ********************************************************************/
void PsPixmapObj::DrawTextLayout(GC gc, Tk_TextLayout layout, 
        Uint32 x, Uint32 y, Int32 start, Int32 end)
{
    PsTextLayout *ps_tl = (PsTextLayout *)layout;
    SetGC(gc);

    if (d_isLandscape) {
        y += ((ps_tl->getHeight()*3)/4);
    }
    translate(x, y);

#if 0
    emit("%d %d (%s)", x, y, ps_tl->getString());

    if (d_isLandscape) {
        emit(" tl\n");
    } else {
        emit(" tp\n");
    }

#else
    emit("save %d %d moveto ", x, y);

    if (d_isLandscape) {
        emit("90 rotate ");
    }

    if (d_fileAcc) {
        ps_tl->DrawLayout(d_fileAcc, x, y, start, end);
    }

    emit("restore\n");
#endif
}

/********************************************************************
 * FreeTextLayout()
 ********************************************************************/
void PsPixmapObj::FreeTextLayout(Tk_TextLayout layout)
{
    PsTextLayout *ps_tl = (PsTextLayout *)layout;

    delete ps_tl;
}

/********************************************************************
 * blank()
 ********************************************************************/
void PsPixmapObj::blank()
{
#ifdef UNDEFINED
    if (!d_background) {
        fprintf(stderr, "PixmapObj ERROR :: blank called w/NULL background\n");
        return;
    }

    if (d_pixmap) {
        Tk_Fill3DRectangle(d_window, d_pixmap, d_background, 0, 0,
                d_width, d_height, 0, TK_RELIEF_FLAT);
    } else if (d_window) {
        Tk_SetWindowBackground(d_window, Tk_3DBorderColor(d_background)->pixel);
    }
#endif /* UNDEFINED */
}

/********************************************************************
 * Configure()
 ********************************************************************/
int PsPixmapObj::Configure(int objc, Tcl_Obj *const objv[])
{
    Tk_SavedOptions        savedOptions;
    int                    mask;

    if (Tcl_SetOptions(d_interp, (char *)&d_config, d_OptTab,
                objc, objv, &savedOptions, &mask) != TCL_OK) {
        Tk_RestoreSavedOptions(&savedOptions);
        return TCL_ERROR;
    }

    if ((mask & Opt_PaperSize) || (mask & Opt_PaperOrient)) {
        ConfigPaper();
    } else if (!d_paperConfigured) {
        ConfigPaper();
    }

    if (mask & Opt_OutputFilename) {
        if (ConfigFile() != TCL_OK) {
            return TCL_ERROR;
        }
    }
    return TCL_OK;
}

/********************************************************************
 * FindPaperSize()
 ********************************************************************/
PsPixmapObj::PaperSize *PsPixmapObj::FindPaperSize(const char *name)
{
    PaperSize *paper = 0;
    Uint32     idx = 0;

    while (d_paperSizes[idx].name) {
        if (!strcasecmp(d_paperSizes[idx].name, name)) {
            paper = &d_paperSizes[idx];
            break;
        }
        idx++;
    }

    return paper;
}

/********************************************************************
 * ConfigPaper()
 ********************************************************************/
int PsPixmapObj::ConfigPaper()
{
    PaperSize   *paper;

    if (!strcasecmp(d_config.paperOrientStr, "portrait")) {
        d_isLandscape = false;
    } else if (!strcasecmp(d_config.paperOrientStr, "landscape")) {
        d_isLandscape = true;
    } else {
        /*** ERROR ***/
    }

    if ((paper = FindPaperSize(d_config.paperSizeStr))) {
        d_paperSize = paper;
        d_originX = d_lsideMargin;
        d_originY = d_headMargin;
        if (d_isLandscape) {
            d_winW    = paper->y;
            d_winH    = paper->x;
        } else {
            d_winW    = paper->x;
            d_winH    = paper->y;
        }

        d_winW -= (d_lsideMargin+d_rsideMargin);
        d_winH -= (d_headMargin+d_footMargin);
    } else {
        /*** ERROR ***/
    }


    d_paperConfigured = true;

    return TCL_OK;
}

/********************************************************************
 * StartPage()
 ********************************************************************/
void PsPixmapObj::StartPage()
{
    if (!d_headerOutput) {
        OutputHeader();
        d_headerOutput = true;
    }

    if (d_pageDone) {
        emit("%%%%Page: %d %d\n", d_currPage, d_currPage);
    }
    d_pageDone = false;
}
/********************************************************************
 * EndPage()
 ********************************************************************/
void PsPixmapObj::EndPage()
{
    if (!d_headerOutput) {
        OutputHeader();
        d_headerOutput = true;
    }

    if (!d_pageDone) {
        emit("showpage\n");
        d_currPage++;
    }
    d_pageDone = true;
}

/********************************************************************
 * OutputHeader()
 ********************************************************************/
void PsPixmapObj::OutputHeader()
{
    static const char *procs[] = {
        "/l { newpath moveto lineto stroke } def",
        "/tp { save moveto show restore } def",
        "/tl { save moveto 90 rotate show restore } def",
        (char *)0
    };

    if (!d_paperSize) {
        return;
    }

    emit("%%!PS-Adobe-2.0\n");
    /*** %%Title: <Title> ***/
    /*** %%For:   <Name>  ***/
    emit("%%%%Creator: Icarus Verilog Interactive version %s\n",
            ivi_get_version());

    emit("%%%%Orientation: %s\n",
            (d_isLandscape)?"Landscape":"Portrait");
    /*** %%Pages: <Number> ***/
    emit("%%%%BoundingBox: 0 0 %d %d\n", d_paperSize->x, d_paperSize->y);
    emit("%%%%DocumentMedia: %s %d %d 0 () ()\n",
            d_paperSize->name, d_paperSize->x, d_paperSize->y);
    emit("%%%%EndComments\n");
    emit("\n");

    /*
    if (!d_isLandscape) {
        emit("0 %d translate -90 rotate\n", d_paperSize->y);
    }
     */

    int idx=0;
    while (procs[idx]) {
        emit("%s\n", procs[idx++]);
    }
}

/********************************************************************
 * ConfigFile()
 ********************************************************************/
int PsPixmapObj::ConfigFile()
{
    if (!d_config.outputFilename || !d_config.outputFilename[0]) {
        return TCL_OK;
    }

    if (d_fileAcc) {
        /**** Already have a file, must close first ****/
        delete d_fileAcc;
    }

    d_fileAcc = new FileAccObj(d_config.outputFilename, 
            FileAccObj::OpenMode_Write);
    if (!d_fileAcc->okay) {
        delete d_fileAcc;
        d_fileAcc = 0;
        return TCL_ERROR;
    }

    OutputHeader();
    
    return TCL_OK;
}

/********************************************************************
 * getType()
 ********************************************************************/
PixmapObj::PixmapType PsPixmapObj::getType() const
{
   return PsPixmap;
}

/********************************************************************
 * getOptionSpec()
 ********************************************************************/
Tk_OptionSpec *PsPixmapObj::getOptionSpec()
{
    static Tk_OptionSpec optSpec[] = {
        {TK_OPTION_STRING, "-paper_size", "paper_size", "PaperSize",
            "Letter", -1, Tk_Offset(PsPixmapConfig, paperSizeStr),
            0, 0, Opt_PaperSize},
        {TK_OPTION_STRING, "-orientation", "orientation", "Orientation",
            "landscape", -1, Tk_Offset(PsPixmapConfig, paperOrientStr),
            0, 0, Opt_PaperOrient},
        {TK_OPTION_STRING, "-file", "file", "File",
            (char *)0, -1, Tk_Offset(PsPixmapConfig, outputFilename),
            0, 0, Opt_OutputFilename},
        {TK_OPTION_END, (char *)NULL, (char *)NULL,
            (char *)NULL, (char *)NULL, -1, 0, 0, 0, 0}
    };

    return optSpec;
}

/********************************************************************
 * PsPixmapObj_Init()
 ********************************************************************/
extern "C" int PsPixmapObj_Init(Tcl_Interp *d_interp)
{
    LogRegionType    *lr;

    lr = new LogRegionType("PsPixmapObj",
            "Postscript Pixmap",
            "",
            "",
            "");

    LogMgr::AddRegionType(lr);

    PsPixmapObj::d_OptTab = Tk_CreateOptionTable(d_interp,
            PsPixmapObj::getOptionSpec());

    return ObjWidgetBase::WidgetInit(
            d_interp,
            "PsPixmapObj",
            "1.0",
            "ps_pixmap",
            &PsPixmapObj::CreateInst);
}


PsPixmapObj::PaperSize PsPixmapObj::d_paperSizes[] = {
    {"Letter",  612,  792  },
    {"Legal",   612,  1008 },
    {"Ledger",  1224, 792  },
    {"Tabloid", 792,  1224 },
    {"A0",      2384, 3370 },
    {"A1",      1684, 2384 },
    {"A2",      1191, 1684 },
    {"A3",      842,  1191 },
    {"A4",      595,  842  },
    {"A5",      420,  595  },
    {"A6",      297,  420  },
    {"A7",      210,  297  },
    {"A8",      148,  210  },
    {"A9",      105,  148  },
    {"B0",      2920, 4127 },
    {"B1",      2064, 2920 },
    {"B2",      1460, 2064 },
    {"B3",      1032, 1460 },
    {"B4",      729,  1032 },
    {"B5",      516,  729  },
    {"B6",      363,  516  },
    {"B7",      258,  363  },
    {"B8",      181,  258  },
    {"B9",      127,  181  },
    {"B10",     91,   127  },
    { (char *)0, 0, 0}
};

char PsPixmapObj::d_prvBuf[8192];
Tk_OptionTable PsPixmapObj::d_OptTab = 0;


