/*****************************************************************************
 *                                 SigDB.cc
 * Author: Matthew Ballance
 * Desc:   Implements the signal database...
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 *****************************************************************************/
#include "SigDB.h"
#include "SdbMgr.h"
#include "DesignDB.h"
#include "TclBitVector.h"

#include "WidgetManager.h"
#include "DFIO.h"
#include "CmdSwitcher.h"
#include "CallbackMgr.h"
#include "DFIOMgr.h"
#include "SdbReader.h"

#include <string.h>

#define FP stderr
#undef  DEBUG_DB

#ifdef DEBUG_DB
#define DBG_MSG(x) fprintf x
#else
#define DBG_MSG(x)
#endif

static int Sdb_InstCmd(
    ClientData        clientData,
    Tcl_Interp       *interp,
    int               argc,
    char            **argv);

/******************************************************************
 * SigDB()
 ******************************************************************/
SigDB::SigDB(
        Tcl_Interp        *interp,
        Uint32             argc,
        Char             **argv) : instName(argv[1])
{
    dfioName = 0;
    dfio     = 0;
    ddbName  = 0;
    ddb      = 0;
    okay     = 0;
    sdbId    = 0;

    d_SimStepEnd_CbId    = 0;
    d_SimStepUpdate_CbId = 0;

    traceVector = new Vector<DFIOTrace>();

    maxTime      = 0;
    maxTimeValid = 0;

    this->interp = interp;

    if (Configure(argc-2, &argv[2], 0) != TCL_OK) {
        fprintf(stderr, "ERROR: Config failed - %s\n",
                Tcl_GetStringResult(interp));
        return;
    }

    WidgetMgr_AddInst(interp, WIDGET_TYPE_SDB, argv[1], this);
    Tcl_CreateCommand(interp, argv[1], 
            (Tcl_CmdProc *)Sdb_InstCmd, this, NULL);

    Tcl_AppendResult(interp, argv[1], 0);

    okay     = 1;
}

/******************************************************************
 * ~SigDB()
 ******************************************************************/
SigDB::~SigDB()
{
    if (d_SimStepEnd_CbId) {
        CallbackMgr_DeleteCb(d_SimStepEnd_CbId);
    }

    if (d_SimStepUpdate_CbId) {
        CallbackMgr_DeleteCb(d_SimStepUpdate_CbId);
    }

    for (Uint32 i=0; i<d_traces.length(); i++) {
        delete d_traces.idx(i);
    }

    SdbMgr_RemoveSDB(this);

    /**** Now, detach ourselves from any client ****/
    Uint32 numClients = d_clients.length();
    Uint32 clntIdx = 0;
    while (numClients && (clntIdx < numClients)) {
        SdbReader *client = d_clients.idx(clntIdx);
        client->removeSDB(this);

        /**** If removing this sdb didn't result in a client 
         **** being removed, then move on to the next
         ****/
        if (numClients == d_clients.length()) {
            fprintf(stderr, "ERROR: Removing SDBR 0x%08x doesn't "
                    "result in any change in num clients\n", client);
            clntIdx++;
        }
        numClients = d_clients.length();
    }
}

typedef enum {
    SDC_Config = 1,
    SDC_Cget,
    SDC_Close,
    SDC_NumCmds
};

static CmdSwitchStruct   sdb_cmds[] = {
    { "configure",         SDC_Config },
    { "config",            SDC_Config },
    { "cget",              SDC_Cget   },
    { "close",             SDC_Close  },
    { "",                  0          }
};

/******************************************************************
 * InstCmd()
 ******************************************************************/
Int32 SigDB::InstCmd(Uint32 argc, Char **argv)
{
    Int32 cmd;

    if (argc < 1) {
        Tcl_AppendResult(interp, "too few args", 0);
        return TCL_ERROR;
    }

    cmd = CmdSwitch(sdb_cmds, argv[1]);
    switch(cmd) {
        case SDC_Config:
            return Configure(argc-2, &argv[2], TK_CONFIG_ARGV_ONLY);
            break;

        case SDC_Cget:
            return Tcl_ConfigureValue(interp, 0, getConfigSpec(),
                    (char *)this, argv[2], 0);
            break;

        case SDC_Close:
            Tcl_AppendResult(interp, "warning -- close unimplemented", 0);
            break;

        default:
            Tcl_AppendResult(interp, "no sub-command ", argv[1], 0);
            break;
    }

    return TCL_OK;
}

/******************************************************************
 * addClient()
 ******************************************************************/
void SigDB::addClient(SdbReader *client)
{
    d_clients.append(client);
}

/******************************************************************
 * removeClient()
 ******************************************************************/
void SigDB::removeClient(SdbReader *client)
{
    d_clients.remove(client);
}

/******************************************************************
 * SigValChangeCB()
 ******************************************************************/
static int SigValChangeCB(p_cb_data    cb_data_p)
{
    DFIOTrace          *sigDesc = (DFIOTrace *)cb_data_p->user_data;
    IviSim             *sim = (IviSim *)sigDesc->userData[0];
    s_cb_data           cb_data_s;
    s_vpi_time          time_s;
    s_vpi_value         value_s;
    Vector<DFIOValChg> *vals;
    Uint32              chk = 0;
    
    cb_data_s = *cb_data_p;

    value_s.format = vpiBinStrVal;
    sim->vpi_get_value(cb_data_p->obj, &value_s);
    time_s.type = vpiSimTime;
    sim->vpi_get_time(0, &time_s); 

    sigDesc->setTime(time_s.low);
    sigDesc->setValueBitStr(0, value_s.value.str);

    return 0;
}

/******************************************************************
 * SDB_SimStepEnd()
 *
 * This callback function is simply a reflector... It translates
 * sim-step-end callbacks to  SDB sig-update events...
 ******************************************************************/
int SigDB::SDB_SimStepEnd(
        ClientData         clientData,
        Tcl_Interp        *interp,
        int                objc,
        Tcl_Obj           *const objv[])
{
    SigDB *sdb = (SigDB *)clientData;
    Uint32 tm;

    sdb->maxTimeValid = 0;

    /**** The time is the 1st arg... ****/
    tm = strtoul(Tcl_GetString(objv[0]), 0, 0);
    sdb->dfio->setMaxTime(tm);

    CallbackMgr_Invoke(CBTYPE_SDB_SIGDATA_UPDATE, 
            sdb->instName.value(), objc, objv);

    return TCL_OK;
}

/******************************************************************
 * Configure()
 ******************************************************************/
Int32 SigDB::Configure(Uint32 argc, Char **argv, Uint32 flags)
{
    Int32      ret;
    IviSim    *sim;

    ret = Tcl_ConfigureWidget(interp, 0, getConfigSpec(), argc, argv,
            (char *)this, flags);

    if (ret != TCL_OK) {
        return ret;
    }

    if (optionSpecified(OPT_DDB) && ddbName) {
        ddb = (DesignDB *)WidgetMgr_GetObjHandle(interp, 
                WIDGET_TYPE_DDB, ddbName);
        if (!ddb) {
            fprintf(stderr, "ERROR :: Cannot locate DDB \"%s\"\n", ddbName);
            Tcl_AppendResult(interp, "no DDB named ", ddbName, 0);
            return TCL_ERROR;
        }
        DBG_MSG((FP, "SDB :: DDB Specified as \"%s\" - %x\n", 
                    ddbName, ddb));

        if ((sim = ddb->get_sim())) {
            CallbackMgr_AddCb(CBTYPE_SIM_RUNSTEP_END, sim->getInstName(),
                    SigDB::SDB_SimStepEnd, this,
                    &d_SimStepEnd_CbId);
            CallbackMgr_AddCb(CBTYPE_SIM_RUNSTEP_UPDATE, sim->getInstName(),
                    SigDB::SDB_SimStepEnd, this,
                    &d_SimStepUpdate_CbId);
        }
    }

    if (optionSpecified(OPT_DFIO) && dfioName) {
        dfio = (DFIO *)WidgetMgr_GetObjHandle(interp, "DFIO", dfioName);
        if (!dfio) {
            fprintf(stderr, "ERROR: no DFIO named \"%s\"\n", dfioName);
            Tcl_AppendResult(interp, "no DFIO named ", dfioName, 0);
            return TCL_ERROR;
        }
        DBG_MSG((FP, "SDB :: DFIO Specified as \"%s\" - %x\n", 
                    dfioName, dfio));
    }

    if (!flags && !sdbId) {
        char *tmp = instName.value(), *ptr;

        if ((ptr = strrchr(tmp, '.'))) {
            sdbId = strdup(ptr);
        } else {
            sdbId = strdup(tmp);
        }
    }

    return TCL_OK;
}

/******************************************************************
 * addSimSignals()
 ******************************************************************/
void SigDB::addSimSignals(IviSim *sim, const char *path)
{
    Vector<TreeNode>   *sigVect;
    TreeNode           *sig;
    vpiHandle           vpiHndl;
    DFIOTrace          *newTrace;
    Char               *name;
    s_vpi_value         value_s;
    s_vpi_time          time_s;
    p_vpi_time          time_p;
    p_cb_data           cb_data_p;
    Uint32              msb, lsb, i;

    /**** Pick up the simulation resolution ****/
    resolution = sim->vpi_get(vpiTimePrecision, 0);

    sigVect = ddb->globElems(0, path, "@signals@", 0);

    for (i=0; i<sigVect->length(); i++) {
        sig = sigVect->idx(i);
        name = sig->findField("fullName")->getStringVal()->value();
        newTrace = dfio->findTrace(name);

        if (!newTrace) {
            vpiHndl = (vpiHandle)sig->findField("vpiHandle")->getVoidVal();
            msb = (Uint32)sig->findField("sigMsb")->getVoidVal();
            lsb = (Uint32)sig->findField("sigLsb")->getVoidVal();

            newTrace = dfio->newTrace(name, 0, msb, lsb, 16);
            newTrace->userData[0] = sim;
            dfio->addTrace(newTrace);

            /**** Add a simulator callback to update this trace on signal
             **** changes...
             ****/
            if (vpiHndl) {
                cb_data_p = (p_cb_data)malloc(sizeof(s_cb_data));
                time_p    = (p_vpi_time)malloc(sizeof(s_vpi_time));
                time_p->type         = vpiSimTime;
                cb_data_p->time      = time_p;
                cb_data_p->user_data = (char *)newTrace;
                cb_data_p->obj       = vpiHndl;
                cb_data_p->reason    = cbValueChange;
                cb_data_p->cb_rtn    = SigValChangeCB;
                cb_data_p->value     = (p_vpi_value)malloc(sizeof(s_vpi_value));
                cb_data_p->value->format = vpiSuppressVal;
                cb_data_p->index     = 0;
                sim->vpi_register_cb(cb_data_p); 
                value_s.format = vpiBinStrVal;
                sim->vpi_get_value(vpiHndl, &value_s);
                time_s.type = vpiSimTime;
                sim->vpi_get_time(0, &time_s);
                newTrace->setTime(time_s.low);
                newTrace->setValueBitStr(0, value_s.value.str);
            }
        }
        traceVector->append(newTrace);
    }
}

/******************************************************************
 * addDFIOSignals()
 ******************************************************************/
void SigDB::addDFIOSignals(DFIO *dfio, const char *path)
{
    Vector<TreeNode>   *sigVect;
    TreeNode           *sig;
    vpiHandle           vpiHndl;
    DFIOTrace          *newTrace;
    Char               *name;
    s_vpi_value         value_s;
    s_vpi_time          time_s;
    p_vpi_time          time_p;
    p_cb_data           cb_data_p;
    Uint32              msb, lsb, i;

    /**** Pick up the simulation resolution ****/
/*    resolution = sim->vpi_get(vpiTimePrecision, 0); */

    sigVect = ddb->globElems(0, path, "@signals@", 0);

    for (i=0; i<sigVect->length(); i++) {
        sig = sigVect->idx(i);
        name = sig->findField("fullName")->getStringVal()->value();
        newTrace = (DFIOTrace *)sig->findField("vpiHandle")->getVoidVal();
        traceVector->append(newTrace);
    }
}

/******************************************************************
 * addSignal()
 *
 * TODO :: Must extract sig from sim/dfio as appropriate 
 ******************************************************************/
Vector <DFIOTrace> *SigDB::addSignal(Char *path)
{
    IviSim             *sim;
    DFIO               *dfio;

    traceVector->empty();

    sim = ddb->get_sim();
    dfio = ddb->get_dfio();

    if (sim) {
        addSimSignals(sim, path);
    } else if (dfio) {
        addDFIOSignals(dfio, path);
    } else {
        fprintf(stderr, "ERROR :: SDB has no DFIO or SIM\n");
    }

    return traceVector;
}

/******************************************************************
 * get_resolution()
 ******************************************************************/
Int32 SigDB::get_resolution()
{
    if (dfio) {
        return dfio->getTimeUnit();
    } else {
        fprintf(stderr, "WARNING: SigDB has no DFIO to give resolution\n");
        return -9;
    }
}

/******************************************************************
 * getMaxTime()
 ******************************************************************/
Uint32 SigDB::getMaxTime()
{
    if (maxTimeValid) {
        return maxTime;
    }

    if (dfio) {
        maxTime = dfio->getMaxTime();
    }

    maxTimeValid = 1;
    return maxTime;
}

/******************************************************************
 * getConfigSpec()
 ******************************************************************/
Tk_ConfigSpec *SigDB::getConfigSpec()
{
    static Tk_ConfigSpec   sdbConfigSpec[] = {
        {TK_CONFIG_STRING, "-ddb", "ddb", "DDB",
            (char *)NULL, Tk_Offset(SigDB, ddbName), 0},
        {TK_CONFIG_STRING, "-dfio", "dfio", "DFIO",
            (char *)NULL, Tk_Offset(SigDB, dfioName), 0},
        {TK_CONFIG_STRING, "-sdb_id", "sdb_id", "SdbId", 
            (char *)NULL, Tk_Offset(SigDB, sdbId),    0},
        {TK_CONFIG_END, (char *)NULL, (char *)NULL, (char *)NULL,
            (char *)NULL, 0, 0}
    };
    return sdbConfigSpec;
}

/*****************************************************************************
 *****************************************************************************
 ****                      Tcl Interface Functions
 *****************************************************************************
 *****************************************************************************/

/******************************************************************
 * Sdb_InstCmd()
 ******************************************************************/
static int Sdb_InstCmd(
    ClientData        clientData,
    Tcl_Interp       *interp,
    int               argc,
    char            **argv)
{
    SigDB    *sdb = (SigDB *)clientData;
    return sdb->InstCmd(argc, argv);
}


