/****************************************************************************
 *                                SimMgr.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements a manager for multiple Simulators
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "SimMgr.h"
#include "WidgetManager.h"
#include "CallbackMgr.h"
#include <vector.h>
#include "ivi_String.h"
#include "ConfigDB.h"
#include "CmdSwitcher.h"
#include <tcl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>


SimMgr *SimMgr::Globl_SimMgr = 0;

/**********************************************************
 * SimMgr()
 **********************************************************/
SimMgr::SimMgr()
{
    simList       = new Vector<SimInfo>();
}

/**********************************************************
 * GetSimMgr()
 **********************************************************/
SimMgr *SimMgr::GetSimMgr()
{
    if (!Globl_SimMgr) {
        Globl_SimMgr = new SimMgr();
    }

    return Globl_SimMgr;
}

/**********************************************************
 * SimMgr_RegisterSim()
 **********************************************************/
Int32 SimMgr_RegisterSim(
        SimInfo    *sim
        )
{
    SimMgr *dMgr = SimMgr::GetSimMgr();

    dMgr->simList->append(sim);
    return 0;
}

typedef enum {
    SimCmd_SimList=1,
    SimCmd_GetSuffixes,
    SimCmd_NumCmds
};

static CmdSwitchStruct simmgr_cmds[] = {
    { "sim_list",                SimCmd_SimList     },
    { "get_suffixes",            SimCmd_GetSuffixes },
    { "",                        0                  }
};

/**********************************************************
 * SimMgr_TclCmd()
 **********************************************************/
static int SimMgr_TclCmd(
        ClientData         clientData,
        Tcl_Interp        *interp,
        int                argc,
        char             **argv)
{
    Int32         cmd;
    Tcl_Obj      *list;
    SimMgr       *mgr = SimMgr::GetSimMgr();
    Int32         i;
    SimInfo      *info;

    if (argc < 2) {
        Tcl_AppendResult(interp, "too few commands - expect 2", 0);
        return TCL_ERROR;
    }

    cmd = CmdSwitch(simmgr_cmds, argv[1]);

    switch (cmd) {
        case SimCmd_SimList:
            list = Tcl_NewListObj(0, 0);

            for (i=0; i<mgr->simList->length(); i++) {
                info = mgr->simList->idx(i);

                Tcl_ListObjAppendElement(interp, list, 
                        Tcl_NewStringObj(info->getName().value(), -1));
            }

            Tcl_SetObjResult(interp, list);
            break;

        /************************************************************
         * SimCmd_GetSuffixes
         *
         * <sim_mgr> <get_suffixes> <sim_name>
         ************************************************************/
        case SimCmd_GetSuffixes:

            if (argc < 3) {
                Tcl_AppendResult(interp, "too few args - expect 3", 0);
                return TCL_ERROR;
            }

            info = 0;
            for (Uint32 i=0; i<mgr->simList->length(); i++) {
                info = mgr->simList->idx(i);

                if (info->getName() == argv[2]) {
                    break;
                }
                info = 0;
            }

            if (!info) {
                Tcl_AppendResult(interp, "simulator ", argv[2], 
                        " doesn't exist", 0);
                return TCL_ERROR;
            }

            Tcl_SetObjResult(interp, Tcl_DuplicateObj(info->getSuffixList()));
            break;

        default:
            Tcl_AppendResult(interp, "unknown command ", argv[1], 0);
            return TCL_ERROR;
            break;
    }

    return TCL_OK;
}

/**********************************************************
 * Sim_TclCmd()
 **********************************************************/
static int Sim_TclCmd(
        ClientData         clientData,
        Tcl_Interp        *interp,
        int                argc,
        char             **argv)
{
    const char  *typeName = 0;
    Char        *logFile  = 0;
    IviSim      *sim = 0;
    Char        *pt_argv[256];
    Uint32       pt_argc = 0;
    Uint32       i;
    SimMgr      *dMgr = SimMgr::GetSimMgr();
    SimInfo     *dInfo = 0;

    for (i=0; i<argc; i++) {
        if (!strcmp(argv[i], "-type")) {
            typeName = argv[++i];
        } else {
            pt_argv[pt_argc++] = argv[i];
        }
    }

    if (!typeName) {
        typeName = ConfigDB_GetCurrent("App.DefaultSim");
    }

    for (i=0; i<dMgr->simList->length(); i++) {
        dInfo = dMgr->simList->idx(i);
        if (dInfo->getName().equal(typeName)) {
            break;
        }
        dInfo = 0;
    }

    if (dInfo) {
        sim = dInfo->newSim(interp, pt_argc, pt_argv);
        if (!sim->ok) {
            delete sim;
            return TCL_ERROR;
        }
    } else {
        if (typeName) {
            Tcl_AppendResult(interp, "simulator ", typeName, 
                    " doesn't exist", 0);
        } else {
            Tcl_AppendResult(interp, "no simulators registered", 0);
        }

        return TCL_ERROR;
    }

    return TCL_OK;
}

static const char *prv_SimCbTypes[] = {
    CBTYPE_SIM_RUNSTEP_START,
    CBTYPE_SIM_RUNSTEP_UPDATE,
    CBTYPE_SIM_RUNSTEP_END,
    CBTYPE_SIM_DESIGN_LOAD,
    CBTYPE_SIM_DESIGN_CLOSE,
    ""
};

/**********************************************************
 * SimMgr_Init()
 **********************************************************/
extern "C" int SimMgr_Init(Tcl_Interp *interp)
{
    Uint32 idx = 0;

    SimMgr::GetSimMgr();
    Tcl_CreateCommand(interp, "sim", (Tcl_CmdProc *)Sim_TclCmd, 0, 0);
    Tcl_CreateCommand(interp, "sim_mgr", (Tcl_CmdProc *)SimMgr_TclCmd, 0, 0);
    WidgetMgr_AddType(interp, WIDGET_TYPE_IVI_SIM);

    while (prv_SimCbTypes[idx][0]) {
        CallbackMgr_AddCbType(prv_SimCbTypes[idx]);
        idx++;
    }

    return TCL_OK;
}


