#*****************************************************************************
#*                          DdbStruct.tcl
#*
#* Author: Matthew Ballance
#* Desc:   File implements a structure browser that is a client of a DDB
#*
#*
#* <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
#*
#*    This source code is free software; you can redistribute it
#*    and/or modify it in source code form under the terms of the GNU
#*    General Public License as published by the Free Software
#*    Foundation; either version 2 of the License, or (at your option)
#*    any later version.
#*
#*    This program is distributed in the hope that it will be useful,
#*    but WITHOUT ANY WARRANTY; without even the implied warranty of
#*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*    GNU General Public License for more details.
#*
#*    You should have received a copy of the GNU General Public License
#*    along with this program; if not, write to the Free Software
#*    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
#*
#* </Copyright>
#*
#*****************************************************************************

namespace eval DdbStruct {

    variable configspec {
        {-ddb        ""    }
        {-sdb        ""    }
        {-font       "-*-arial-medium-r-normal--*-*-*-*-*-*-*"}
        {-command    ""    }
        {-rmb_cmd    ""    }
        {-browsecmd  ""    }
        {-sepchar    "."   }
        {-sb_width   12    }
    }
}

#********************************************************************
#* DdbStruct
#********************************************************************
proc DdbStruct::DdbStruct {path args} {
    variable configspec
    global   CallbackTypes
    global   ivi_global

    frame $path

    rename $path ::$path:cmd
    proc ::$path {cmd args} "return \[eval DdbStruct::cmd $path \$cmd \$args\]"

    array set $path {_dummy _dummy}
    upvar #0 $path data

    #**** Setup widget defaults...
    foreach spec $configspec {
        set data([lindex $spec 0]) [lindex $spec 1]
    }

    eval DdbStruct::Configure $path $args

    set data(w:t_frame) [frame $path.t_frame]
    set data(w:b_frame) [frame $path.b_frame]
    set data(w:p_frame) [frame $data(w:b_frame).p_frame]

    set data(w:tree)     [tree_widget $data(w:t_frame).tree \
        -background white \
        -redraw 0 -opencmd [list DdbStruct::opencmd $path]  \
        -browsecmd [list DdbStruct::selectcmd $path]]

    set data(w:vsb)     [scrollbar $data(w:t_frame).vsb \
        -orient vertical]
    set data(w:hsb)     [scrollbar $data(w:b_frame).vsb \
        -orient horizontal]


    pack $data(w:vsb) -side right -fill y
    pack $data(w:tree) -side right -expand yes -fill both

    pack $data(w:p_frame) -side right -fill y
    pack $data(w:hsb)     -side right -fill x -expand yes


    pack $data(w:b_frame) -side bottom -fill x
    pack $data(w:t_frame) -side bottom -fill both -expand yes

    bind $data(w:vsb) <Configure> [list DdbStruct::padconfig $path]

    $data(w:tree) configure -yscrollcommand [list $data(w:vsb) set] \
                            -xscrollcommand [list $data(w:hsb) set]
    $data(w:hsb)  configure -command [list $data(w:tree) xview]
    $data(w:vsb)  configure -command [list $data(w:tree) yview]

    set data(update_cb) [callback add $CallbackTypes(DDB_UPDATE) $data(-ddb) \
         [list DdbStruct::DdbUpdateCB $path]]

    if {$data(-ddb) != ""} {
        DdbStruct::UpdateTree $path
    }

    bind $path <Destroy> "DdbStruct::Destroy $path"

    return $path
}

#********************************************************************
#* Destroy
#********************************************************************
proc DdbStruct::Destroy {path} {
    upvar #0 $path data
    global   CallbackTypes

    callback destroy $data(update_cb)
}

#********************************************************************
#* selectcmd
#********************************************************************
proc DdbStruct::selectcmd {path args} {
    upvar #0 $path data

    if {$data(-browsecmd) != ""} {
        eval $data(-browsecmd) [lindex $args 1]
    }
}

#********************************************************************
#* opencmd
#********************************************************************
proc DdbStruct::opencmd {path node} {
    upvar #0 $path data

    DdbStruct::UpdateTreeHelper $path $node $node 0
}

#*********************************************************
#* padconfig
#*********************************************************
proc DdbStruct::padconfig {path} {
    upvar #0 $path data

    $data(w:p_frame) configure -width [winfo width $data(w:vsb)]
}

#*********************************************************
#* cmd
#*********************************************************
proc DdbStruct::cmd {path cmd args} {
    upvar #0 $path data

    if {$cmd == "cget"} {
        set arg [lindex $args 0]
        set val [array get data $arg]
        if {$val == ""} {
            error "no option \"$arg\""
        }
        return $data($arg)
    } elseif {$cmd == "configure" || $cmd == "config"} {
        eval DdbStruct::Configure $args
    } elseif {$cmd == "selected"} {
        return [DdbStruct::selected $path]
    } else {
        error "unknown DdbStruct sub-cmd \"$cmd\""
    }
}

#*********************************************************
#* Configure
#*********************************************************
proc DdbStruct::Configure {path args} {
    upvar #0 $path data

    while {[llength $args] > 0} {
        set arg [lindex $args 0]

        set val [array get data $arg]
        if {$val == ""} {
            error "no option \"$arg\""
        }

        if {[info commands DdbStruct::config$arg] != ""} {
            set data($arg) [DdbStruct::config$arg $path [lindex $args 1]]
        } else {
            set data($arg) [lindex $args 1]
        }

        set args [lrange $args 2 end]
    }
}

#*********************************************************
#* DdbUpdateCB
#*********************************************************
proc DdbStruct::DdbUpdateCB args {
    set w [lindex $args 0]
    DdbStruct::UpdateTree $w
}

#*********************************************************
#* selected
#*********************************************************
proc DdbStruct::selected {w} {
    upvar #0 $w data

    return [$data(w:tree) selection get]
}

#*********************************************************
#* UpdateTreeHelper
#*********************************************************
proc DdbStruct::UpdateTreeHelper {w des_pth p_node recurse} {
    upvar #0 $w data
    set children ""

    foreach elem [$data(-ddb) glob -modules "$des_pth$data(-sepchar)*"] {

        set p "$p_node$data(-sepchar)$elem"
        if {[$data(w:tree) exists $p] == 1} {
            return
        }

        set nlist [$data(-ddb) glob -modules \
            "$des_pth$data(-sepchar)$elem$data(-sepchar)*"]
        if {[llength $nlist] > 0} {
            set drawcross always
        } else {
            set drawcross never
        }

        $data(w:tree) insert end $p_node "$p_node$data(-sepchar)$elem" \
            -text $elem -drawcross $drawcross

        if {$recurse == 1} {
            DdbStruct::UpdateTreeHelper $w "$des_pth$data(-sepchar)$elem" \
                "$des_pth$data(-sepchar)$elem" 1
        }
    }
}

#*********************************************************
#* UpdateTree
#*********************************************************
proc DdbStruct::UpdateTree {path} {
    upvar #0 $path data

    $data(w:tree) delete root

    $data(w:tree) configure -redraw false

    foreach elem [$data(-ddb) glob -modules $data(-sepchar)] {

        $data(w:tree) insert end root $data(-sepchar)$elem -text $elem  \
            -open 1

        set children [DdbStruct::UpdateTreeHelper $path \
            "$data(-sepchar)$elem" "$data(-sepchar)$elem" 0]
    }

    $data(w:tree) configure -redraw true
}

#*********************************************************
#* config-ddb
#*********************************************************
proc DdbStruct::config-ddb {w ddb} {
    upvar #0 $w data

    set data(-ddb) $ddb

    if {$ddb != ""} {
        DdbStruct::UpdateTree $w
    }

    return $ddb
}


