/****************************************************************************
 *                             ShmDFIOFile.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements access to shared-mem files
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "ShmDFIOFile.h"
#include "ivi_print.h"
#include <sys/types.h>
#include <sys/stat.h>

#ifndef __MINGW32__
#include <sys/mman.h>
#endif

#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <memory.h>

#undef  DEBUG_DFIO_FILE
#define FP stderr

#ifdef DEBUG_DFIO_FILE
#define DBG_MSG(x) fprintf x ; fflush(FP)
#else
#define DBG_MSG(x)
#endif

/********************************************************************
 * ShmDFIOFile()
 ********************************************************************/
ShmDFIOFile::ShmDFIOFile(const char *filename, bool read)
{
    d_base    = 0;
    d_fd      = 0;
    d_read    = read;
    d_maplen  = 0;

#ifndef __MINGW32__
    if (read) {
        d_fd = open(filename, O_RDWR);
    } else {
        d_fd = open(filename, O_RDWR|O_CREAT, S_IRWXU);
    }

    if (d_fd < 0) {
        fprintf(stderr, "ERROR: Cannot open \"%s\" for %s\n",
                filename, (read)?"reading":"writing");
        fflush(stderr);
        return;
    }
#else
    DWORD           fmaccess = 0;
    OSVERSIONINFO   os_info;
    char           *size_str;
    DWORD           shm_size;

    os_info.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);

    GetVersionEx(&os_info);

    /*** Decide which version of Windows we're running on... 
     *** 
     *** NT-like systems do fine with treating a mem-mapped file as
     *** coherent, shared memory. 95/98 do not. On 95/98, it is necessary
     *** to use a named, fixed-size file mapping...
     ***/
    if ((os_info.dwMajorVersion >= 5) && !getenv("IVI_FORCE_NAMED_SHM")) {
        d_useNamedShm = false;
        ivi_print("NOTE: Using anonymous (NT-style) "
                  "shared-memory mapping\n");
    } else {
        d_useNamedShm = true;
        ivi_print("NOTE: Using named (95/98-style) "
                  "shared-memory mapping\n");
    }

    if (d_useNamedShm) {
        if ((size_str = getenv("IVI_WIN9X_SHM_SIZE"))) {
            shm_size = strtoul(size_str, 0, 16);
         
            if (shm_size > 0x40000000) {
                ivi_print("WARNING: shm_size %s exceeds 0x40000000\n",
                    size_str);
            }
        } else {
            ivi_print("WARNING: IVI_WIN9X_SHM_SIZE env var isn't set - "
                      "default to 16MB wave file\n");
            shm_size = 0x01000000;
        }
    }

    if (!d_read) {
        fmaccess = PAGE_READWRITE;
        d_fd = CreateFile(filename, GENERIC_READ|GENERIC_WRITE,
                (FILE_SHARE_READ|FILE_SHARE_WRITE), 0, CREATE_ALWAYS, 0, 0);

        if (!d_fd || d_fd == INVALID_HANDLE_VALUE) {
            fprintf(stderr, "ERROR: Cannot open \"%s\" for %s - err=%d\n",
                filename, (read)?"reading":"writing", GetLastError());
            fflush(stderr);
        }

        /*** If we're using named shared memory, then create a named
         *** file-mapping object...
         ***/
        if (d_useNamedShm) {
            d_mapHandle = CreateFileMapping(d_fd, 0, fmaccess, 0, 
                shm_size, "ivi_waves3");

            if (!d_mapHandle || d_mapHandle == INVALID_HANDLE_VALUE) {
                ivi_print("CreateFileMapping(length=0x%08x) "
                    "failed - error=%d\n", shm_size, GetLastError());
            }
        }
    } else {
        if (d_useNamedShm) {
            d_mapHandle = OpenFileMapping(FILE_MAP_ALL_ACCESS, 
                FALSE, "ivi_waves3");

            if (!d_mapHandle || d_mapHandle == INVALID_HANDLE_VALUE) {
                ivi_print("OpenFileMapping() failed - error=%d\n", 
                    GetLastError());
            }
        } else {
            /*** Otherwise, open a file for reading... ***/
            d_fd = CreateFile(filename, GENERIC_READ|GENERIC_WRITE,
                (FILE_SHARE_READ|FILE_SHARE_WRITE), 0, OPEN_EXISTING, 0, 0);
            if (!d_fd || d_fd == INVALID_HANDLE_VALUE) {
                fprintf(stderr, "ERROR: Cannot open \"%s\" "
                    "for %s - err=%d\n",
                    filename, (read)?"reading":"writing", GetLastError());
                fflush(stderr);
            }
        }
    }
#endif

    ok = true;
}

/********************************************************************
 * ~ShmDFIOFile()
 ********************************************************************/
ShmDFIOFile::~ShmDFIOFile()
{
    unmap();

#ifdef __MINGW32__
    CloseHandle(d_fd);
#else

#endif
}

/********************************************************************
 * map()
 ********************************************************************/
int ShmDFIOFile::map(Uint32 maplen)
{
    int    pos;
    Uint32 tmpMapLen = d_maplen;
#ifdef __MINGW32__
    DWORD  fmaccess = 0;
    DWORD  shaccess = 0;
#endif
    int    ret;

    DBG_MSG((FP, "----> map(%d)\n", maplen));

    if (d_base) {
        unmap();
    }

#ifndef __MINGW32__
    if (!d_read) {
        Uchar   *buf;
        lseek(d_fd, tmpMapLen, SEEK_SET);

        buf = (Uchar *)malloc(maplen-tmpMapLen);
        memset(buf, 0, maplen-tmpMapLen);

        write(d_fd, buf, maplen-tmpMapLen);

        free(buf);
    } else {
        lseek(d_fd, maplen, SEEK_SET);
    }
#else
    /*** On windows, we can do file-extension the same way 
     *** regardless of whether we're open for read or write
     ***/

    /*** If we're not using the named (fixed-sized) file mapping, then 
     *** we must increase the size of the file...
     ***/
    if (!d_useNamedShm) {
        ret = SetFilePointer(d_fd, maplen, 0, FILE_BEGIN);

        if (ret != maplen) {
            ivi_print("SetFilePointer(%d) failed - error=%d\n",
                maplen, GetLastError());
        }

        if (!SetEndOfFile(d_fd)) {
            if (!d_read) {
                ivi_print("SetEndOfFile(type=%s, length=%d) failed "
                    "- error=%d\n",
                    (d_read)?"read":"write", maplen, GetLastError());
            }
        }
    }
#endif

#ifndef __MINGW32__
    d_base = (char *)mmap(0, maplen, PROT_READ|PROT_WRITE,
            MAP_SHARED, d_fd, 0);

    if (!d_base || ((int)d_base) == -1) {
        perror("cannot map file");
        return -1;
    }
#else

    if (!d_read) {
        shaccess = FILE_MAP_READ | FILE_MAP_WRITE;
        fmaccess = PAGE_READWRITE;

        /*** If we're not using named shared memory, then we need to 
         *** create a mapping for the file...
         ***/
        if (!d_useNamedShm) {
            d_mapHandle = CreateFileMapping(d_fd, 0, fmaccess, 0, 
                maplen, 0);

            if (!d_mapHandle || d_mapHandle == INVALID_HANDLE_VALUE) {
                ivi_print("CreateFileMapping(length=0x%08x) "
                    "failed - error=%d\n", maplen, GetLastError());
            } else {
               // ivi_print("CreateFileMapping(length=0x%08x) succeeded\n");
            }
        }
    } else {
        shaccess = FILE_MAP_READ;
        fmaccess = PAGE_READWRITE;

        if (!d_useNamedShm) {
            d_mapHandle = CreateFileMapping(d_fd, 0, fmaccess, 0, 
                maplen, 0);

            if (!d_mapHandle || d_mapHandle == INVALID_HANDLE_VALUE) {
                ivi_print("CreateFileMapping(length=0x%08x) "
                    "failed - error=%d\n", maplen, GetLastError());
            }
        }
    }

    d_base = (char *)MapViewOfFile(d_mapHandle, FILE_MAP_ALL_ACCESS, 
            0, 0, maplen);

    if (!d_base) {
        ivi_print("MapViewOfFile(length=%d) failed - error=%d\n",
                maplen, GetLastError());
    }

    if (!d_read) {
        memset(&d_base[tmpMapLen], 0, maplen-tmpMapLen);
    }
#endif

    d_maplen = maplen;

    DBG_MSG((FP, "<---- map(%d)\n", maplen));

    return 0;
}

/********************************************************************
 * unmap()
 ********************************************************************/
int ShmDFIOFile::unmap()
{
    int ret;

    if (!d_base) {
        return -1;
    }

    DBG_MSG((FP, "----> ShmDFIOFile::unmap()\n"));
    DBG_MSG((FP, "\tunmapping %d bytes\n", d_maplen));

#ifndef __MINGW32__
    msync(d_base, d_maplen, MS_SYNC);

    ret = munmap(d_base, d_maplen);

    if (ret < 0) {
        fprintf(stderr, "ERROR: cannot unmap\n");
        return -1;
    }
#else
    if (!UnmapViewOfFile((void *)d_base)) {
        fprintf(stderr, "ERROR: UnmapViewOfFile() failed\n");
        fflush(stderr);
    }

    /*** If we're using NT-style shared-memory mapping, then it's 
     *** necessary to close the FileMapping object
     ***/
    if (!d_useNamedShm) {
        if (!CloseHandle(d_mapHandle)) {
            ivi_print("ERROR: CloseHandle(d_mapHandle) failed\n");
        }
        d_mapHandle = 0;
    }
#endif

    d_base   = 0;
    d_maplen = 0;

    DBG_MSG((FP, "<---- ShmDFIOFile::unmap()\n"));
    return 0;
}

/********************************************************************
 * dump()
 ********************************************************************/
void ShmDFIOFile::dump(const char *dumper, int bytes)
{
    int i;

    ivi_print("ShmDFIOFile::dump(%s) - start=0x%08x\n", dumper, d_base);

    for (i=0; i<bytes; i++) {
        if (!(i % 8)) {
            ivi_print("\n\t");
        }
        ivi_print("%02x ", d_base[i]);
    }
    ivi_print("\n");
}



