/****************************************************************************
 *                          TreeWidgetNode.h
 *
 * Author: Matthew Ballance
 * Desc:   Describes a node in the tree-widget tree
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#ifndef INCLUDED_TREE_WIDGET_NODE_H
#define INCLUDED_TREE_WIDGET_NODE_H
#include "types.h"
#include "ivi_String.h"
#include "vector.h"
#include "PixmapObj.h"

class TreeWidget;
class TreeWidgetNode {

    friend class TreeWidget;

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************
         * TreeWidgetNode()
         ************************************************************/
        TreeWidgetNode();

        /************************************************************
         * TreeWidgetNode(text)
         ************************************************************/
        TreeWidgetNode(const char *text);

        /************************************************************
         * TreeWidgetNode(text, data)
         ************************************************************/
        TreeWidgetNode(const char *text, const char *data);

        /************************************************************
         * ~TreeWidgetNode()
         ************************************************************/
        virtual ~TreeWidgetNode();

        /************************************************************
         * SetParent()
         ************************************************************/
        virtual void SetParent(TreeWidget *parent);

        /************************************************************
         * SetParentNode()
         ************************************************************/
        virtual void SetParentNode(TreeWidgetNode *parent);

        /************************************************************
         * GetDimensions()
         ************************************************************/
        virtual void GetDimensions(Uint32 &width, Uint32 &height);

        /************************************************************
         * DrawNode()
         ************************************************************/
        virtual void DrawNode(PixmapObj &pixmap, Uint32 x_off, Uint32 y_off);

        /************************************************************
         * SetText()
         ************************************************************/
        virtual void SetText(const char *text);

        /************************************************************
         * SetData()
         ************************************************************/
        virtual void SetData(const char *text) { d_nodeData = text; }

        /************************************************************
         * SetUserData()
         ************************************************************/
        virtual void SetUserData(voidPtr userData) { d_userData = userData; }

        /************************************************************
         * Configure()
         ************************************************************/
        virtual int Configure(Tcl_Interp *interp,
                Uint32 argc, Tcl_Obj *const objv[]);

        /************************************************************
         * Cget()
         ************************************************************/
        virtual int Cget(Tcl_Interp *interp, char *option);

        /************************************************************
         * IsOverCheck()
         *
         * Returns true if the point is over the check-mark
         ************************************************************/
        virtual bool IsOverCheck(Uint32 x, Uint32 y);

        /************************************************************
         * IsOpen()
         ************************************************************/
        bool IsOpen() const { return d_expanded; }

        /************************************************************
         * DrawCheck()
         ************************************************************/
        bool DrawCheck() { return
            (((d_cross == CP_Auto) && d_children.length()) || 
                 (d_cross == CP_Always)); }

        /************************************************************
         * DrawBranch()
         ************************************************************/
        bool DrawBranch();

        /************************************************************
         * Open()
         ************************************************************/
        virtual void Open(bool is_open);

        /************************************************************
         * getChildren();
         ************************************************************/
        Vector<TreeWidgetNode> *getChildren() { return &d_children; }

    /****************************************************************
     * Public Data
     ****************************************************************/
    public:

    /****************************************************************
     * Protected Methods
     ****************************************************************/
    protected:

        enum CrossPolicy {
            CP_Auto,
            CP_Always,
            CP_Never
        };

    /****************************************************************
     * Protected Data
     ****************************************************************/
    protected:
        TreeWidget        *d_parent;
        TreeWidgetNode    *d_parentNode;
        String             d_nodeName;
        String             d_nodeText;
        String             d_nodeData;
        voidPtr            d_userData;

        CrossPolicy        d_cross;

    /****************************************************************
     * Private Methods
     ****************************************************************/
    private:
        void base_init();

    /****************************************************************
     * Private Data
     ****************************************************************/
    private:
        Uint32                   d_DimValid;
        Uint32                   d_ItemWidth;
        Uint32                   d_ItemHeight;

        Uint32                   d_ItemSetHeight;

        Vector<TreeWidgetNode>   d_children;

        Uint32                   d_indent;

        int                      d_expanded;

        Uint32                   d_chkBoxX;
        Uint32                   d_chkBoxY;
        Uint32                   d_chkBoxW;
        Uint32                   d_chkBoxH;

        Int32                    d_selected;

        Uint32                   d_idx;
        Int32                    d_drawBranch;
};

#endif /* INCLUDED_TREE_WIDGET_NODE_H */

