/***************************************************************************
                          zplutils.cpp  -  description
                             -------------------
    begin                : Son Okt 12 2003
    copyright            : (C) 2003 by Dominik Seichter
    email                : domseichter@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "zplutils.h"

// Qt includes
#include <qbuffer.h>
#include <qcstring.h>
#include <qimage.h>
#include <qpaintdevicemetrics.h>
#include <qregexp.h>
#include <qstring.h>
#include <qtextstream.h>

// font table for IPL, thanks to Erich Kitzmueller
struct { int size; int c; int h; int w; } iplfonttable[] = {
    { 4, 7, 1, 1 },
    { 5, 0, 1, 1 },
    { 6, 1, 1, 1 },
    { 7, 2, 1, 1 },
    { 8, 20, 1, 1 },
    { 10, 24, 1, 1 },
    { 11, 23, 1, 1 },
    { 12, 21, 1, 1 },
    { 14, 2, 2, 2 },
    { 16, 20, 2, 2 },
    { 19, 22, 1, 1 }
};


ExportUtils::ExportUtils( QTextStream* stream, QPaintDevice* source )
    : t( stream ), src( source )
{

}

int ExportUtils::convertX( int x )
{
    QPaintDeviceMetrics p( src );

    return ( x * res ) / p.logicalDpiX();
}

int ExportUtils::convertY( int y )
{
    QPaintDeviceMetrics p( src );

    return ( y * res ) / p.logicalDpiY();
}

ZPLUtils::ZPLUtils( QTextStream* stream, QPaintDevice* source )
    : ExportUtils( stream, source )
{
    /* See ^JM command */
    setResolution( 304 );

    *t << "^FXLabel created by KBarcode www.kbarcode.net" << endl;
    *t << "^XA" << endl; // Label start
    *t << "^JMA" << endl; // set printer to 304dpi
    *t << "^LH0,0" << endl; // set label origin to 0, 0

    //TODO: SEE ^JM for resolution
    if( !encodings.count() )
        fillEncodings();
}

QMap<QString,QString> ZPLUtils::encodings;

void ZPLUtils::setTextField( int x, int y, const QString & text )
{
    setFieldOrigin( x, y );
    setFont();
    setFieldData( QString( text ).replace( QRegExp("<[^>]*>"), "" )  );
}

void ZPLUtils::setBarcode( int x, int y, int, const QString & value, const QString & type )
{
    if( encodings [type].isEmpty() )
        qDebug( "ERROR: No ZPL barcode found");
        
    setFieldOrigin( x, y );
    *t << "^B" + encodings[type];
    setFieldData( value );
}

void ZPLUtils::setFieldOrigin( int x, int y )
{
    *t << QString("^FO%1,%2").arg( convertX( x ) ).arg( convertY( y ) ); // field origin
}

void ZPLUtils::setFont( QString font )
{
    // valid fonts:
    // A - H, GS
    
    if( font.isNull() )
        font = "D";

    *t << "^A" + font; // select font
}

void ZPLUtils::setFieldData( const QString & data )
{
    *t << "^FD" <<  data << "^FS" << endl; // field data
}

void ZPLUtils::setImage( int x, int y, const QImage* image )
{
    QCString data;
    QBuffer buffer( data );
    buffer.open( IO_WriteOnly );

    QImageIO io( &buffer, "PNG" );
    // create a black and white image
    io.setImage( image->convertDepth( 1 ) );
    io.write();
    buffer.close();

    setFieldOrigin( x, y );
    *t << "~DYD,p,P," + QString::number( data.size() ) + ",0," + QString ( data );
}

void ZPLUtils::setRect( int x, int y, const QSize size, bool circle, int width )
{
    setFieldOrigin( x, y );
    if( circle )
        setEllipse( size, width );
    else
        setRectangle( size, width );
}

void ZPLUtils::setRectangle( const QSize & size, int thick )
{
    *t << QString("~GB%1,%2,%3,B,0").arg( convertX( size.width() ) ).arg( convertY( size.height() ) ).arg( thick );
}

void ZPLUtils::setEllipse( const QSize & size, int thick )
{
    *t << QString("~GE%1,%2,%3,B").arg( convertX( size.width() ) ).arg( convertY( size.height() ) ).arg( thick );
}

void ZPLUtils::close()
{
    *t << "^XZ" << endl; // Label end
}

void ZPLUtils::fillEncodings()
{
    encodings.insert( "b1", "1" ); // Code11
    encodings.insert( "b2", "2" ); // Interlieved 2 of 5
    encodings.insert( "i25", "2" ); // Interlieved 2 of 5
    encodings.insert( "i25 -c", "2" ); // Interlieved 2 of 5 no checksum
    encodings.insert( "code39", "3" ); // code39
    encodings.insert( "code39 -c", "3" ); // code39 no checksum
    encodings.insert( "b8", "3" ); // code39
    encodings.insert( "b9", "3" ); // code39
    encodings.insert( "pdf417", "7" ); // PDF417
    encodings.insert( "b55", "7" ); // PDF417
    encodings.insert( "b10", "8" ); // EAN8
    encodings.insert( "b11", "8" ); // EAN8
    encodings.insert( "b12", "8" ); // EAN8
    encodings.insert( "ean", "8" ); // EAN8
    encodings.insert( "b37", "9" ); // UPC-E
    encodings.insert( "b38", "9" ); // UPC-E
    encodings.insert( "b39", "9" ); // UPC-E
    encodings.insert( "upc", "9" ); // UPC-E
    encodings.insert( "code93", "A" ); // Code 93
    encodings.insert( "b25", "A" ); // Code 93
    encodings.insert( "code128", "C" ); // Code 128
    encodings.insert( "code128b", "C" ); // Code 128
    encodings.insert( "code128c", "C" ); // Code 128
    encodings.insert( "b20", "C" ); // Code 128
    encodings.insert( "b59", "C" ); // Code 128
    encodings.insert( "b60", "C" ); // Code 128
    encodings.insert( "b61", "C" ); // Code 128
    encodings.insert( "b57", "D" ); // Maxicode
    encodings.insert( "ean", "E" ); // EAN 13
    encodings.insert( "isbn", "E" ); // EAN 13
    encodings.insert( "b13", "E" ); // EAN 13
    encodings.insert( "b14", "E" ); // EAN 13
    encodings.insert( "b15", "E" ); // EAN 13
    encodings.insert( "b56", "F" ); // micro PDF417
    encodings.insert( "b2", "JF" ); // 2 of 5 Standard
    encodings.insert( "cbr", "K" ); // codabar
    encodings.insert( "b18", "K" ); // codabar
    encodings.insert( "b19", "K" ); // codabar
    encodings.insert( "b50", "L" ); // LOGMARS
    encodings.insert( "msi", "M" ); // MSI
    encodings.insert( "b47", "M" ); // MSI
    encodings.insert( "pls", "P" ); // Plessey
    encodings.insert( "b46", "P" ); // Plessey
    encodings.insert( "b58", "Q" ); // QR Code
    encodings.insert( "upc", "U" ); // UPC A
    encodings.insert( "b34", "U" ); // UPC A
    encodings.insert( "b35", "U" ); // UPC A
    encodings.insert( "b36", "U" ); // UPC A
    encodings.insert( "b71", "X" ); // Datamatrix
    encodings.insert( "b40", "Z" ); // Postnet
    encodings.insert( "b41", "Z" ); // Postnet
    encodings.insert( "b42", "Z" ); // Postnet
    encodings.insert( "b43", "Z" ); // Postnet
    encodings.insert( "b44", "Z" ); // Postnet
    encodings.insert( "b45", "Z" ); // Postnet
}

/***************************************************************************/

IPLUtils::IPLUtils( QTextStream* stream, QPaintDevice* source )
    : ExportUtils( stream, source )
{
    setResolution( 300 );
    counter = 0;

    // start form definition:
    // ---
    // set the printer into propram mode
    setField( "<ESC>P" );
    // set darkness to 0 (range -10 to 10)
    setField( "<SI>d0" );
    // set print speed
    setField( "<SI>S20" );
    // erase format 3, create format 3
    // a Intermec printer can store several "formats" (form definitions), we choose abitrarely number 3
    setField( "E3;F3;" );

    if( !encodings.count() )
        fillEncodings();
}

QMap<QString,QString> IPLUtils::encodings;


void IPLUtils::setTextField( int x, int y, const QString & text )
{
    // TODO: parse text field HTML

    QString t = QString( text ).replace( QRegExp("<[^>]*>"), "" );
    
    QString s = QString("H%1;").arg( counter ); // field number
    counter++;

    s += setFieldOrigin( x, y );
    s += QString("c%1;").arg( 2 ); // font
    s += QString("h%1;").arg( 2 ); // vertical magnification ("height")
    s += QString("w%1;").arg( 2 ); // horicontyl magnification ("width")
    s += QString("d0,%1;").arg( t.length() ); // max length of data !
    setField( s );

    values.append( t );
}

void IPLUtils::setBarcode( int x, int y, int h, const QString & value, const QString & type )
{
    if( encodings [type].isEmpty() )
        qDebug( "ERROR: No IPL barcode found");

    QString s = QString("B%1;").arg( counter ); // field number
    counter++;

    s += setFieldOrigin( x, y );    
    s += QString("c%1;").arg( encodings[type] ); // encoding type
    s += QString("h%1;").arg( convertY( h ) ); // height of barcode
    s += QString("w%1;").arg( 3 ); // width of barcode (per line)
    s += QString("d0,%1;").arg( 30 ); // max length of data
    setField( s );

    values.append( value );
}

void IPLUtils::setRect( int x, int y, const QSize size, bool, int width )
{
    QString s = QString("W%1;").arg( counter ); // field number
    counter++;

    s += setFieldOrigin( x, y );
    s += QString("l%1;").arg( size.width() ); // box length
    s += QString("h%1;").arg( size.height() ); // box height
    s += QString("w%1;").arg( width );

    setField( s );
}

void IPLUtils::setField( const QString & data )
{
    *t << "<STX>" << data << "<ETX>" << endl;
}

void IPLUtils::close()
{
    // set the printer to print mode
    setField( "R" );
    // start with actual data
    setField( "<ESC>E3<CAN>" );  // choose format number 3

    for( unsigned int i = 0; i < values.count(); i++ )
        setField( values[i] + ( i != values.count() - 1 ? "<CR>" : QString::null ) );
    
    // end actual data
    setField( "<ETB>" );
    
}

QString IPLUtils::setFieldOrigin( int x, int y )
{
    QString s = QString("o%1,%2;f0;").arg( convertX( x ) ).arg( convertY( y ) ); // field origin
    return s;
}

void IPLUtils::setImage( int, int, const QImage* )
{
    qDebug("No image support in IPL!");
}


void IPLUtils::fillEncodings()
{
    encodings.insert( "code39", "0" ); // code39
    encodings.insert( "code39 -c", "0" ); // code39 no checksum
    encodings.insert( "b8", "0" ); // code39
    encodings.insert( "b9", "0" ); // code39
    encodings.insert( "code93", "1" ); // Code 93
    encodings.insert( "b25", "1" ); // Code 93
    encodings.insert( "b2", "2" ); // Interlieved 2 of 5
    encodings.insert( "i25", "2" ); // Interlieved 2 of 5
    encodings.insert( "i25 -c", "2" ); // Interlieved 2 of 5 no checksum
    encodings.insert( "b2", "3" ); // 2 of 5 Standard
    encodings.insert( "cbr", "4" ); // codabar
    encodings.insert( "b18", "4" ); // codabar
    encodings.insert( "b19", "4" ); // codabar
    encodings.insert( "b1", "5" ); // Code11    
    encodings.insert( "code128", "6" ); // Code 128
    encodings.insert( "code128b", "6" ); // Code 128
    encodings.insert( "code128c", "6" ); // Code 128
    encodings.insert( "b20", "6" ); // Code 128
    encodings.insert( "b59", "6" ); // Code 128
    encodings.insert( "b60", "6" ); // Code 128
    encodings.insert( "b61", "6" ); // Code 128
    encodings.insert( "b10", "7" ); // EAN8
    encodings.insert( "b11", "7" ); // EAN8
    encodings.insert( "b12", "7" ); // EAN8
    encodings.insert( "ean", "7" ); // EAN8
    encodings.insert( "b37", "7" ); // UPC-E
    encodings.insert( "b38", "7" ); // UPC-E
    encodings.insert( "b39", "7" ); // UPC-E
    encodings.insert( "upc", "7" ); // UPC-E
    encodings.insert( "ean", "7" ); // EAN 13
    encodings.insert( "isbn", "7" ); // EAN 13
    encodings.insert( "b13", "7" ); // EAN 13
    encodings.insert( "b14", "7" ); // EAN 13
    encodings.insert( "b15", "7" ); // EAN 13
    encodings.insert( "upc", "7" ); // UPC A
    encodings.insert( "b34", "7" ); // UPC A
    encodings.insert( "b35", "7" ); // UPC A
    encodings.insert( "b36", "7" ); // UPC A
    encodings.insert( "b40", "10" ); // Postnet
    encodings.insert( "b41", "10" ); // Postnet
    encodings.insert( "b42", "10" ); // Postnet
    encodings.insert( "b43", "10" ); // Postnet
    encodings.insert( "b44", "10" ); // Postnet
    encodings.insert( "b45", "10" ); // Postnet
    encodings.insert( "pdf417", "12" ); // PDF417
    encodings.insert( "b55", "12" ); // PDF417
    encodings.insert( "b57", "14" ); // Maxicode
    encodings.insert( "b71", "17" ); // Datamatrix
    encodings.insert( "b58", "18" ); // QR Code
    encodings.insert( "b56", "19" ); // micro PDF417    
}
