#include <string.h>

#include <glib.h>
#include <glib-object.h>

#include "kpsettings.h"
#include "kpplugin.h"
#include "kputil.h"

static GList *plugins_loaded = NULL;
static GList *plugins = NULL;

static gboolean file_is_plugin (const gchar *file);
static void kp_plugin_class_init (GObjectClass * klass,
                                  gpointer data);
static void kp_plugin_instance_init (GObject * object,
                                     gpointer data);
static void kp_plugin_instance_finalize (GObject * object);

GType
kp_plugin_get_type (void)
{
  static GType kp_plugin_type = 0;

  if (!kp_plugin_type) {
    static const GTypeInfo kp_plugin_info = {
      sizeof (KPPluginClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) kp_plugin_class_init,
      (GClassFinalizeFunc) NULL,
      NULL,
      sizeof (KPPlugin),
      0,
      (GInstanceInitFunc) kp_plugin_instance_init,
      NULL
    };

    kp_plugin_type = g_type_register_static (G_TYPE_OBJECT,
                                            "KPPlugin",
                                            &kp_plugin_info,
                                             0);
  }
  return kp_plugin_type;
}

static void
kp_plugin_class_init (GObjectClass * klass,
                      gpointer data)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_plugin_instance_finalize;
}

static void
kp_plugin_instance_init (GObject * object,
                         gpointer data)
{
  KPPlugin *plugin = KP_PLUGIN (object);
  plugin->loaded = FALSE;
}

static void
kp_plugin_instance_finalize (GObject * object)
{
  GObjectClass *parent_class;

  parent_class = g_type_class_peek_parent (G_OBJECT_GET_CLASS (object));
  parent_class->finalize (object);
}

/**
 * kp_plugin_new: 
 * @file: Full path to the plugin file.
 *
 * Creates a new #KPPlugin object and prepare to load it later.
 * This function should probably be called from this module only.
 *
 * Returns: a #KPPlugin.
 */
KPPlugin *
kp_plugin_new (const gchar *file)
{
  gboolean (*kp_plugin_init)(KPPlugin *);
  KPPlugin *plugin;
 
  g_return_val_if_fail (file != NULL, NULL);

  plugin = g_object_new (kp_plugin_get_type (), NULL);

  plugin->handle = g_module_open (file, 0);
  plugin->fn = g_strdup (file);
   
  if (plugin->handle == NULL) {
    g_warning ("Can't open plugin %s, %s\n", plugin->fn, g_module_error ());
    return FALSE;
  }

  if (!g_module_symbol (plugin->handle, "kp_plugin_init",
                       (gpointer)&kp_plugin_init)) {
    g_module_close (plugin->handle);
    plugin->handle = NULL;

    g_warning ("Can't load plugin %s, %s\n", plugin->fn, g_module_error ());
    return FALSE;
  }
  
  if (!kp_plugin_init (plugin) || !plugin->info) {
    g_object_unref (plugin);
    return NULL;
  }

  return KP_PLUGIN (plugin);
}


/**
 * kp_plugin_load:
 * @plugin: a #KPPlugin
 *
 * Runs a plugin that is registered earlier.
 *
 * Returns: TRUE if successful and FALSE otherwise.
 */
gboolean
kp_plugin_load (KPPlugin *plugin)
{
  g_return_val_if_fail (KP_IS_PLUGIN (plugin), FALSE);
  
  if (plugin->loaded)
    return TRUE;

  plugins_loaded = g_list_append (plugins_loaded, plugin);
  
  if (plugin->info && plugin->info->load) {
    plugin->info->load (plugin);
    plugin->loaded = TRUE;
    
    kp_debug ("Plugin loaded.\n");
    return TRUE;
  }

  return FALSE;
}

/**
 * kp_plugin_load_plugins:
 *
 * Load all the plugins that are in the config file
 * and marked for loading at the startup.
 */
void
kp_plugin_load_plugins (void)
{
  GSList *list = kp_settings_get_list ("load_plugin");

  while (list) {
    kp_plugin_load_by_name (list->data);
    list = list->next;
  }
  kp_settings_list_free (list);
}
 

/**
 * kp_plugin_load_by_name:
 * @name: The name of the plugin
 *
 * Load a plugin. See kp_plugin_load () for more.
 *
 * Returns: TRUE if ok, FALSE otherwise.
 */
gboolean
kp_plugin_load_by_name (const gchar *name)
{
  GList *list = plugins;
  g_return_val_if_fail (name != NULL, FALSE);
 
  while (list) {
    if (strcmp (KP_PLUGIN (list->data)->info->name, name) == 0)
      return kp_plugin_load (KP_PLUGIN (list->data));
    list = list->next;
  }
  return FALSE;
}

/**
 * kp_plugin_unload:
 * @plugin: The plugin
 *
 * Unload the plugin. That means, the plugin will be removed from
 * the list of loaded plugins and all memory allocated for the plugin
 * will be freed, but KPPlugin structure must be destroyed too. That
 * should probably be done by g_object_unref ().
 */
void
kp_plugin_unload (KPPlugin *plugin)
{
  g_return_if_fail (KP_IS_PLUGIN (plugin));
  
  if (!plugin->loaded)
    return;

  plugins_loaded = g_list_remove (plugins_loaded, plugin);
  
  if (plugin->info && plugin->info->unload) {
    plugin->info->unload (plugin);
    plugin->loaded = FALSE;
    
    kp_debug ("Plugin unloaded.\n");
  }
}


/**
 * kp_plugin_unload_by_name:
 * @name: the name of the plugin
 *
 * Unloads the plugin with name @name if found from the list
 * of loaded plugins. See kp_plugin_unload () for more information.
 */
void
kp_plugin_unload_by_name (const gchar *name)
{
  GList *list = plugins;
  g_return_if_fail (name != NULL);
  
  while (list) {
    if (strcmp (KP_PLUGIN (list->data)->info->name, name) == 0) {
      kp_plugin_unload (KP_PLUGIN (list->data));
      return;
    }
    list = list->next;
  }
}


/**
 * kp_plugin_scan:
 * @dir: The directory to scan or NULL to scan default directories.
 *
 * Scans directory or directories for loadable plugins
 * and adds them to list of plugins.
 */
void
kp_plugin_scan (const gchar *dir)
{
  KPPlugin *plugin;
  const gchar *file;
  gchar *tmpfile;

  GList *dirs = NULL;
  GList *list;
  GList *tmp;
  GDir *d;
  
  if (!dir) {
    dirs = g_list_append (dirs, g_strdup (KIPINA_PLUGIN_DIR));
    dirs = g_list_append (dirs, g_build_path (G_DIR_SEPARATOR_S,
                          g_get_home_dir (), ".kipina", "plugins", NULL));
  } else
    dirs = g_list_append (dirs, g_strdup (dir));
 
  list = dirs;
  
  while (list) {
    d = g_dir_open (list->data, 0, NULL);

    /* We are not interested in why, if opening the directory fails, because
     * plugins are not mandatory */
    
    while (d && (file = g_dir_read_name (d))) {
      if (file_is_plugin (file)) {
        tmpfile = g_strdup_printf ("%s%c%s", (gchar *) list->data,
                                   G_DIR_SEPARATOR, file);

        plugin = kp_plugin_new (tmpfile);
        
        kp_debug ("Found plugin: %s\n", plugin->fn);

        g_free (tmpfile);
      }
    }
    tmp = list;
    list = list->next;

    g_free (tmp->data);
    g_list_free_1 (tmp);
  }
}

gboolean
kp_plugin_register (KPPlugin *plugin)
{
  g_return_val_if_fail (KP_IS_PLUGIN (plugin), FALSE);

  if (plugin->info)
    kp_debug ("Registering plugin, id: %s\n", plugin->info->id);
  
  if (g_list_find (plugins, plugin))
    return TRUE;

  plugins = g_list_append (plugins, plugin);
  
  return TRUE;
}

GList *
kp_plugin_get_plugins (void)
{
  return plugins;
}

GList *
kp_plugin_get_plugins_loaded (void)
{
  return plugins_loaded;
}

static gboolean
file_is_plugin (const gchar *file)
{
  return g_str_has_suffix (file, ".so");
}
