/* @(#) network.c 1.38 @(#) */
/***************************************************************\
*	Copyright (c) 1999-2000 First Step Internet Services, Inc.
*		All Rights Reserved
*	Distributed under the BSD Licenese
*
*	Module: NETWORK
\***************************************************************/

#define _KOALAMUD_NETWORK_C "@(#) nitehawk@localhost.1ststep.net|lib/koala/network.c|20000827025248|08933 @(#)"

#include "autoconf.h"

#include "version.h"
#include "koalatypes.h"
#include "memory.h"
#include "network.h"
#include "log.h"
#include "llist.h"

/* module local prototypes */
daemonmagic magic;
koalaerror netcreatemagic(void);

/* netstartup
 * 	Startup networking
 */
koalaerror netstartup(void)
{
	netcreatemagic();
	return KESUCCESS;
}

/* netlisten
 *    Startup network listen sockets
 *    Returns negative on error.
 *    Returns listen socket on success
 */
/* AF Independant Version */
#if AFINDEP
/* With the AF Independant code, we can create more then one descriptor.  This
 *	happens when IPv6 is enabled and configured on the system
 */
koalaerror netlisten(listnodeptr list, char *bindaddr, int port)
{
	struct addrinfo hints, *res, *tmp;
	char portstr[PORTSTRLEN];
	pdescriptor desc = NULL;
	int socketcount = 0;
	int error = 0;
	char *name = NULL;  // If bindaddr is not any, we will point there before
						// doing the lookups
	
	if (!list)
	{
		logmsg(LOGERR, "Caught NULL list pointer!");
		return -KEMISSINGARG;
	}

	/* Prepare hints struct */
	memset(&(hints), 0, sizeof(hints));
	hints.ai_family = PF_UNSPEC;
	hints.ai_flags = AI_PASSIVE;
	hints.ai_socktype = SOCK_STREAM;

	if (bindaddr != NULL)
	{
		/* If the addr is *not* any, resolve it */
		if (strcasecmp(bindaddr, "any"))
		{
			name = bindaddr;
		}
	}

	/* Get a string version of the port */
	snprintf(portstr, PORTSTRLEN, "%d", port);

	/* Get address information */
	if ((error = getaddrinfo(name, portstr, &hints, &res)) != 0)
	{
		logmsg(LOGERR, "Error looking up address info");
		return KEBIND;
	}

	/* Now we need to loop through the 'res' list and bind and listen */
	tmp = res;	// Need to hold the head pointer to free later 
	while(tmp)
	{
		/* Allocate memory for new descriptor */
		if ((desc = allocdescriptor()) == NULL)
		{
			logmsg(LOGCRIT, "Unable to allocate memory for descriptor");
			return KENOMEM;
		}
		desc->type = DESCRIPTOR_LISTEN;

		/* Create listen socket */
		desc->socket = socket(tmp->ai_family, tmp->ai_socktype,
				tmp->ai_protocol);
		if (desc->socket == -1)
		{
			logmsg(LOGERR, "Error %d creating listen socket", errno);
			return -KENOPROTO;
		}

		ksetsocketopts(desc->socket);

		/* Bind to the ip6 addresses */
		if (bind(desc->socket, tmp->ai_addr, tmp->ai_addrlen) < 0)
		{
			logmsg(LOGERR, "Error binding port");
			close(desc->socket);
			return -KEBIND;
		}

		/* Start listening */
		listen(desc->socket, LISTENBACKQUEUE);

		/* Add the descriptor to the list */
		listaddnode(list, desc);
		
		/* We have another listener */
		socketcount++;
		
		/* Move to the next ai struct */
		tmp = tmp->ai_next;
	}

	logmsg(LOGINFO, "Daemon now listening with %d sockets on port %d",
		socketcount, port);

	freeaddrinfo(res);
	
	return desc->socket;
}
#else /* Non-AF Independant version */
koalaerror netlisten(listnodeptr list, char *bindaddr, int port)
{
	struct protoent *protoentry = NULL;
	struct hostent *hp = NULL;
	struct sockaddr_in addr;
	pdescriptor desc;
	
	if (!list)
	{
		logmsg(LOGERR, "Caught NULL list pointer!");
		return -KEMISSINGARG;
	}

	/* Allocate memory */
	if ((desc = allocdescriptor()) == NULL)
	{
		logmsg(LOGCRIT, "Unable to allocate memory for descriptor");
		return KENOMEM;
	}

	desc->type = DESCRIPTOR_LISTEN;

	/* Get Protocol information */
	if ((protoentry = getprotobyname("tcp")) == NULL)
	{
		/* No protocol information available - Time to die */
		logmsg(LOGCRIT, "Unable to get tcp protocol information");
		return -KENOPROTO;
	}

	/* Create listen socket */
	desc->socket = socket(AF_INET, SOCK_STREAM, protoentry->p_proto);
	if (desc->socket == -1)
	{
		logmsg(LOGERR, "Error %d creating listen socket", errno);
		return -KENOPROTO;
	}

	ksetsocketopts(desc->socket);

	addr.sin_family = AF_INET;
	addr.sin_port = htons(port);
	addr.sin_addr.s_addr = htons(INADDR_ANY);

	if (bindaddr != NULL)
	{
		/* If the addr is *not* any, resolve it */
		if (strcasecmp(bindaddr, "any"))
		{
			addr.sin_addr.s_addr = inet_addr(bindaddr);
			if ((addr.sin_addr.s_addr & 0xffffffff) == 0xffffffff) {
				/* Resolve hostname to the ipaddress */
				hp = gethostbyname(bindaddr);
				if (!hp)
				{
					logmsg(LOGERR, "Unable to lookup hostname");
					close(desc->socket);
					freedescriptor(desc);
					return -KENOHOST;
				}
				if (!hp->h_addr_list[0]) {
					logmsg(LOGERR, "Host has no addresses");
					close(desc->socket);
					freedescriptor(desc);
					return -KENOHOST;
				}

				/* Set the address to connect to */
				addr.sin_family = hp->h_addrtype;
				memcpy(&addr.sin_addr, hp->h_addr_list[0],
						sizeof(addr.sin_addr));
			}
		}
	}
	
	/* Bind to the ip6 addresses */
	if (bind(desc->socket, (struct sockaddr *)&addr, sizeof(addr)) < 0)
	{
		logmsg(LOGERR, "Error binding port");
		close(desc->socket);
		freedescriptor(desc);
		return -KEBIND;
	}

	listen(desc->socket, 5);

	logmsg(LOGINFO, "Daemon now listening on port %d", port);
	
	/* Add descriptor to list */
	listaddnode(list, desc);

	return desc->socket;
}
#endif /* netlisten versions */

/* ksetsocketopts
 *	Set socket options for a network socket
 */
koalaerror ksetsocketopts(int sock)
{
#ifdef SO_LINGER
	// Turn off linger
	{
		struct linger l;
		l.l_onoff = 0;
		l.l_linger = 0;

		setsockopt(sock, SOL_SOCKET, SO_LINGER, &l, sizeof(l));
	}
#endif

#if GONONBLOCK
#if HAVE_FCNTL_H
#ifndef O_NONBLOCK
#ifdef O_NDELAY
#define O_NONBLOCK O_NDELAY
#endif
#endif
	{
		int sockflags;
		int result;

		sockflags = fcntl(sock, F_GETFL, 0);

		sockflags |= O_NONBLOCK;

		result = fcntl(sock, F_SETFL, sockflags);
		if (result == -1)
		{
			logmsg(LOGCRIT, "Error switching socket to Non-Blocking");
		}
	}
#endif
#endif

#ifdef SO_REUSEADDR
	{
		int optval = 1;
		setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &optval,
				sizeof(optval));
	}
#endif

	return KESUCCESS;
}

/* Accept new connection to server */
koalaerror netaccept(pdescriptor desc, pdescriptor *newconnection)
{
#if AFINDEP
	struct sockaddr_storage addr;
	char portstr[PORTSTRLEN];
#else
	struct sockaddr_in addr;
	char *addrbuffer;
#endif
	size_t addrlen = sizeof(addr);

	*newconnection = allocdescriptor();
	if (*newconnection == NULL)
	{
		logmsg(LOGCRIT, "Error allocating new descriptor");
		return KENOMEM;
	}

	(*newconnection)->socket = accept(desc->socket,
			(struct sockaddr *)&(addr),
			&(addrlen));
	if ((*newconnection)->socket < 0)
	{
		logmsg(LOGERR, "Error accepting connection");
		freedescriptor(*newconnection);
		return KENOACCEPT;
	}

	ksetsocketopts((*newconnection)->socket);

#if AFINDEP
	getnameinfo((struct sockaddr *)&addr, addrlen, (*newconnection)->hostname,
			MAXHOSTNAMELEN, portstr, PORTSTRLEN, NI_NUMERICSERV);
#else
	addrbuffer = inet_ntoa(addr.sin_addr);
	strncpy((*newconnection)->hostname, addrbuffer,
			MAXHOSTNAMELEN > strlen(addrbuffer) ? MAXHOSTNAMELEN :
			strlen(addrbuffer));
	free(addrbuffer);
#endif

	logmsg(LOGINFO, "Accepted new connection from %s",
			(*newconnection)->hostname);

	(*newconnection)->status = STATUS_CONNECTING;
	(*newconnection)->type = DESCRIPTOR_UNKNOWN;

	return KESUCCESS;
}

/* Close an open descriptor */
koalaerror netclose(pdescriptor desc)
{
	/* Unlink the descriptor from the list */
	/*desc->prev->next = desc->next;
	desc->next->prev = desc->prev;*/

	/* close the network socket */
	close(desc->socket);

	freedescriptor(desc);

	logmsg(LOGINFO, "Closing connection");

	return KESUCCESS;
}

/* netread -
 * 		Read data from the network
 * 		if the number read is less then the number available, we will append a
 * 		null terminator
 */
koalaerror netread(pdescriptor desc, char *buffer, unsigned int *len)
{
	int numread;

	/* Validate parameters */
	if (!desc || !buffer || !len)
	{
		logmsg(LOGERR, "Invalid pointer caught in netread");
		return KEMISSINGARG;
	}

	if (desc->type == DESCRIPTOR_NULL || desc->type == DESCRIPTOR_DUMMY
			|| desc->type == DESCRIPTOR_LISTEN)
	{
		logmsg(LOGERR, "Cannot read from this descriptor type");
		return KEBADDESCRIPTORTYPE;
	}

	numread = read(desc->socket, buffer, *len);
	if (numread == 0)
	{
		desc->status = STATUS_CLOSE;
	}
	if (numread < 0)
	{
		if (errno == EAGAIN)
		{
			numread = 0;
		}
		else
		{
			/* If there was an IO error, we just mark the socket to be closed in
			 * the next loop pass */
			desc->status = STATUS_CLOSE;
		}
	}

	if (numread < *len)
	{
		buffer[numread] = '\0';
		numread++;
	}

	*len = numread;
	
	return KESUCCESS;
}

/* Write data to a descriptor */
koalaerror netwrite(pdescriptor desc, char *data, unsigned int len)
{
	int retval;

	/* Validate parameters */
	if (!desc || !data)
	{
		logmsg(LOGERR, "Invalid pointer caught in netwrite");
		return KEMISSINGARG;
	}

	retval = write(desc->socket, data, len);

	if (retval < 0)
	{
		desc->status = STATUS_CLOSE;
	}

	return KESUCCESS;
}

/* Build a connection to a hub or zone server */
koalaerror netuplink(pdescriptor uplink, char *bindaddr, int port)
{
#if AFINDEP
	struct addrinfo hints, *ai;
	char portstr[PORTSTRLEN];
#else
	struct sockaddr_in hostaddr;
	struct hostent *hp = NULL;
	struct protoent *protoentry = NULL;
	char *namebuf;
#endif

	/* Make sure we got a pointer */
	if (uplink == NULL)
	{
		logmsg(LOGERR, "Caught null pointer in netuplink!");
		return -KEMISSINGARG;
	}

	/* Get address information */
#if AFINDEP
	/* Separate the two versions of uplink code.  AF_INET specific code will
	 * be removed at a later date */
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	snprintf(portstr, PORTSTRLEN, "%d", port);
	if (getaddrinfo(bindaddr, portstr, &hints, &ai) != 0)
	{
		logmsg(LOGERR, "Error getting address information, bail out.");
		return -KENOHOST;
	}

	/* Create a socket for the uplink */
	uplink->socket = socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol);
	if (uplink->socket == -1)
	{
		logmsg(LOGERR, "Error %d creating uplink socket", errno);
		return -KENOPROTO;
	}

	/* We have the socket - lets connect to our uplink */
	if (connect(uplink->socket, ai->ai_addr, ai->ai_addrlen) == -1)
	{
		logmsg(LOGERR, "Error connecting to uplink");
		close(uplink->socket);
		return -KENOUPLINK;
	}

	/* Get hostname and port information */
	getnameinfo(ai->ai_addr, ai->ai_addrlen, uplink->hostname, MAXHOSTNAMELEN,
			portstr, PORTSTRLEN, NI_NUMERICSERV);
	uplink->port = atoi(portstr);

	/* Make sure we don't leak memory */
	freeaddrinfo(ai);

#else /* AF Independant code not enabled */
	/* Get Protocol information */
	if ((protoentry = getprotobyname("tcp")) == NULL)
	{
		/* No protocol information available - Time to die */
		logmsg(LOGCRIT, "Unable to get tcp protocol information");
		return -KENOPROTO;
	}

	memset(&hostaddr, 0, sizeof(hostaddr));
	hostaddr.sin_family = AF_INET;
	hostaddr.sin_port = htons(port);
	hostaddr.sin_addr.s_addr = inet_addr(bindaddr);
	if ((hostaddr.sin_addr.s_addr & 0xffffffff) != 0xffffffff)
	{
		/* At this point, the addr struct should be fully populated */
		/* Create a socket to use for the uplink */
		uplink->socket = socket(AF_INET, SOCK_STREAM, protoentry->p_proto);
		if (uplink->socket == -1)
		{
			logmsg(LOGERR, "Error %d creating uplink socket", errno);
			return -KENOPROTO;
		}

		/* We have the socket - lets connect to our uplink */
		if (connect(uplink->socket, (struct sockaddr *)&hostaddr,
				 sizeof(hostaddr)) == -1)
		{
			logmsg(LOGCRIT, "Error connecting to uplink -"
					" refusing to run disconnected");
			close(uplink->socket);
			return -KENOUPLINK;
		}
	}
	else
	{
		/* Resolve hostname to the ipaddress */
		hp = gethostbyname2(bindaddr, AF_INET);
		if (!hp)
		{
			logmsg(LOGERR, "Unable to lookup hostname");
			close(uplink->socket);
			return -KENOUPLINK;
		}
		if (!hp->h_addr_list[0])
		{
			logmsg(LOGERR, "Host has no addresses");
			close(uplink->socket);
			return -KENOUPLINK;
		}

		/* Set the address to connect to */
		hostaddr.sin_family = hp->h_addrtype;
		memcpy(&hostaddr.sin_addr, hp->h_addr_list[0],
				sizeof(hostaddr.sin_addr));

		/* At this point, the addr struct should be fully populated */
		/* Create a socket to use for the uplink */
		uplink->socket = socket(AF_INET, SOCK_STREAM, protoentry->p_proto);
		if (uplink->socket == -1)
		{
			logmsg(LOGERR, "Error %d creating uplink socket", errno);
			return -KENOPROTO;
		}

		/* We have the socket - lets connect to our uplink */
		if (connect(uplink->socket, (struct sockaddr *)&hostaddr,
				 sizeof(hostaddr)) == -1)
		{
			logmsg(LOGERR, "Error connecting to uplink");
			close(uplink->socket);
			return -KENOUPLINK;
		}
	}

	/* Get hostname and port information */
	namebuf = inet_ntoa(hostaddr.sin_addr);
	strncpy(uplink->hostname, namebuf, MAXHOSTNAMELEN);
	free(namebuf);
#endif

	/* Set the type and status of the new uplink */
	uplink->type = DESCRIPTOR_UNKNOWN; 
	uplink->status = STATUS_CONNECTING;

	/* Now that the socket is connected, we leave it up to the daemon loop to
	 * make calls on partially connected uplinks to do the protocol
	 * negotiation as there is data on the socket.  See uplinkprotocol.c
	 */

	return KESUCCESS;
}

/* netcreatemagic -
 * 		This function creates 'magic' strings that are used during uplink
 * 		creation to determine the type of daemon that is connecting and to
 * 		ensure that the daemons are of compatible versions.  This function
 * 		should not be changed unless changes are made that invalidate the
 * 		linkage protocol.  If in doubt, ask Matthew first.
 */
koalaerror netcreatemagic(void)
{
	strncpy(magic.daemonmagic, "KOALA-PROTOCOL-101001-UPLINK", MAGICLEN);
	strncpy(magic.clientmagic, "KOALA-CLIENT-DAEMON-112412", MAGICLEN);
	strncpy(magic.zonemagic, "KOALA-ZONE-DAEMON-112412", MAGICLEN);
	strncpy(magic.hubmagic, "KOALA-HUB-DAEMON-112412", MAGICLEN);
	
	return KESUCCESS;
}
