// =============================================================================
//
//      --- kvi_irc_network.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviIrcNetwork"

#include "kvi_config.h"
#include "kvi_debug.h"
#include "kvi_irc_network.h"
#include "kvi_netutils.h"
#include "kvi_settings.h"
#include "kvi_string.h"

KviIrcNetwork::KviIrcNetwork(const char *szName)
{
	m_szName = szName;
	m_pServerList = new QPtrList<KviIrcServer>;
	m_pServerList->setAutoDelete(true);
	m_pCurrentServer = 0;
}

KviIrcNetwork::KviIrcNetwork(KviIrcNetwork *net)
{
	__range_valid(net);
	m_pServerList = new QPtrList<KviIrcServer>;
	m_pServerList->setAutoDelete(true);
	copyFrom(net);
}

KviIrcNetwork::~KviIrcNetwork()
{
	clear();
	if( m_pServerList ) {
		delete m_pServerList;
		m_pServerList = 0;
	}
}

QPtrList<KviIrcServer> *KviIrcNetwork::serverList()
{
	return m_pServerList;
}

void KviIrcNetwork::clear()
{
	m_szName = "UNNAMED";
	while( !m_pServerList->isEmpty() ) {
		KviIrcServer *s = m_pServerList->first();
		removeServer(s);
	}
	m_pCurrentServer = 0;
}

uint KviIrcNetwork::serverCount()
{
	return m_pServerList->count();
}

void KviIrcNetwork::appendServer(KviIrcServer *ptr)
{
	__range_valid(ptr);
	m_pServerList->append(ptr);
	m_pCurrentServer = ptr;
}

bool KviIrcNetwork::removeServer(KviIrcServer *ptr)
{
	__range_valid(ptr);
	bool bWasHere = m_pServerList->removeRef(ptr);
	__range_valid(bWasHere);
	if( ptr == m_pCurrentServer ) {
		if( m_pServerList->isEmpty() )
			m_pCurrentServer = 0;
		else
			m_pCurrentServer = m_pServerList->first();
	}
	return bWasHere;
}

bool KviIrcNetwork::setCurrentServer(KviIrcServer *ptr)
{
	__range_valid(ptr);
	if( m_pServerList->isEmpty() ) {
		m_pCurrentServer = 0;
		return false;
	}
	if( m_pServerList->findRef(ptr) == -1 ) return false;
	m_pCurrentServer = ptr;
	return true;
}

KviIrcServer *KviIrcNetwork::currentServer()
{
	if( m_pServerList->isEmpty() ) return 0;
	if( m_pCurrentServer ) {
		__range_valid(m_pServerList->findRef(m_pCurrentServer) != -1); // Sanity check
		return m_pCurrentServer;
	}
	m_pCurrentServer = m_pServerList->first();
	return m_pCurrentServer;
}

KviIrcServer *KviIrcNetwork::nextServer()
{
	if( m_pServerList->isEmpty() ) return 0;
	if( !m_pCurrentServer ) {
		m_pCurrentServer = m_pServerList->first();
		return m_pCurrentServer;
	}
	__range_valid(m_pServerList->findRef(m_pCurrentServer) != -1); // Sanity check
	m_pCurrentServer = m_pServerList->next();
	if( !m_pCurrentServer )
		m_pCurrentServer = m_pServerList->first();
	return m_pCurrentServer;
}

void KviIrcNetwork::serverFromString(KviStr &szSrv, KviIrcServer *srv)
{
	srv->addresses.clear();
	srv->szHost        = "irc.unknown.net";
	srv->szPassword    = "";
	srv->szDescription = "Unknown";
	srv->szPort        = "6667";
#ifdef COMPILE_NEED_IPV6
	srv->bIPv6         = false;
#endif

	const char *ptr = szSrv.ptr();
	if( (!*ptr) || (*ptr == ':') ) return;
	ptr = kvi_extractUpTo(srv->szHost, ptr, ':');
	if(  *ptr ) ptr++;
	if( !*ptr ) return;
	//
	// Create a list of IP addresses, separated by commas in the input string
	//
	QString addrs(ptr);
	addrs = addrs.mid(0, addrs.find(':'));
	ptr = (ptr + addrs.length());
	QStringList list = QStringList::split(",", addrs);
	QStringList::Iterator it = list.begin();
	for( ; it != list.end(); ++it ) {
		QString s = *it;
		s.replace('#', ":");
		QHostAddress addr;
		if( addr.setAddress(s) )
			srv->addresses.append(addr);
	}
	if(  *ptr ) ptr++;
	if( !*ptr ) return;
	ptr = kvi_extractUpTo(srv->szPassword, ptr, ':');
	if(  *ptr ) ptr++;
	if( !*ptr ) return;
	ptr = kvi_extractUpTo(srv->szPort, ptr, ':');
	if(  *ptr ) ptr++;
	if( !*ptr ) return;
	ptr = kvi_extractUpTo(srv->szDescription, ptr, ':');
	if(  *ptr ) ptr++;
	if( !*ptr ) return;
#ifdef COMPILE_NEED_IPV6
	KviStr tmp;
	kvi_extractUpTo(tmp, ptr, ':');
	srv->bIPv6 = kvi_strEqualCI("IPv6", tmp.ptr());
#endif
}

void KviIrcNetwork::serverToString(KviStr &szSrv, KviIrcServer *srv)
{
	szSrv = (srv->szHost.hasData() ? srv->szHost.ptr() : "irc.unknown.net");
	szSrv.append(':');
	KviIpAddresses::Iterator it = srv->addresses.begin();
	for( ; it != srv->addresses.end(); ) {
		QString addr = (*it).toString();
		addr.replace(':', "#");
		szSrv.append(addr);
		++it;
		if( it != srv->addresses.end() )
			szSrv.append(",");
	}
	szSrv.append(':');
	szSrv.append(srv->szPassword);
	szSrv.append(':');
	szSrv.append(srv->szPort.hasData() ? srv->szPort.ptr() : "6667");
	szSrv.append(':');
	szSrv.append(srv->szDescription);
#ifdef COMPILE_NEED_IPV6
	szSrv.append(':');
	szSrv.append(srv->bIPv6 ? "IPv6" : "IPv4");
#endif
}

void KviIrcNetwork::save(KviConfig *cfg)
{
	cfg->writeEntry("Name",    m_szName.ptr());
	cfg->writeEntry("Servers", m_pServerList->count());
	KviStr szSrv, szNum;
	for( uint i = 0; i < m_pServerList->count(); i++ ) {
		KviIrcServer *srv = m_pServerList->at(i);
		if( srv == m_pCurrentServer ) cfg->writeEntry("CurrentServer", i);
		__range_valid(srv);
		serverToString(szSrv, srv);
		szNum.sprintf("Server_%d", i);
		cfg->writeEntry(szNum.ptr(), szSrv.ptr());
	}
}

void KviIrcNetwork::load(KviConfig *cfg)
{
	m_szName = cfg->readEntry("Name", "UnknownNet");
	uint count = cfg->readUIntEntry("Servers", 0);
	m_pCurrentServer = 0;
	uint curr = cfg->readUIntEntry("CurrentServer", 0);
	KviStr szSrv, szNum;
	for( uint i = 0; i < count; i++ ) {
		KviIrcServer *srv = new KviIrcServer;
		__range_valid(srv);
		szNum.sprintf("Server_%d", i);
		szSrv = cfg->readEntry(szNum.ptr(), "irc.unknown.net:127.0.0.1::6667:Unknown:IPv4");
		serverFromString(szSrv, srv);
		m_pServerList->append(srv);
		if( curr == i) m_pCurrentServer = srv;
	}
	if( !m_pCurrentServer ) {
		if( !m_pServerList->isEmpty() )
			m_pCurrentServer = m_pServerList->first();
	}
}

void KviIrcNetwork::copyFrom(KviIrcNetwork *net)
{
	__range_valid(net);
	clear();
	m_szName = net->m_szName.ptr();
	m_pCurrentServer = 0;
	for( KviIrcServer *s = net->m_pServerList->first(); s; s = net->m_pServerList->next() ) {
		KviIrcServer  *v = new KviIrcServer();
		v->szHost        = s->szHost.ptr();
		v->addresses     = s->addresses;
		v->szDescription = s->szDescription.ptr();
		v->szPassword    = s->szPassword.ptr();
		v->szPort        = s->szPort.ptr();
#ifdef COMPILE_NEED_IPV6
		v->bIPv6         = s->bIPv6;
#endif
		if( s == net->m_pCurrentServer ) m_pCurrentServer = v;
		m_pServerList->append(v);
	}
	__range_valid((m_pCurrentServer || m_pServerList->isEmpty()) && (!(m_pCurrentServer && m_pServerList->isEmpty())));
}

KviIrcServer *KviIrcNetwork::getServerByName(const char *szServer)
{
	if( m_pServerList->isEmpty() ) return 0;
	for( KviIrcServer *s = m_pServerList->first(); s; s = m_pServerList->next() ) {
		if( kvi_strEqualCI(szServer, s->szHost.ptr()) ) return s;
	}
	return 0;
}

char *KviIrcNetwork::name()
{
	return m_szName.ptr();
}

void KviIrcNetwork::setName(const char *name)
{
	m_szName = name;
}

void KviIrcNetwork::updateServerIp(const char *szServer, KviIpAddresses list)
{
	KviIrcServer *srv = getServerByName(szServer);
	if( !srv ) return;
#ifdef COMPILE_NEED_IPV6
	if( srv->bIPv6 ) {
		KviIpAddresses::Iterator it = list.begin();
		for( ; it != list.end(); ++it ) {
			QString s = (*it).toString();
			if( !kvi_isValidStringIP_v6(s.ascii()) ) {
				srv->bIPv6 = false;
				break;
			}
		}
	}
#endif
	srv->addresses = list;
}

void KviIrcNetwork::sortServers()
{
	m_pServerList->setAutoDelete(false);
	QPtrList<KviIrcServer> l(*m_pServerList); // Shallow copy
	l.setAutoDelete(false);
	while( !m_pServerList->isEmpty() ) m_pServerList->removeLast();
	KviStr sz1, sz2;
	while( !l.isEmpty() ) {
		KviIrcServer *s = l.first();
		getServerCompareString(sz1, s);
		uint idx = 0;
		bool bDone = false;
		for( KviIrcServer *is = m_pServerList->first(); is && !bDone; is = m_pServerList->next() ) {
			getServerCompareString(sz2, is);
			if( qstricmp(sz1.ptr(), sz2.ptr()) > 0 ) {
				// The new one is smaller than the current one
				m_pServerList->insert(idx, s);
				bDone = true;
			}
			idx++;
		}
		if( !bDone ) m_pServerList->append(s);
		l.removeFirst();
	}
	m_pServerList->setAutoDelete(true);
}

void KviIrcNetwork::getServerCompareString(KviStr &szStr, KviIrcServer *srv)
{
	int idx = srv->szHost.findLastIdx('.');
	if( idx >= 0 ) {
		szStr = srv->szHost.right(srv->szHost.len() - idx);
		szStr += srv->szHost.left(idx);
	} else szStr = srv->szHost.ptr();
}
