#ifndef _KVI_USERPARSER_H_INCLUDED_
#define _KVI_USERPARSER_H_INCLUDED_

// =============================================================================
//
//      --- kvi_userparser.h ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#include <qobject.h>
#include <qptrlist.h>

#include "kvi_string.h"

class KviAlias;
class KviAsyncDialog;
class KviAsyncWhois;
class KviCommand;
class KviConsole;
class KviDns;
class KviExprTree;
class KviFrame;
class KviInput;
class KviIrcSocket;
class KviProcess;
class KviScriptObject;
class KviScriptObjectController;
class KviSlaveIO;
class KviUserParser;
class KviUserPopupMenu;
class KviVariableCache;
class KviWindow;

#define KVI_GLOBAL_VAR_CHAR    '%'
#define KVI_COMMENT_BEGIN_CHAR '#'

typedef struct KviDnsData;

typedef struct KviPluginCommandHandler;
typedef struct KviPluginFunctionHandler;

typedef struct KviWindowTimer
{
	KviStr     job;
	KviStr     name;
	int        id;
	int        timeout;
	KviWindow *window;
	KviStr     magic;
	bool       singleShot;
};

typedef bool (KviUserParser::*commandParseProc) (KviCommand *);
typedef bool (KviUserParser::*identParseProc)   (KviCommand *, KviStr &);

typedef struct KviCommandEntry
{
	const char       *cmdName;
	int               cmdLen;
	commandParseProc  cmdProc;
};

typedef struct KviIdentifierEntry
{
	const char     *idnName;
	identParseProc  idnProc;
};

typedef struct KviFunctionEntry
{
	const char     *fncName;
	identParseProc  fncProc;
};

typedef struct KviCommandErrorInfo
{
	int    iLineNum;
	int    iCharNum;
	KviStr szLineStr;
};

class KviUserParser : public QObject
{
	friend class KviFrame;           // Only KviFrame can construct it
	friend class KviCommand;
	friend class KviAsyncDialog;     // Only this can access m_szLastDialogMagic & co.
	friend class KviAsyncWhois;
	friend class KviApp;             // Access to m_bLastIsRemoteExec etc...
	friend class KviScriptObject;    // Access to m_pConsole member
	friend class KviScriptObjectController;
	friend class KviInput;           // Access to KviCommandEntry tables

	Q_OBJECT
public:
	KviScriptObjectController *m_pScriptObjectController;
	KviAsyncWhois             *m_pAsyncWhoisController;

	void setLastSignalSender(const char *);
	void unregisterTimersFor(KviWindow *);

	int completeCommand             (KviStr &word, KviStr &match, KviStr &multiple);
	int completeFunctionOrIdentifier(KviStr &word, KviStr &match, KviStr &multiple);

	void setup(KviIrcSocket *, KviConsole *);

	void parseUserCommand(QString &str, KviWindow *);
	bool parseCommand(const char *ptr, KviWindow *, const char *params = 0);
	void parseAsyncDialogCallback(KviStr &str, KviWindow *);
	bool processString(const char *ptr, KviWindow *, KviStr &buffer);

	bool callEvent(int index, KviWindow *, const KviStr &params);
	bool callRawEvent(
		const KviStr &szCommand, const KviStr &szPrefix, const KviStr &szBuffer,
		const KviStr &szParams, const KviStr &szSourceMask, KviStr &retValue
	);
	bool doAsyncDnsCallForHost(const char *host, const char *data = 0);
	void slaveAsyncProcessExited(
		KviStr &job, KviStr &commandline, KviStr &stdoutBuf, KviStr &stderrBuf,
		KviStr &magic, KviStr &window, unsigned int uPid, int iExitCode
	);
	void asyncWhoisEnd(
		KviStr &asyncCmd, KviStr &magic, KviStr &window, KviStr &nick, KviStr &user,
		KviStr &host, KviStr &realName, KviStr &server, KviStr &idle, KviStr &status,
		bool bIrcOp, KviStr &chans, bool bSuccess
	);
	bool triggerObjectEvent(KviScriptObject *, const char *eventName, const char *eventBuf, const KviStr &params);
	bool callObjectFunction(
		KviScriptObject *, const char *fncName, const char *fncBuf, QPtrList<KviStr> *params, KviStr &buffer
	);
public slots:
	void asyncDnsFinished(KviDnsData *);
protected slots:
	void processExited(KviProcess *);
protected:
	KviUserParser(KviFrame *);
	~KviUserParser();

	KviConsole               *m_pConsole;
	KviFrame                 *m_pFrm;
	KviIrcSocket             *m_pSocket;
	static KviCommandEntry    cmdTable_A_G[];
	static KviCommandEntry    cmdTable_H_N[];
	static KviCommandEntry    cmdTable_O_S[];
	static KviCommandEntry    cmdTable_T_Z[];

	static KviIdentifierEntry idnTable_A_G[];
	static KviIdentifierEntry idnTable_H_N[];
	static KviIdentifierEntry idnTable_O_S[];
	static KviIdentifierEntry idnTable_T_Z[];

	static KviFunctionEntry   fncTable_A_G[];
	static KviFunctionEntry   fncTable_H_N[];
	static KviFunctionEntry   fncTable_O_S[];
	static KviFunctionEntry   fncTable_T_Z[];

	KviStr m_szLastDialogResult;
	KviStr m_szLastDialogMagic;
	KviStr m_szLastTimerMagic;
	KviStr m_szLastTimerName;

	KviStr m_szLastProcCommandline;
	KviStr m_szLastProcMagic;
	KviStr m_szLastProcStdout;
	KviStr m_szLastProcStderr;
	KviStr m_szLastProcPid;
	KviStr m_szLastProcWindow;
	KviStr m_szLastProcExitCode;

	KviStr m_szLastAsyncWhoisNick;
	KviStr m_szLastAsyncWhoisUsername;
	KviStr m_szLastAsyncWhoisHost;
	KviStr m_szLastAsyncWhoisRealName;
	KviStr m_szLastAsyncWhoisServer;
	KviStr m_szLastAsyncWhoisIdleTime;
	KviStr m_szLastAsyncWhoisStatus;
	KviStr m_szLastAsyncWhoisIrcOp;
	KviStr m_szLastAsyncWhoisChannels;
	KviStr m_szLastAsyncWhoisWindow;
	KviStr m_szLastAsyncWhoisMagic;
	KviStr m_szLastAsyncWhoisSuccess;

	KviStr m_szLastTryResult;

	KviStr m_szLastSignalSender;

	KviStr m_szLastExecFilename;
	bool   m_bLastIsRemoteExec;

	QPtrList<KviAsyncDialog> *m_pAsyncDialogList;
	QPtrList<KviDns>         *m_pDnsList;
	QPtrList<KviWindowTimer> *m_pTimerList;
	KviSlaveIO               *m_pSlaveIOController;

	bool notConnectedToServer(KviCommand *, const char *inCommand);
	bool recoverableError(KviCommand *, int error, const char *cmdName, const char *additionalData = 0);

	static bool skipCommand          (KviCommand *);
	static bool skipCommandBlock     (KviCommand *);
	static bool skipSingleCommand    (KviCommand *);
	static bool skipIdentifier       (KviCommand *);
	static bool skipExpressionBody   (KviCommand *);
	static bool skipIf               (KviCommand *);
	static bool skipWhile            (KviCommand *);
	static bool skipDo               (KviCommand *);
	static bool skipSwitch           (KviCommand *);
	static bool skipAlias            (KviCommand *);
	static bool skipDialog           (KviCommand *);
	static bool skipPopup            (KviCommand *);
	static bool skipPopupBuffer      (KviCommand *);
	static bool skipForeach          (KviCommand *);
	static bool skipTimer            (KviCommand *);
	static bool skipExecv            (KviCommand *);
	static bool skipAwhois           (KviCommand *);
	static bool skipUtoolbar         (KviCommand *);
	static bool skipVariable         (KviCommand *);
	static bool skipVariableKey      (KviCommand *);
	static bool skipSwitches         (KviCommand *);
	static bool skipSingleSwitch     (KviCommand *);
	static bool skipTry              (KviCommand *);
	static bool skipObj_setEvent     (KviCommand *);
	static bool skipClass            (KviCommand *);
	static bool skipObjectScopeRValue(KviCommand *);
	static bool skipLValue           (KviCommand *);
	static bool skipComment          (KviCommand *);

	bool execCommandBuffer(KviCommand *);
	bool execCommandBlock (KviCommand *);
	bool execSingleCommand(KviCommand *);

	bool extractSwitches    (KviCommand *);
	bool extractSingleSwitch(KviCommand *);

	void getErrorInfo(KviCommand *, const char *, KviCommandErrorInfo *);
	void printError  (KviCommand *, const char *);

	bool processVariable           (KviCommand *, KviStr &buffer);
	bool processVariableKey        (KviCommand *, KviStr &buffer);
	bool processIdentifier         (KviCommand *, KviStr &buffer);
	bool processParameterIdentifier(KviCommand *, KviStr &buffer);

	bool processFncSingleParamInternal(KviCommand *, KviStr &buffer);
	bool processFncSingleParam        (KviCommand *, KviStr &buffer);
	bool processFncFinalPart          (KviCommand *, KviStr &buffer);
	bool processCmdSingleToken        (KviCommand *);
	bool processCmdFinalPart          (KviCommand *);
	bool processCmdFinalPartNoIdent   (KviCommand *);

	bool processObjectScopeRValue(KviScriptObject *, KviCommand *, KviStr &buffer);

	// In kvi_userpopupmenu.cpp
	bool generatePopupFromBuffer(KviUserPopupMenu *, KviCommand *, bool bAppend);

	// In kvi_command.cpp
	bool processOperator(KviCommand *, KviVariableCache *cache, KviStr &szVariable, KviStr &szKey);

	bool parseCmd_LVALUE           (KviCommand *);
	bool parseCmd_ECHO             (KviCommand *);
	bool parseCmd_ECHOPRIVMSG      (KviCommand *);
	bool parseCmd_HELP             (KviCommand *);
	bool parseCmd_PARSE            (KviCommand *);
	bool parseCmd_HALT             (KviCommand *);
	bool parseCmd_IF               (KviCommand *);
	bool parseCmd_WHILE            (KviCommand *);
	bool parseCmd_DO               (KviCommand *);
	bool parseCmd_DIALOG           (KviCommand *);
	bool parseCmd_ALIAS            (KviCommand *);
	bool parseCmd_DEBUG            (KviCommand *);
	bool parseCmd_SWITCH           (KviCommand *);
	bool parseCmd_QUIT             (KviCommand *);
	bool parseCmd_NICK             (KviCommand *);
	bool parseCmd_USER             (KviCommand *);
	bool parseCmd_POPUP            (KviCommand *);
	bool parseCmd_UTOOLBAR         (KviCommand *);
	bool parseCmd_RAW              (KviCommand *);
	bool parseCmd_FOREACH          (KviCommand *);
	bool parseCmd_QUERY            (KviCommand *);
	bool parseCmd_TOPIC            (KviCommand *);
	bool parseCmd_AME              (KviCommand *);
	bool parseCmd_AMSG             (KviCommand *);
	bool parseCmd_ANOTICE          (KviCommand *);
	bool parseCmd_ONOTICE          (KviCommand *);
	bool parseCmd_STATUS           (KviCommand *);
	bool parseCmd_TIMER            (KviCommand *);
	bool parseCmd_KILLTIMER        (KviCommand *);
	bool parseCmd_CONFIG           (KviCommand *);
	bool parseCmd_DIRBROWSER       (KviCommand *);
	bool parseCmd_BREAK            (KviCommand *);
	bool parseCmd_SAVEOPTIONS      (KviCommand *);

	bool parseCmd_JOIN             (KviCommand *);
	bool parseCmd_PART             (KviCommand *);
	bool parseCmd_MODE             (KviCommand *);
	bool parseCmd_UMODE            (KviCommand *);

	bool parseCmd_OP               (KviCommand *);
	bool parseCmd_DEOP             (KviCommand *);
	bool parseCmd_VOICE            (KviCommand *);
	bool parseCmd_DEVOICE          (KviCommand *);
	bool parseCmd_HALFOP           (KviCommand *);
	bool parseCmd_DEHALFOP         (KviCommand *);
	bool parseCmd_USEROP           (KviCommand *);
	bool parseCmd_DEUSEROP         (KviCommand *);
	bool parseCmd_OWNER            (KviCommand *);
	bool parseCmd_DEOWNER          (KviCommand *);
	bool parseCmd_BAN              (KviCommand *);
	bool parseCmd_UNBAN            (KviCommand *);
	bool parseCmd_EXCEPT           (KviCommand *);
	bool parseCmd_UNEXCEPT         (KviCommand *);

	bool parseCmd_LOG              (KviCommand *);
	bool parseCmd_OPTION           (KviCommand *);
	bool parseCmd_LOADCONF         (KviCommand *);
	bool parseCmd_CLEAR            (KviCommand *);

	bool parseCmd_KILL             (KviCommand *);
	bool parseCmd_OPER             (KviCommand *);
	bool parseCmd_SETHOST          (KviCommand *);
	bool parseCmd_REHASH           (KviCommand *);

	bool parseCmd_OPENURL          (KviCommand *);

	bool parseCmd_DESTROY          (KviCommand *);
	bool parseCmd_OBJ_SETEVENT     (KviCommand *);
	bool parseCmd_OBJ_CLEAREVENT   (KviCommand *);
	bool parseCmd_OBJ_TRIGGEREVENT (KviCommand *);
	bool parseCmd_CLEAROBJECTS     (KviCommand *);
	bool parseCmd_CLASS            (KviCommand *);

	bool parseCmd_PLAY             (KviCommand *);
	bool parseCmd_MULTIMEDIA       (KviCommand *);
	bool parseCmd_SOUND            (KviCommand *);
	bool parseCmd_MULTIMEDIAORSOUND(KviCommand *, const char *ctcpName);

	bool parseCmd_WHOIS            (KviCommand *);
	bool parseCmd_AWHOIS           (KviCommand *);
	bool parseCmd_WHOWAS           (KviCommand *);
	bool parseCmd_WHO              (KviCommand *);
	bool parseCmd_USERHOST         (KviCommand *);
	bool parseCmd_NAMES            (KviCommand *);
	bool parseCmd_BEEP             (KviCommand *);

	bool parseCmd_MSG              (KviCommand *);
	bool parseCmd_NOTICE           (KviCommand *);
	bool parseCmd_SAY              (KviCommand *);

	bool parseCmd_HELPOP           (KviCommand *);
	bool parseCmd_LOCOPS           (KviCommand *);
	bool parseCmd_GLOBOPS          (KviCommand *);
	bool parseCmd_WALLOPS          (KviCommand *);

	bool parseCmd_CTCP             (KviCommand *);
	bool parseCmd_CTCPREPLY        (KviCommand *);

	bool parseCmd_KICK             (KviCommand *);

	bool parseCmd_SERVER           (KviCommand *);
	bool parseCmd_NEWSWINDOW       (KviCommand *);
	bool parseCmd_ME               (KviCommand *);
	bool parseCmd_HOST             (KviCommand *);
	bool parseCmd_DCC              (KviCommand *);
	bool parseCmd_PLUGIN           (KviCommand *);
	bool parseCmd_FINDTEXT         (KviCommand *);

	bool parseCmd_DNS              (KviCommand *);
	bool parseCmd_SETRETURN        (KviCommand *);

	bool parseCmd_REGISTER         (KviCommand *);
	bool parseCmd_UNREGISTER       (KviCommand *);
	bool parseCmd_IGNORE           (KviCommand *);
	bool parseCmd_NOTIFY           (KviCommand *);
	bool parseCmd_SETCOMMENT       (KviCommand *);
	bool parseCmd_SETFLAGS         (KviCommand *);
	bool parseCmd_SETPASS          (KviCommand *);

	bool parseCmd_LINKS            (KviCommand *);
	bool parseCmd_MAP              (KviCommand *);
	bool parseCmd_LIST             (KviCommand *);
	bool parseCmd_MOTD             (KviCommand *);
	bool parseCmd_LUSERS           (KviCommand *);
	bool parseCmd_INFO             (KviCommand *);
	bool parseCmd_STATS            (KviCommand *);
	bool parseCmd_ADMIN            (KviCommand *);
	bool parseCmd_TIME             (KviCommand *);
	bool parseCmd_VERSION          (KviCommand *);
	bool parseCmd_LAG              (KviCommand *);

	bool parseCmd_CHANSERV         (KviCommand *);
	bool parseCmd_NICKSERV         (KviCommand *);
	bool parseCmd_MEMOSERV         (KviCommand *);
	bool parseCmd_OPERSERV         (KviCommand *);
	bool parseCmd_HELPSERV         (KviCommand *);
	bool parseCmd_STATSERV         (KviCommand *);

	bool parseCmd_DIR              (KviCommand *);
	bool parseCmd_CD               (KviCommand *);

	bool parseCmd_INVITE           (KviCommand *);

	bool parseCmd_RUN              (KviCommand *);
	bool parseCmd_WINDOW           (KviCommand *);
	bool parseCmd_GEOMETRY         (KviCommand *);

	bool parseCmd_EXEC             (KviCommand *);
	bool parseCmd_PROCLIST         (KviCommand *);
	bool parseCmd_KILLPROC         (KviCommand *);
	bool parseCmd_WRITEPROC        (KviCommand *);
	bool parseCmd_EXECV            (KviCommand *);

	bool parseCmd_TRY              (KviCommand *);

	bool parseCmd_WRITEFILE        (KviCommand *);

	bool parseCmd_CONNECT          (KviCommand *);
	bool parseCmd_DISCONNECT       (KviCommand *);

	bool parseMultipleModeCommand(KviCommand *, const char *cmdName, bool bModeAdd, char flag);

	bool parseAlias(KviCommand *, KviAlias *);

	bool parsePluginCommand   (KviPluginCommandHandler  *, KviCommand *);
	bool parsePluginFunction  (KviPluginFunctionHandler *, KviCommand *, KviStr &buffer);
	bool parsePluginIdentifier(KviPluginFunctionHandler *, KviCommand *, KviStr &buffer);

	bool parseUnknownCommand(KviStr &cmd, KviCommand *);

	// In kvi_identifiers.cpp
	bool parseIdn_ISCONNECTED        (KviCommand *, KviStr &buffer);
	bool parseIdn_WINDOW             (KviCommand *, KviStr &buffer);
	bool parseIdn_DIALOGRESULT       (KviCommand *, KviStr &buffer);
	bool parseIdn_DIALOGMAGIC        (KviCommand *, KviStr &buffer);
	bool parseIdn_TIMERMAGIC         (KviCommand *, KviStr &buffer);
	bool parseIdn_TIMERNAME          (KviCommand *, KviStr &buffer);
	bool parseIdn_CR                 (KviCommand *, KviStr &buffer);
	bool parseIdn_LF                 (KviCommand *, KviStr &buffer);
	bool parseIdn_K                  (KviCommand *, KviStr &buffer);
	bool parseIdn_B                  (KviCommand *, KviStr &buffer);
	bool parseIdn_U                  (KviCommand *, KviStr &buffer);
	bool parseIdn_R                  (KviCommand *, KviStr &buffer);
	bool parseIdn_O                  (KviCommand *, KviStr &buffer);
	bool parseIdn_TIME               (KviCommand *, KviStr &buffer);
	bool parseIdn_DATE               (KviCommand *, KviStr &buffer);
	bool parseIdn_DAY                (KviCommand *, KviStr &buffer);
	bool parseIdn_MONTH              (KviCommand *, KviStr &buffer);
	bool parseIdn_YEAR               (KviCommand *, KviStr &buffer);
	bool parseIdn_ME                 (KviCommand *, KviStr &buffer);
	bool parseIdn_MYUSERNAME         (KviCommand *, KviStr &buffer);
	bool parseIdn_MYREALNAME         (KviCommand *, KviStr &buffer);
	bool parseIdn_MYHOSTNAME         (KviCommand *, KviStr &buffer);
	bool parseIdn_MYIP               (KviCommand *, KviStr &buffer);
	bool parseIdn_MYMASK             (KviCommand *, KviStr &buffer);
	bool parseIdn_SERVERHOSTNAME     (KviCommand *, KviStr &buffer);
	bool parseIdn_SERVERIP           (KviCommand *, KviStr &buffer);
	bool parseIdn_SERVERPORT         (KviCommand *, KviStr &buffer);
	bool parseIdn_SERVERNETWORK      (KviCommand *, KviStr &buffer);
	bool parseIdn_VERSION            (KviCommand *, KviStr &buffer);
	bool parseIdn_CONSOLE            (KviCommand *, KviStr &buffer);
	bool parseIdn_MYUMODE            (KviCommand *, KviStr &buffer);
	bool parseIdn_SELECTED           (KviCommand *, KviStr &buffer);
	bool parseIdn_SELECTEDMASKS      (KviCommand *, KviStr &buffer);
	bool parseIdn_SELECTEDHOSTS      (KviCommand *, KviStr &buffer);
	bool parseIdn_SELECTEDUSERNAMES  (KviCommand *, KviStr &buffer);
	bool parseIdn_KVIRCDIR           (KviCommand *, KviStr &buffer);
	bool parseIdn_KVIRCLOGDIR        (KviCommand *, KviStr &buffer);
	bool parseIdn_KVIRCSCRIPTDIR     (KviCommand *, KviStr &buffer);
	bool parseIdn_CHAN               (KviCommand *, KviStr &buffer);
	bool parseIdn_QUERY              (KviCommand *, KviStr &buffer);
	bool parseIdn_CHAT               (KviCommand *, KviStr &buffer);
	bool parseIdn_ALLWINDOWS         (KviCommand *, KviStr &buffer);
	bool parseIdn_ALLQUERIES         (KviCommand *, KviStr &buffer);
	bool parseIdn_ALLCHANNELS        (KviCommand *, KviStr &buffer);
	bool parseIdn_ALLCHATS           (KviCommand *, KviStr &buffer);
	bool parseIdn_PROCCMDLINE        (KviCommand *, KviStr &buffer);
	bool parseIdn_PROCPID            (KviCommand *, KviStr &buffer);
	bool parseIdn_PROCSTDERR         (KviCommand *, KviStr &buffer);
	bool parseIdn_PROCSTDOUT         (KviCommand *, KviStr &buffer);
	bool parseIdn_PROCMAGIC          (KviCommand *, KviStr &buffer);
	bool parseIdn_PROCWINDOW         (KviCommand *, KviStr &buffer);
	bool parseIdn_PROCEXITCODE       (KviCommand *, KviStr &buffer);
	bool parseIdn_CONNTIME           (KviCommand *, KviStr &buffer);
	bool parseIdn_CONNTIMES          (KviCommand *, KviStr &buffer);
	bool parseIdn_ACTIVEWINDOW       (KviCommand *, KviStr &buffer);
	bool parseIdn_ISREMOTEEXEC       (KviCommand *, KviStr &buffer);
	bool parseIdn_EXECFILENAME       (KviCommand *, KviStr &buffer);
	bool parseIdn_GETRETURN          (KviCommand *, KviStr &buffer);
	bool parseIdn_SIGNALSENDER       (KviCommand *, KviStr &buffer);
	bool parseIdn_MDISIZE            (KviCommand *, KviStr &buffer);

	bool parseIdn_WHOISNICK          (KviCommand *, KviStr &buffer);
	bool parseIdn_WHOISUSER          (KviCommand *, KviStr &buffer);
	bool parseIdn_WHOISHOST          (KviCommand *, KviStr &buffer);
	bool parseIdn_WHOISREAL          (KviCommand *, KviStr &buffer);
	bool parseIdn_WHOISSERVER        (KviCommand *, KviStr &buffer);
	bool parseIdn_WHOISIRCOP         (KviCommand *, KviStr &buffer);
	bool parseIdn_WHOISIDLE          (KviCommand *, KviStr &buffer);
	bool parseIdn_WHOISSTATUS        (KviCommand *, KviStr &buffer);
	bool parseIdn_WHOISCHANNELS      (KviCommand *, KviStr &buffer);
	bool parseIdn_WHOISMAGIC         (KviCommand *, KviStr &buffer);
	bool parseIdn_WHOISSUCCESS       (KviCommand *, KviStr &buffer);
	bool parseIdn_TRYRESULT          (KviCommand *, KviStr &buffer);
	bool parseIdn_TIMES              (KviCommand *, KviStr &buffer);
	bool parseIdn_UPTIME             (KviCommand *, KviStr &buffer);
	bool parseIdn_CONNSTARTTIME      (KviCommand *, KviStr &buffer);
	bool parseIdn_THIS               (KviCommand *, KviStr &buffer);
	bool parseIdn_ROOT               (KviCommand *, KviStr &buffer);

	// kvi_functions.cpp
	bool parseFnc_HOSTNAME           (KviCommand *, KviStr &buffer);
	bool parseFnc_USERNAME           (KviCommand *, KviStr &buffer);
	bool parseFnc_ISWELLKNOWN        (KviCommand *, KviStr &buffer);
	bool parseFnc_STRLEN             (KviCommand *, KviStr &buffer);
	bool parseFnc_STRCAT             (KviCommand *, KviStr &buffer);
	bool parseFnc_CALC               (KviCommand *, KviStr &buffer);
	bool parseFnc_ISNUMBER           (KviCommand *, KviStr &buffer);
	bool parseFnc_GETENV             (KviCommand *, KviStr &buffer);
	bool parseFnc_ISON               (KviCommand *, KviStr &buffer);
	bool parseFnc_ISOWNER            (KviCommand *, KviStr &buffer);
	bool parseFnc_ISOP               (KviCommand *, KviStr &buffer);
	bool parseFnc_ISHALFOP           (KviCommand *, KviStr &buffer);
	bool parseFnc_ISVOICE            (KviCommand *, KviStr &buffer);
	bool parseFnc_ISUSEROP           (KviCommand *, KviStr &buffer);
	bool parseFnc_ITEMCOUNT          (KviCommand *, KviStr &buffer);
	bool parseFnc_SELECTED           (KviCommand *, KviStr &buffer);
	bool parseFnc_SELECTEDMASKS      (KviCommand *, KviStr &buffer);
	bool parseFnc_SELECTEDHOSTS      (KviCommand *, KviStr &buffer);
	bool parseFnc_SELECTEDUSERNAMES  (KviCommand *, KviStr &buffer);
	bool parseFnc_ICON               (KviCommand *, KviStr &buffer);
	bool parseFnc_DEFLOGFILE         (KviCommand *, KviStr &buffer);
	bool parseFnc_ISME               (KviCommand *, KviStr &buffer);
	bool parseFnc_ISREG              (KviCommand *, KviStr &buffer);
	bool parseFnc_RAND               (KviCommand *, KviStr &buffer);
	bool parseFnc_UNAME              (KviCommand *, KviStr &buffer);
	bool parseFnc_CLASSDEFINED       (KviCommand *, KviStr &buffer);
	bool parseFnc_CHANMODE           (KviCommand *, KviStr &buffer);
	bool parseFnc_CHANLIMIT          (KviCommand *, KviStr &buffer);
	bool parseFnc_CHANKEY            (KviCommand *, KviStr &buffer);

	bool parseFnc_NEW                (KviCommand *, KviStr &buffer);
	bool parseFnc_HASEVENTHANDLER    (KviCommand *, KviStr &buffer);

	bool parseFnc_HEXTOSTR           (KviCommand *, KviStr &buffer);
	bool parseFnc_STRTOHEX           (KviCommand *, KviStr &buffer);

	bool parseFnc_STRLEFT            (KviCommand *, KviStr &buffer);
	bool parseFnc_STRRIGHT           (KviCommand *, KviStr &buffer);
	bool parseFnc_STRCUTLEFT         (KviCommand *, KviStr &buffer);
	bool parseFnc_STRCUTRIGHT        (KviCommand *, KviStr &buffer);
	bool parseFnc_STRMID             (KviCommand *, KviStr &buffer);
	bool parseFnc_STRFIND            (KviCommand *, KviStr &buffer);
	bool parseFnc_STRFINDCS          (KviCommand *, KviStr &buffer);
	bool parseFnc_STRREVFIND         (KviCommand *, KviStr &buffer);
	bool parseFnc_STRREVFINDCS       (KviCommand *, KviStr &buffer);
	bool parseFnc_STRLEFTTOFIRST     (KviCommand *, KviStr &buffer);
	bool parseFnc_STRLEFTTOLAST      (KviCommand *, KviStr &buffer);
	bool parseFnc_STRRIGHTFROMFIRST  (KviCommand *, KviStr &buffer);
	bool parseFnc_STRRIGHTFROMLAST   (KviCommand *, KviStr &buffer);
	bool parseFnc_STRLEFTTOFIRSTCS   (KviCommand *, KviStr &buffer);
	bool parseFnc_STRLEFTTOLASTCS    (KviCommand *, KviStr &buffer);
	bool parseFnc_STRRIGHTFROMFIRSTCS(KviCommand *, KviStr &buffer);
	bool parseFnc_STRRIGHTFROMLASTCS (KviCommand *, KviStr &buffer);
	bool parseFnc_STRMATCH           (KviCommand *, KviStr &buffer);
	bool parseFnc_STRMATCHCS         (KviCommand *, KviStr &buffer);
	bool parseFnc_NUMTIMETOSTRING    (KviCommand *, KviStr &buffer);
	bool parseFnc_DURATIONTOSTRING   (KviCommand *, KviStr &buffer);

	bool parseFnc_OPTION             (KviCommand *, KviStr &buffer);
	bool parseFnc_ISWINDOW           (KviCommand *, KviStr &buffer);
	bool parseFnc_RANGE              (KviCommand *, KviStr &buffer);
	bool parseFnc_UPCASE             (KviCommand *, KviStr &buffer);
	bool parseFnc_LOWCASE            (KviCommand *, KviStr &buffer);
	bool parseFnc_STRREPLACE         (KviCommand *, KviStr &buffer);
	bool parseFnc_STRREPLACECS       (KviCommand *, KviStr &buffer);
	bool parseFnc_ISDOCKED           (KviCommand *, KviStr &buffer);
	bool parseFnc_MASK               (KviCommand *, KviStr &buffer);
	bool parseFnc_ISKNOWN            (KviCommand *, KviStr &buffer);
	bool parseFnc_CHAR               (KviCommand *, KviStr &buffer);
	bool parseFnc_ASCII              (KviCommand *, KviStr &buffer);
	bool parseFnc_ISVALIDIP          (KviCommand *, KviStr &buffer);
	bool parseFnc_INETATON           (KviCommand *, KviStr &buffer);
	bool parseFnc_INETNTOA           (KviCommand *, KviStr &buffer);
	bool parseFnc_COMMANDLINE        (KviCommand *, KviStr &buffer);
	bool parseFnc_ALLONCHAN          (KviCommand *, KviStr &buffer);
	bool parseFnc_OPONCHAN           (KviCommand *, KviStr &buffer);
	bool parseFnc_VOICEONCHAN        (KviCommand *, KviStr &buffer);
	bool parseFnc_NOPONCHAN          (KviCommand *, KviStr &buffer);
	bool parseFnc_TOPIC              (KviCommand *, KviStr &buffer);
	bool parseFnc_READFILE           (KviCommand *, KviStr &buffer);
	bool parseFnc_FILEEXISTS         (KviCommand *, KviStr &buffer);
	bool parseFnc_DIREXISTS          (KviCommand *, KviStr &buffer);
	bool parseFnc_FILESIZE           (KviCommand *, KviStr &buffer);
	bool parseFnc_DICTLIST           (KviCommand *, KviStr &buffer);
	bool parseFnc_DICTCOUNT          (KviCommand *, KviStr &buffer);
	bool parseFnc_GETFLAGS           (KviCommand *, KviStr &buffer);
	bool parseFnc_GETPASS            (KviCommand *, KviStr &buffer);
	bool parseFnc_GETCOMMENT         (KviCommand *, KviStr &buffer);
	bool parseFnc_GETFLAGSEXACT      (KviCommand *, KviStr &buffer);
	bool parseFnc_CONFIG             (KviCommand *, KviStr &buffer);
	bool parseFnc_PLUGINLOADED       (KviCommand *, KviStr &buffer);
	bool parseFnc_ISTIMER            (KviCommand *, KviStr &buffer);
	bool parseFnc_BANLIST            (KviCommand *, KviStr &buffer);
	bool parseFnc_BANEXCEPTIONLIST   (KviCommand *, KviStr &buffer);
	bool parseFnc_INVITEEXCEPTIONLIST(KviCommand *, KviStr &buffer);

	bool evaluateExpr(KviCommand *, long *result);

	KviExprTree *evaluateExpression   (KviCommand *);
	KviExprTree *evaluateOperand      (KviCommand *);
	KviExprTree *evaluateSimpleOperand(KviCommand *);
	KviExprTree *evaluateStringOperand(KviCommand *);
	KviExprTree *evaluateOperator     (KviCommand *);

	KviExprTree *evaluateExpression_RightOperand(KviExprTree *curTopOperator, KviCommand *);

	void    addAsyncDialog(KviAsyncDialog *);
	void removeAsyncDialog(KviAsyncDialog *);

	void dumpChildObjects(KviWindow *pWnd, QObject *parent, const char *spacing, bool bWidgetsOnly);

	KviWindowTimer *findTimerById(int id);
	KviWindowTimer *findTimerByName(const char *name);
	bool removeTimer(const char *name);
	void addTimer(KviWindowTimer *);
	virtual void timerEvent(QTimerEvent *);
};

inline void KviUserParser::setLastSignalSender(const char *lss)
{
	m_szLastSignalSender = lss;
}

#endif // _KVI_USERPARSER_H_INCLUDED_
