/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <qlabel.h>
#include <qptrlist.h>
#include <kmessagebox.h>
#include "flosslistboxitem.h"
#include "addcolordialog.h"
#include "deletecolordialog.h"
#include "flosspalette.h"
#include "flossscheme.h"
#include "schememanager.h"
#include "kxstitch.h"

FlossPalette::FlossPalette()
  : m_current(-1)
{
}

bool FlossPalette::addColors()
{
  bool added;
  FlossListBoxItem *pFlossItem;
  AddColorDialog *pDlg;
  FlossScheme *pScheme = ((KXStitchApplication*)kapp)->scheme(m_schemeName);

  added=false;
  if (pScheme)
  {                                               // make sure that pScheme actually points to a FlossScheme
    pDlg = new AddColorDialog(0,m_schemeName);    // create an instance of the AddColorDialog
    if (pDlg)
    {                                             // and check that it was created successfully
      QPtrListIterator<Floss> it = pScheme->flossIterator();
      Floss *f;
      for ( ; (f = it.current()) ; ++it)
      {
        if (!contains(f->name))                   // ignore any already in the palette
          pFlossItem = new FlossListBoxItem(pDlg->ColorList, f);
      }
      if (pDlg->exec() == QDialog::Accepted)
      {                                           // process the dialog if the user selected OK
        int count = pDlg->ColorList->count();
        for (int i = 0 ; i < count ; i++)
        {
          if (pDlg->ColorList->isSelected(i))
          {                                       // isSelected is true if the QListBoxItem was selected in the QListBox
                                                  // get the floss details, create a new Floss from it and add it to the palette
            pFlossItem = (FlossListBoxItem *)(pDlg->ColorList->item(i));
            int mi = freeIndex();
            m_palette[mi] = pFlossItem->floss;
            m_symbols[mi] = freeSymbol();
            added=true;                           // indicate that colors were added, this is used by the app to update the paletteview
          }
        }
      }
      delete pDlg;                                // clean up
    }
  }
  return added;
}

int FlossPalette::addColor(QColor color)
{
  int i;
  if ((i = contains(color)) == -1)
  { // contains(QColor) will return either the colors index or -1 if not in the palette
    QPtrListIterator<Floss> it = ((KXStitchApplication*)kapp)->scheme(m_schemeName)->flossIterator();
    Floss *f;
    for ( ; (f = it.current()) ; ++it)
    {
      if (color == f->color)
      {
        int mi = freeIndex();
        m_palette[mi] = f;
        m_symbols[mi] = freeSymbol();
        return mi;
      }
    }
  }
  return i;
}

bool FlossPalette::removeColors(QValueList<int> usedColors)
{
  bool removed = false;
  FlossListBoxItem *pFlossItem;
  DeleteColorDialog *pDlg;
  QMapIterator<int,Floss*> it;

  pDlg = new DeleteColorDialog(0);             // create an instance of the AddColorDlg
  if (pDlg)
  {                                             // and check that it was created successfully
    for (it = m_palette.begin() ; it != m_palette.end() ; ++it)
    {
      pFlossItem = new FlossListBoxItem(pDlg->ColorList, it.data());
      if (usedColors.contains(it.key()))       // if the color has been used in the pattern
        pFlossItem->setSelectable(false);
    }
    if (pDlg->exec() == QDialog::Accepted)
    {                                         // process the dialog if the user selected OK
      int count = pDlg->ColorList->count();
      for (int i = 0 ; i < count ; i++)
      {
        if (pDlg->ColorList->isSelected(i))
        {                                     // isSelected is true if the QListBoxItem was selected in the QListBox
                                                // get the floss details and remove it from the palette
          pFlossItem = (FlossListBoxItem *)(pDlg->ColorList->item(i));
          int mi = contains(pFlossItem->floss->color);
          if (mi == m_current) m_current = -1;
          m_palette.remove(mi);
          m_symbols.remove(mi);
          removed=true;                         // indicate that colors were removed, this is used by the app to update the paletteview
        }
      }
    }
    delete pDlg;                                // clean up
  }
  return removed;
}

bool FlossPalette::removeUnused(QValueList<int> usedColors)
{
  bool removed = false;
  QMapIterator<int,Floss*> it;

  for (it = m_palette.begin() ; it != m_palette.end() ; ++it)
  {
    int mi = it.key();
    if (!usedColors.contains(mi))
    {                                           // if the color has not been used in the pattern
      if (mi == m_current) m_current = -1;
      m_palette.remove(mi);
      m_symbols.remove(mi);
      removed=true;                         // indicate that colors were removed, this is used by the app to update the paletteview
    }
  }
  return removed;
}

void FlossPalette::swapColors(int a)
{
  // swap a with the current color
  Floss *f = m_palette[m_current];
  m_palette[m_current] = m_palette[a];
  m_palette[a] = f;
  QChar c = m_symbols[m_current];
  m_symbols[m_current] = m_symbols[a];
  m_symbols[a] = c;
}

bool FlossPalette::contains(QString name)
{
  QMapIterator<int,Floss*> it;
  for (it = m_palette.begin() ; it != m_palette.end() ; ++it )
  {
    if (it.data()->name == name)
      return true;
  }
  return false;
}

int FlossPalette::contains(QColor color)
{
  QMapIterator<int,Floss*> it;
  for (it = m_palette.begin() ; it != m_palette.end() ; ++it )
  {
    if (it.data()->color == color)
      return it.key();
  }
  return -1;
}

QChar FlossPalette::freeSymbol()
{
  int c = -1;
  bool found = false;
  QMapIterator<int,QChar> it;

  while (!found)
  {
    QChar rc(++c);
    if (rc.isPrint() && !rc.isSpace() && !rc.isPunct())
    {
      found = true;
      for (it = m_symbols.begin() ; it != m_symbols.end() && found ; ++it)
        if (it.data() == rc)
          found = false;
    }
  }
  return QChar(c);
}

int FlossPalette::freeIndex()
{
  int i = 0;
  while (m_palette.contains(i)) i++;
  return i;
}

void FlossPalette::setCurrentColor(int i)
{
  m_current = i;
}

int FlossPalette::currentColor()
{
  return m_current;
}

int FlossPalette::colors()
{
  return m_palette.count();
}

void FlossPalette::setScheme(QString s)
{
  m_schemeName = s;
  if (((KXStitchApplication*)kapp)->scheme(s) == 0)
  {
    // there is no scheme held by the scheme manager by this name
    // TODO add error message
  }
  m_palette.clear();
  m_symbols.clear();
  m_current = -1;
}

QString FlossPalette::getScheme()
{
  return m_schemeName;
}

void FlossPalette::clear()
{
  m_palette.clear();
  m_current = -1;
}

Floss *FlossPalette::flossAt(int i)
{
  return m_palette[i];
}

QChar FlossPalette::symbolAt(int i)
{
  return m_symbols[i];
}

bool FlossPalette::loadPalette(QString schemeName)
{
  FlossScheme* scheme = ((KXStitchApplication*)kapp)->scheme(schemeName);
  if (scheme)
  {
    setScheme(schemeName);
    QPtrListIterator<Floss> it = scheme->flossIterator();
    for (Floss* floss ; (floss = it.current()) ; ++it)
    {
      int i = freeIndex();
      m_palette[i] = floss;
      m_symbols[i] = freeSymbol();
    }
    return true;
  }
  return false;
}

bool FlossPalette::savePalette(QString schemeName)
{
  FlossScheme* scheme = ((KXStitchApplication*)kapp)->scheme(schemeName);
  if (scheme == 0)
    scheme = ((KXStitchApplication*)kapp)->createScheme(schemeName);
  else
    scheme->clearScheme();
  if (scheme)
  {
    // iterate flosses and write to scheme
    QMapIterator<int,Floss*> flosses;
    for (flosses = m_palette.begin() ; flosses != m_palette.end() ; ++flosses)
    {
      Floss* floss = new Floss(flosses.data()->name,flosses.data()->description,flosses.data()->color);
      scheme->addFloss(floss);
      m_palette[flosses.key()] = floss;
    }
    ((KXStitchApplication*)kapp)->writeScheme(schemeName);
    m_schemeName = schemeName;
    return true;
  }
  return false;
}

bool FlossPalette::writePalette(QDataStream &s)
{
  // output the contents of the floss palette to the data stream using portable units
  s << m_schemeName;                                                                                    // QString
  s << (Q_INT32)m_current << (Q_INT32)m_palette.count();                                                  // Q_INT32, Q_INT32
  QMapIterator<int,Floss*> flosses;
  for (flosses = m_palette.begin() ; flosses != m_palette.end() ; ++flosses)
  {
    s << (Q_INT32)flosses.key() << flosses.data()->name << (Q_INT16)m_symbols[flosses.key()].unicode(); // Q_INT32, QString, Q_INT16
  }
  return true;
}

bool FlossPalette::readPCStitch5Palette(QDataStream& s)
{
  char*         buffer = new char[51];
  Q_INT16       colors;
  Q_INT32       unknown;
  Q_INT16       symbolNameLength;
  Q_INT8        symbol;
  FlossScheme*  scheme;
  QColor        color;
  QString       colorName;
  QString       description;
  s.readRawBytes(buffer,25);
  buffer[25] = '\0';
  if (QString(buffer) == "PCStitch 5 Floss Palette!")
  {
    s >> colors;
    s.readRawBytes(buffer,30); // this should be 'DMC       Anchor    Coates    '
    s >> symbolNameLength; // font text length
    if (symbolNameLength >= 50)
    {
      delete buffer;
      buffer = new char[symbolNameLength];
    }
    s.readRawBytes(buffer,symbolNameLength); // the font name, usually 'PCStitch Symbols'
    // assume this palette will be DMC
    setScheme("DMC");
    scheme = ((KXStitchApplication *)kapp)->scheme("DMC");
    if (scheme == 0)
      return false;
    s >> unknown; // unknown
    for (int i = 0 ; i < colors ; i++)
    {
      s >> color; // unknown, probably RGBA
      s.readRawBytes(buffer,30); // includes DMC,Anchor and Coates names
      buffer[10] = '\0'; // only interested in DMC name
      colorName = QString(buffer).stripWhiteSpace(); // minus the white space at the end
      if (colorName == "White")
        colorName = "Blanc"; // difference in names for white
      s.readRawBytes(buffer,50); // floss discription, discard as comes from scheme manager
      buffer[50] = '\0';
      description = QString(buffer).stripWhiteSpace();
      s >> symbol; // symbol - this will look different due to the inconsistant fonts used
      m_palette[i] = scheme->find(colorName);
      if (m_palette[i] == 0)
      {
        if (KMessageBox::questionYesNo(0,QString("Adding new floss (%1:%2:RGB(%3,%4,%5))").arg(colorName).arg(description).arg(color.red()).arg(color.green()).arg(color.blue())) == KMessageBox::Yes)
        {
          scheme->addFloss(new Floss(colorName,description,color));
          m_palette[i] = scheme->find(colorName);
          if (m_palette[i] == 0)
            return false;
          ((KXStitchApplication *)kapp)->writeScheme("DMC");
        }
      }
      m_symbols[i] = QChar(symbol);
      s >> unknown; // unknown, 0x00010002 is usual value
                    // this may refer to the number of strands for backstitching(0x0001) and stitches(0x0002)
    }
    m_current = -1;
  }
  return true;
}

bool FlossPalette::readKXStitchPalette(QDataStream& s,int fileFormatVersion)
{
  Q_INT32       current;
  Q_INT32       colors;
  QString       schemeName;
  QString       colorName;
  QString       description;
  QColor        color;
  Q_INT16       symbol;
  FlossScheme*  scheme;
  switch (fileFormatVersion)
  {
    case 5: // no changes from version 4 for this section
    case 4: // no changes from version 3 for this section
    case 3:
      s >> schemeName;
      setScheme(schemeName);
      scheme = ((KXStitchApplication *)kapp)->scheme(schemeName);
      s >> current;
      m_current = current;
      s >> colors;
      while (colors--)
      {
        s >> current;
        s >> colorName;
        s >> symbol;
        m_palette[current] = scheme->find(colorName);
        if (m_palette[current] == 0)
          return false;
        m_symbols[current] = QChar((ushort)symbol);
      }
      break;
    case 2:
    case 1:
      /** create the floss palette from the data stream */
      s >> current;
      s >> colors;
      s >> schemeName;
      if (schemeName == "dmc")
        schemeName = "DMC"; // bug workaround
      setScheme(schemeName);
      scheme = ((KXStitchApplication *)kapp)->scheme(schemeName);
      if (scheme == 0)
        return false;
      m_current = current;
      for (current = 0 ; current < colors ; current++)
      {
        s >> colorName;
        // bug workaround for lowercase names
        if (colorName == "blanc") colorName = "Blanc";
        if (colorName == "ecru") colorName = "Ecru";
        if (colorName == "b5200") colorName = "B5200";
        if (colorName == "black") colorName = "Black";
        if (colorName == "white") colorName = "White";
        s >> description;
        s >> color;
        s >> symbol;
        m_palette[current] = scheme->find(colorName);
        if (m_palette[current] == 0)
          return false;
        m_symbols[current] = QChar(symbol);
      }
      break;
    default:
      return false;
  }
  return true;
}
