
/*
 * Logging routines.
 * Contains routines that are used with the glib logging facility.
 */

#include "ledd.h"

#include <syslog.h>
#include <stdio.h>


static void log_via_syslog(const gchar *domain, GLogLevelFlags level,
			   const gchar *message,gpointer data);
static void log_via_stderr(const gchar *domain, GLogLevelFlags level,
			   const gchar *message,gpointer data);
static void log_via_none(const gchar *domain, GLogLevelFlags level,
			 const gchar *message,gpointer data);


/*
 * Sets the correct logging facility. If file is NULL, sets logging to
 * STDERR.
 */
void log_set_facility(File *file) {
	File std;

	if (file==NULL) {
		std.name="";
		std.type=TYPE_LOG_STDERR;
		std.fd=-1;
		std.pid=0;
		std.timer=NULL;
		file=&std;
	}

	if (file->type==TYPE_LOG_SYSLOG) {
		g_log_set_handler(G_LOG_DOMAIN,
				  G_LOG_LEVEL_MESSAGE|
				  G_LOG_LEVEL_WARNING|
				  G_LOG_LEVEL_ERROR|
				  G_LOG_FLAG_FATAL,
				  log_via_syslog,NULL);
#ifdef HAVE_SYSLOG
		openlog("ledd",0,LOG_DAEMON);
#endif
	} else if (file->type==TYPE_LOG_STDERR) {
		g_log_set_handler(G_LOG_DOMAIN,
				  G_LOG_LEVEL_MESSAGE|
				  G_LOG_LEVEL_WARNING|
				  G_LOG_LEVEL_ERROR|
				  G_LOG_FLAG_FATAL,
				  log_via_stderr,NULL);
	} else if (file->type==TYPE_LOG_NONE) {
		g_log_set_handler(G_LOG_DOMAIN,
				  G_LOG_LEVEL_MESSAGE|
				  G_LOG_LEVEL_WARNING|
				  G_LOG_LEVEL_ERROR|
				  G_LOG_FLAG_FATAL,
				  log_via_none,NULL);
	} else {
		G_ERROR("unknown logging type %d",file->type);
	}
	return;
}		

/*
 * Closes neccessary stuff from logging. Sets the handler to STDERR.
 */
void log_close(File *file) {

#ifdef HAVE_SYSLOG
	if (file->type==TYPE_LOG_SYSLOG) {
		closelog();
	}
#endif
	log_set_facility(NULL);
	return;
}



/*
 * Actual logging functions. Function name describes it.
 */
static void log_via_syslog(const gchar *domain, GLogLevelFlags level,
			   const gchar *message,gpointer data) {
#ifdef HAVE_SYSLOG
	int lvl;

	if (level & G_LOG_LEVEL_ERROR)
		lvl=LOG_ERR;
	else if (level & G_LOG_LEVEL_WARNING)
		lvl=LOG_WARNING;
	else if (level & G_LOG_LEVEL_MESSAGE)
		lvl=LOG_NOTICE;
	/* Unknown, guess. */
	else if (level & G_LOG_FLAG_FATAL)
		lvl=LOG_ERR;
	else
		lvl=LOG_WARNING;

	syslog(lvl,"%s",message);
#endif
	return;
}

static void log_via_stderr(const gchar *domain, GLogLevelFlags level,
			   const gchar *message, gpointer data) {

	fprintf(stderr,"ledd: %s\n",message);
	return;
}

static void log_via_none(const gchar *domain, GLogLevelFlags level,
			 const gchar *message, gpointer data) {
	return;
}


