import re, types, operator
from linda import libchecks, checks

class ControlCheck(libchecks.LindaChecker):
    'Perform general checks of the control file.'
    def check_binary_1(self):
        self.package()
        self.priority()
        self.section()
        self.maintainer()
        self.essential()
        self.description()
        self.depends()
        self.provides()
        self.obselete()
        self.recommends_suggests()
        
    def package(self):
        if self.information['control']['self'][0].has_key('package'):
            upper = reduce(operator.add, [x.isupper() for x in \
                self.information['control']['self'][0]['package']])
            if upper:
                self.signal_error('upper-in-package')
            if self.information['control']['self'][0]['package'].endswith('-perl'):
                if self.information['control']['self'][0]['section'][-1] == \
                    'interpreters':
                    self.signal_error('perl-pkg-should-be-sct-perl')
            if self.information['control']['self'][0]['package'].startswith('python-'):
                if self.information['control']['self'][0]['section'][-1] == \
                    'interpreters':
                    self.signal_error('python-pkg-should-be-sct-py')
    def priority(self):
        if self.information['control']['self'][0].has_key('priority'): 
            if not DataValidator('priority', self.information['control']['self'][0]['priority']):
                self.signal_error('unknown-priority', \
                    [self.information['control']['self'][0]['priority']])
    def section(self):
        if self.information['control']['self'][0].has_key('section'):
            if len(self.information['control']['self'][0]['section']) == 2:
                if self.information['control']['self'][0]['section'][0] == \
                    'non-us':
                    self.signal_error('spelling-non-us')
                elif self.information['control']['self'][0]['section'][0] \
                    not in ('contrib', 'non-free', 'non-US'):
                    self.signal_error('unknown-component', \
                        [self.information['control']['self'][0]['section'][0]])
            if not DataValidator('section', \
                self.information['control']['self'][0]['section'][-1]):
                self.signal_error('unknown-section', \
                    [self.information['control']['self'][0]['section'][-1]])
    def maintainer(self):
        if self.information['control']['self'][0].has_key('maintainer'):
            if not self.information['control']['self'][0]['maintainer']:
                self.signal_error('empty-maintainer')
            else:
                if self.information['control']['self'][0]['maintainer'].find('<') == -1:
                    self.signal_error('no-maintainer-email')
                else:
                    maint_mail = self.information['control']['self'][0]['maintainer']\
                        [self.information['control']['self'][0]['maintainer'].find('<'):\
                        self.information['control']['self'][0]['maintainer'].find('>')]
                    if not re.search(r"[a-zA-Z0-9_']+@.*\.", maint_mail):
                        self.signal_error('malformed-maint-mail-addr')
        else:
            self.signal_error('no-maintainer')
    def essential(self):
        if self.information['control']['self'][0].has_key('essential'):
            if self.information['control']['self'][0]['essential'].lower() == 'no':
                self.signal_error('essential-no-redundant')
            if self.information['control']['self'][0]['essential'].lower() not in ('no', 'yes'):
                self.signal_error('essential-not-yes-or-no', \
                    [self.information['control']['self'][0]['essential']])
            if not DataValidator('essential', self.pkg_name):
                self.signal_error('new-essential-package')
    def description(self):
        if self.information['control']['self'][0].has_key('description'):
            if self.information['control']['self'][0]['description'][0].startswith(self.pkg_name):
                self.signal_error('desc-starts-with-package')
            if self.information['control']['self'][0]['description'][0].find('  ') != -1:
                self.signal_error('multiple-spaces-in-desc')
            if len(self.information['control']['self'][0]['description'][0]) > 80:
                self.signal_error('description-too-long')
            for y in self.information['control']['self'][0]['description']:
                if y.find('\t') != -1:
                    self.signal_error('tab-in-desc')
            if self.information['control']['self'][0]['description'][1].find\
                    (self.information['control']['self'][0]['description'][0]) != -1:
                    self.signal_error('ex-desc-contains-desc')
        else:
            self.signal_error('no-description')
    def depends(self):
        if self.information['control']['self'][0].has_key('depends'):
            for x in self.information['control']['self'][0]['depends'].keys():
                if DataValidator('essential', x) and not \
                    self.information['control']['self'][0]['depends'][x]['relation']:
                    self.signal_error('no-version-dep-on-essential', [x])
                if x == self.pkg_name:
                    self.signal_error('pkg-depends-on-self')
    def provides(self):
        if self.information['control']['self'][0].has_key('provides'):
            for x in self.information['control']['self'][0]['provides'].keys():
                for version in self.information['control']['self'][0]['provides'][x]['relation']:
                    if version != [None, None]:
                        self.signal_error('versioned-provides', [x])
        if not self.information['control']['self'][0].has_key('package'):
            self.signal_error('no-package-line')
        if not self.information['control']['self'][0].has_key('priority'):
            self.signal_error('no-priority-line')
        if not self.information['control']['self'][0].has_key('version'):
            self.signal_error('no-version-line')
        else:
            if not self.is_native():
                deb_rev = self.information['control']['self'][0]['version'].split('-')[-1]
                if deb_rev == '0':
                    self.signal_error('deb-revision-zero')
    def obselete(self):
        obsel_count = 0
        for x in ('revision', 'package-revision', 'package_revision', \
            'recommended', 'optional', 'class'):
            if self.information['control']['self'][0].has_key(x):
                obsel_count += 1
        if obsel_count:
            self.signal_error('obsolete-field-in-control')
    def recommends_suggests(self):
        check_fields = []
        for field in ('recommends', 'suggests'):
            if self.information['control']['self'][0].has_key(field):
                check_fields.append(field)
        for field in check_fields:
            for pkg in self.information['control']['self'][0][field]:
                if self.information['control']['self'][0].has_key('depends'):
                    if pkg in self.information['control']['self'][0]['depends']:
                        self.signal_error('recommends-suggests-dependant-pkg',\
                            [field.title(), pkg])
                    
    def check_source_2(self):
        self.build_depends()
        self.source()

    def build_depends(self):
        if self.information['control']['self'][0].has_key('build-depends'):
            for build_dep in self.information['control']['self'][0]['build-depends'].keys():
                for arch in self.information['control']['self'][0]['build-depends'][build_dep]['arch']:
                    if not DataValidator('arch', arch):
                        self.signal_error('invalid-arch-in-b-d', [arch])
            found_auto = 0
            for x in self.information['control']['self'][0]['build-depends'].keys():
                if x == 'autoconf' or x.startswith('automake'):
                    found_auto = 1
            if found_auto:
                self.signal_error('package-b-d-on-autostar')
    def source(self):
        if self.information['control']['self'][0].has_key('essential'):
            self.signal_error('essential-field-in-source')
        if not self.information['control']['self'][0].has_key('source'):
            self.signal_error('no-source-field-in-source')
        else:
            for x in self.information['control']['self'][0]['source']:
                if (ord(x) < 48 or ord(x) > 122) and ord(x) not in (43, 45, \
                    46):
                    self.signal_error('bad-source-field', [x])
            if self.pkg_name != \
                self.information['control']['self'][0]['source']:
                self.signal_error('source-not-like-package')
        packages = {'arch-indep': 0}
        for stanza in self.information['control']['self']:
            for field in stanza:
                if field in ('source', 'priority', 'standards-version', \
                    'package'):
                    self.check_forbidden_chars(stanza[field])
                elif field in ('description', 'section'):
                    for desc in stanza[field]:
                        self.check_forbidden_chars(desc)
                elif field == 'architecture':
                    if stanza['architecture'] == 'all':
                        packages['arch-indep'] += 1
        if self.information['control']['self'][0].has_key('build-depends-indep') and not packages['arch-indep']:
            self.signal_error('build-dep-indep-with-no-all')

    def check_forbidden_chars(self, data):
        for char in data:
            if char == '\n':
                continue
            if ord(char) < 32 or ord(char) > 126:
                self.signal_error('dangerous-char-in-control', [char])
    
checks.register(ControlCheck)

