import unittest, gettext, os, linda, shutil, tempfile
from po_count import POFileParser
from linda.collector import Collector
from linda.debug import dprint
from linda.parser.datavalidator import DataValidator
from linda.libchecks import Checks
from sets import Set

linda.checks = Checks()

class LibraryCheckBinaryTest(unittest.TestCase):
    def setUp(self):
        execfile('checks/library.py', {'_': gettext.gettext, 'dprint': \
            dprint, 'DataValidator': DataValidator, 'os': os})
        for klass in linda.checks.registry['binary'][2]:
            self.instance = klass('/tmp', '/tmp/test_0.0.1-1_all.deb', {})
        self.instance.information['dir'] = tempfile.mkdtemp()
        os.mkdir('%s/unpacked' % self.instance.information['dir'])
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        self.po_keys = POFileParser('po/en.po')
        
    def tearDown(self):
        shutil.rmtree(self.instance.information['dir'])

    def test_discover_libraries(self):
        "Check discover_libraries() method of LibraryCheck"
        method = getattr(self.instance, 'discover_libraries')
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/bin/lala': '', '/lib/foo.so': '', '/usr/lib/perl5/intl.a': \
            '', '/usr/lib/lala.la': ''})
        self.instance.information['collector'].set('files', 'elf', \
            ['/usr/bin/lala', '/lib/foo.so', '/usr/lib/perl5/intl.a'])
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/lala': 'ELF 32-bit LSB executable, Intel 80386, version 1 (SYSV), for GNU/Linux 2.2.0, dynamically linked (uses shared libs), stripped', '/lib/foo.so': 'ELF 32-bit LSB shared object, Intel 80386, version 1 (SYSV), stripped', '/usr/lib/perl5/intl.a': 'ELF 32-bit LSB shared object, Intel 80386, version 1 (SYSV), stripped'})
        method()
        self.assertEqual(self.instance.numbers['binaries'], 1)
        self.assertEqual(self.instance.numbers['libraries'], 1)
        self.assert_('/lib/foo.so' in self.instance.libraries.keys())
        self.assertEqual(self.instance.la_files, ['/usr/lib/lala.la'])
        os.mkdir('%s/unpacked/lib' % self.instance.information['dir'])
        os.symlink('/lib/libc-2.3.2.so', '%s/unpacked/lib/foo.so' % \
            self.instance.information['dir'])
        method()
        self.assert_('so-symlink-non-dev' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'so-symlink-non-dev'), True)
        os.unlink('%s/unpacked/lib/foo.so' % self.instance.information['dir'])
        
    def test_discover_soname(self):
        "Check discover_sonames() method of LibraryCheck"
        method = getattr(self.instance, 'discover_sonames')
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/bin/lala': '', '/lib/foo.so': ''})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/lala': 'ELF 32-bit LSB executable, Intel 80386, version 1 (SYSV), for GNU/Linux 2.2.0, dynamically linked (uses shared libs), stripped', '/lib/foo.so': 'ELF 32-bit LSB shared object, Intel 80386, version 1 (SYSV), stripped'})
        self.instance.information['collector'].set('output', 'objdump', \
            {'/lib/foo.so': '  SONAME      libfoo.so.3'})
        method('/lib/foo.so')
        self.assertEqual(self.instance.libraries['/lib/foo.so'], 'libfoo')
    
    def test_check_libs(self):
        "Check check_against_libs() method of LibraryCheck"
        method = getattr(self.instance, 'check_against_libs')
        self.instance.libraries['/usr/lib/apache/php3.so'] = ''
        self.instance.pkg_name = 'php3'
        self.instance.information['control'] = {'self': [{'section': ['foo']}]}
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.libraries['/usr/lib/apache/php3.so'] = 'php3.5'
        self.instance.information['control']['self'][0]['section'] = ['non-free', 'libs']
        method()
        self.assert_('section-not-libs' not in self.instance.err_dict.keys())
        self.instance.err_dict.clear()
        self.instance.information['control']['self'][0]['section'] = ['non-free', 'foo']
        method()
        expected_errs = Set(('section-not-libs', 'package-not-start-lib'))
        self.assertEqual(expected_errs.intersection(self.instance.err_dict.keys()), expected_errs)
        for key in expected_errs:
            self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
                key), True)
        self.instance.err_dict.clear()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.libraries['/lib/foo.so'] = 'foo'
        self.instance.pkg_name = 'foo3'
        self.instance.information['control'] = {'self': [{'section': ['foo']}]}
        method()
        expected_errs = Set(('section-not-libs', 'package-not-start-lib'))
        self.assertEqual(expected_errs.intersection(self.instance.err_dict.keys()), expected_errs)
        self.instance.err_dict.clear()
        self.instance.numbers['binaries'] = 5
        self.instance.numbers['libraries'] = 1
        self.instance.libraries['/lib/libpython.so'] = 'py2.3'
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.numbers['binaries'] = 0
        self.instance.numbers['libraries'] = 1
        self.instance.info['soname_in_pkg'] = 0
        self.instance.pkg_name = 'bar4'
        self.instance.libraries = {'/lib/foo.so': 'foo'}
        method()
        self.assert_('package-notlike-soname' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'package-notlike-soname'), True)

    def test_check_la_files(self):
        "Check check_la_files() method of LibraryCheck"
        method = getattr(self.instance, 'check_la_files')
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'lib'))
        f = open(os.path.join(self.instance.information['dir'], 'unpacked', \
            'usr/lib/libfoo.la'), 'w')
        f.write("libdir='/usr/lib'\n")
        f.close()
        self.instance.la_files = ['/usr/lib/libfoo.la']
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        f = open(os.path.join(self.instance.information['dir'], 'unpacked', \
            'usr/lib/libfoo.la'), 'w')
        f.write("libdir='/home/buildd/build-2/usr/lib'\n")
        f.close()
        method()
        self.assert_('libdir-not-equal-dir' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'libdir-not-equal-dir'), True)

