/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

/*
 * Display the L2 book using a generic streamer datafeed (e.g. schwab)
 *
 * N.B. Since only one streamer has been implemented so far, this model
 * might need to be changed to accomodate other streamers.
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <time.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include "curse.h"
#include "error.h"
#include "rc.h"
#include "streamer.h"
#include "linuxtrade.h"
#include "l2sr.h"
#include "p2open.h"
#include "util.h"

static WINDOW	*Win;
static WINDOW	*Hdrwin;
static WINDOW	*Subwin;
static PANEL	*Panel;
static char	Symbol[SYMLEN+1];
static STREAMER	Sr;

/*
 * Book
 */
typedef struct
{
	char	recid[RECIDLEN+1];	// Record ID, key for adding/deleting
	char	mmid[MMIDLEN+1];	// Market Maker name
	double	price;
	int	size;
	int	time;			// Time in seconds since midnight
	int	key2;			// Secondary sorting key
	char	code;			// Display selection code
	char	tick;			// =, +, -
} BOOKE;

#define		NUMBOOK	2000
#define		NUMSET	5

// Used for displaying bar
typedef struct
{
	int	num;
	int	size;
} SET;

// Entire book
typedef struct
{
	int	num;		// num orders
	int	total;		// total size
	SET	set[NUMSET];
	BOOKE	ent[NUMBOOK];
} BOOK;

static BOOK	Ask, Bid;
static	char	Code = 0;
static int	DisplayOn = TRUE;

/*
 * Calculate and display the bid/ask relative volume bar
 */
static attr_t SetColors[NUMSET] =
{
	COLOR_PAIR(BLACKonGREEN),
	COLOR_PAIR(BLACKonYELLOW),
	COLOR_PAIR(BLACKonCYAN),
	A_REVERSE|COLOR_PAIR(REDonWHITE),
	A_REVERSE|COLOR_PAIR(BLUEonWHITE)
};

static void
calc_bar(BOOK *b)
{
	int	i;
	int	n = -1;
	double	lastprice = -1;

	b->total = 0;
	for (i = 0; i < b->num; ++i)
	{
		BOOKE	*e = &b->ent[i];

		b->total += e->size * 100;
		if (n < (NUMSET-1) && e->price != lastprice)
		{
			++n;
			lastprice = e->price;
			b->set[n].size = e->size*100;
			b->set[n].num = 1;
		}
		else
		{
			b->set[n].size += e->size*100;
			++b->set[n].num;
		}
	}
	while (++n < NUMSET)
	{
		b->set[n].size = 0;
		b->set[n].num = 0;
	}
}

static void
disp_bar(void)
{
	int	i, x;
	int	total;
	int	barcols = (getmaxx(Hdrwin) - 1) & ~1;
	int	bidcols, askcols;

	calc_bar(&Ask);
	calc_bar(&Bid);

	total = Ask.total + Bid.total;

	wblankrect(Hdrwin, 1, 0, 1, getmaxx(Hdrwin)-1, FALSE);
	if (total == 0)
	{
		return;
	}
	else if (Ask.total == 0)
	{
		askcols = 0;
		bidcols = barcols;
	}
	else if (Bid.total == 0)
	{
		bidcols = 0;
		askcols = barcols;
	}
	else
	{
		bidcols = lrint((double)barcols * Bid.total / total);
		askcols = lrint((double)barcols * Ask.total / total);
	}

	wattrset(Hdrwin, A_REVERSE);
	mvwaddch(Hdrwin, 1, bidcols, ACS_VLINE);

	if (bidcols)
	{
		x = bidcols;
		for (i = 0; i < NUMSET; ++i)
		{
			int	cols;

			cols = lrint((double)bidcols * Bid.set[i].size
							/ Bid.total);
			if (cols == 0 && Bid.set[i].size)
				cols = 1;

			wattrset(Hdrwin, SetColors[i]);
			while (cols--)
				mvwaddch(Hdrwin, 1, --x, ' ');
		}
	}

	if (askcols)
	{
		x = bidcols;
		for (i = 0; i < NUMSET; ++i)
		{
			int	cols;

			cols = lrint((double)askcols * Ask.set[i].size
							/ Ask.total);
			if (cols == 0 && Ask.set[i].size)
				cols = 1;

			wattrset(Hdrwin, SetColors[i]);
			while (cols--)
				mvwaddch(Hdrwin, 1, ++x, ' ');
		}
	}
	wattrset(Hdrwin, A_NORMAL);
}

static void
disp_book(BOOK *b)
{
	int	y, x;
	int	maxy = getmaxy(Subwin) - 1;
	int	n = -1;
	double	lastprice = -1;
	int	incnt = 0;
	int	insize = 0;
	int	remain = 0;
	int	wid =  getmaxx(Subwin)/3;
	int	i;

	if (!DisplayOn)
		return;

	x = (b == &Bid) ? 0 : wid;

	for (y = i = 0; i < b->num; ++i)
        {
		BOOKE	*e = &b->ent[i];

		if (Code && Code != e->code)
			continue;

		if (e->price != lastprice)
		{
			if (++n == NUMSET) n = NUMSET - 1;
			lastprice = e->price;
		}
		if (n == 0)
		{
			++incnt;
			insize += e->size;
		}
		wattrset(Subwin, SetColors[n]);
		if (e->price >= 1000)
			mvwprintw(Subwin, y, x,
				"%-5.5s%6.1f %4d",
				e->mmid, e->price, e->size);
		else if (e->price < 1.0)
			mvwprintw(Subwin, y, x,
				"%-5.5s%6.3f %4d",
				e->mmid, e->price, e->size);
		else
			mvwprintw(Subwin, y, x,
				"%-5.5s%6.2f %4d",
				e->mmid, e->price, e->size);
		wprintw(Subwin, " %02d:%02d:%02d",
				e->time/3600,
				(e->time/60) % 60,
				e->time % 60);
		wattrset(Subwin, A_NORMAL);
		if (y++ == maxy)
			break;
        }
	if (y == 0)
		mvwprintw(Subwin, y++, x, "%*s", -wid, "...no entries...");
	while (y <= maxy)
	{
		// \240 charater is a hack to force gnome-terminal to
		// clear these lines
		mvwprintw(Subwin, y++, x, "%*s", wid, "\240");
	}
	remain = b->num - i;
	if (remain > 0)
	{
		mvwprintw(Subwin, maxy, x, "%*s", wid, "");
		mvwprintw(Subwin, maxy, x, "%d more", remain);
	}

	disp_bar();
}

static void
disp_codes(void)
{
	mvwprintw(Win, getmaxy(Win)-1, getmaxx(Win)/2 - 24/2,
		"Showing Code (%c) of (*,K,O)", Code ? Code : '*');
}

static int
askcmp(const void *ve1, const void *ve2)
{
	BOOKE	*e1 = (BOOKE *) ve1;
	BOOKE	*e2 = (BOOKE *) ve2;

	if (e1->price < e2->price)
		return -1;
	if (e1->price > e2->price)
		return 1;

	if (e1->key2 > e2->key2)
		return 1;
	if (e1->key2 < e2->key2)
		return -1;

	if (e1->time > e2->time)
		return 1;
	if (e1->time < e2->time)
		return -1;

	if (e1->size < e2->size)
		return 1;
	if (e1->size > e2->size)
		return -1;

	return (strcmp(e1->mmid, e2->mmid));

}

static int
bidcmp(const void *ve1, const void *ve2)
{
	BOOKE	*e1 = (BOOKE *) ve1;
	BOOKE	*e2 = (BOOKE *) ve2;

	if (e1->price < e2->price)
		return 1;
	if (e1->price > e2->price)
		return -1;

	if (e1->key2 > e2->key2)
		return 1;
	if (e1->key2 < e2->key2)
		return -1;

	if (e1->time > e2->time)
		return 1;
	if (e1->time < e2->time)
		return -1;

	if (e1->size < e2->size)
		return 1;
	if (e1->size > e2->size)
		return -1;

	return (strcmp(e1->mmid, e2->mmid));
}

static void
add_entry(BOOK *b, BOOKE *e)
{
	int	i;

	for (i = 0; i < b->num; ++i)
		if (strcmp(b->ent[i].recid, e->recid) == 0)
			break;

	if (b->num == NUMBOOK)
	{
		error(1, "Too many book entries\n");
		return;
	}

	b->ent[i++] = *e;
	if (i > b->num)
		b->num = i;

	qsort(b->ent, b->num, sizeof(b->ent[0]),
		b == &Bid ? bidcmp : askcmp);
}

static int
del_entry(BOOK *b, char *recid)
{
	int	i;

	for (i = 0; i < b->num; ++i)
		if (strcmp(b->ent[i].recid, recid) == 0)
			break;
	if (i == b->num)
		return 0;

	for (i = i+1; i < b->num; ++i)
		b->ent[i-1] = b->ent[i];
	--b->num;

	return 1;
}

int
l2sr_getlast(STREAMER sr, char *sym, char *mmid, L2BA *bap)
{
	int	i;

	if (!Win)
		return -1;
	if (sr != Sr)
		return -1;
	if (strcmp(sym, Symbol))
		return -1;

	memset(bap, 0, sizeof(*bap));

	for (i = 0; i < Bid.num; ++i)
	{
		if (strcmp(Bid.ent[i].mmid, mmid))
			continue;
		// strcpy(bap->mmid, Bid.ent[i].mmid);
		bap->bid = Bid.ent[i].price;
		bap->bidsz = Bid.ent[i].size;
		bap->time = Bid.ent[i].time;
		bap->code = Bid.ent[i].code;
	}
	for (i = 0; i < Ask.num; ++i)
	{
		if (strcmp(Ask.ent[i].mmid, mmid))
			continue;
		// strcpy(bap->mmid, Ask.ent[i].mmid);
		bap->ask = Ask.ent[i].price;
		bap->asksz = Ask.ent[i].size;
		bap->time = Ask.ent[i].time;
		bap->code = Ask.ent[i].code;
	}
	return (bap->ask || bap->bid);
}

void
l2sr_ba(STREAMER sr, char *sym, L2BA *bap, int numba)
{
	if (!Win)
		return;
	if (sr != Sr)
		return;
	if (strcmp(sym, Symbol))
		return;

	while (numba--)
	{
		BOOKE	be;

		strcpy(be.mmid, bap->mmid);
		strcpy(be.recid, bap->mmid);
		be.time = bap->time;
		be.code = bap->code;
		be.tick = bap->bidtick;
		be.key2 = 0;

		be.price = bap->bid;
		be.size = bap->bidsz;
		if (be.price)
			add_entry(&Bid, &be);
		else
			del_entry(&Bid, be.recid);

		be.price = bap->ask;
		be.size = bap->asksz;
		be.tick = bap->asktick;
		if (be.price)
			add_entry(&Ask, &be);
		else
			del_entry(&Ask, be.recid);

		++bap;
	}

	disp_book(&Bid);
	disp_book(&Ask);
	disp_codes();
	touchwin(Win);
	update_panels(); refresh(); // doupdate();
}

void
l2sr_sets(STREAMER sr, char *sym, L2SETS *sp, int numsets)
{
	int	bids = 0;
	int	asks = 0;

	if (!Win)
		return;
	if (sr != Sr)
		return;
	if (strcmp(sym, Symbol))
		return;

	while (numsets--)
	{
		BOOKE	be;

		strcpy(be.mmid, sp->mmid);
		strcpy(be.recid, sp->recid);
		be.time = sp->time;
		be.code = sp->code;
		be.tick = sp->tick;
		be.key2 = sp->key2;

		be.price = sp->price;
		be.size = sp->size;
		if (sp->isbid)
		{
		    if (be.price)
			    add_entry(&Bid, &be);
		    else
			    del_entry(&Bid, be.recid);
		    ++bids;
		}
		else
		{
		    if (be.price)
			    add_entry(&Ask, &be);
		    else
			    del_entry(&Ask, be.recid);
		    ++asks;
		}
		++sp;
	}

	if (bids)
	    disp_book(&Bid);
	if (asks)
	    disp_book(&Ask);
	if (!bids && !asks)
	    return;
	disp_codes();
	touchwin(Win);
	update_panels(); refresh(); // doupdate();
}

void
l2sr_trade(STREAMER sr, double price, int size, int time, int row, int tstype)
{
	attr_t	attr = A_NORMAL;

	if (!Win)
		return;
	if (sr != Sr)
		return;

	if (row == -1)
	{
		wvscrollrect(Subwin,
			0, getmaxx(Subwin)*2/3,
			getmaxy(Subwin), getmaxx(Subwin),
			-1);
		row = 0;
	}

	switch(tstype)
	{
	case TS_ATHIGH: attr = COLOR_PAIR(1) | RevOrBold; break;
	case TS_ABOVE:
	case TS_ATASK:	attr = COLOR_PAIR(1) | A_NORMAL; break;
	case TS_ATBID:
	case TS_BELOW:	attr = COLOR_PAIR(2) | A_NORMAL; break;
	case TS_ATLOW:	attr = COLOR_PAIR(2) | RevOrBold; break;
	}

	qfmt_init();
	wattrset(Subwin, attr);
	if (price < 1.0)
		mvwprintw(Subwin, row, getmaxx(Subwin)*2/3,
			"%7.3f %-6.6s %02d:%02d:%02d",
			price, volfmt6b(size),
			time/3600, (time/60)%60, time%60);
	else
		mvwprintw(Subwin, row, getmaxx(Subwin)*2/3,
			"%7.2f %-6.6s %02d:%02d:%02d",
			price, volfmt6b(size),
			time/3600, (time/60)%60, time%60);
	wattrset(Subwin, A_NORMAL);
	touchwin(Win);
	update_panels(); refresh(); // doupdate();
}

void
l2sr_display(STREAMER sr, int on)
{
	if (!Win)
		return;
	if (sr != Sr)
		return;
	if (on == DisplayOn)
		return;

	DisplayOn = on;
	if (DisplayOn)
	{
		disp_book(&Bid);
		disp_book(&Ask);
		disp_codes();
		touchwin(Win);
		update_panels(); refresh(); // doupdate();
	}
}

void
l2sr_popup(STOCK *sp, STREAMER sr)
{
	char	title[256];
	int	rc;

	Win = minmaxwin(22, 36, FALSE, FALSE);
	if (!Win)
		error(1, "Can't create l2sr window\n");

	wbkgd(Win, Reverse ? A_REVERSE : A_NORMAL);

	box(Win, 0, 0);

	wattrset(Win, A_BOLD);
	mvwprintw(Win, 0, 3, "%s", sp->sym);
	sprintf(title, "%s Level II", sr->id);
	mvwprintw(Win, 0, getmaxx(Win)-strlen(title)-3, title);
	wattrset(Win, A_NORMAL);

	Hdrwin = derwin(Win, 4, getmaxx(Win) - 2, 1, 1);
	if (!Hdrwin)
		error(1, "Can't create l2sr hdrwindow\n");

	Subwin = derwin(Win,
			getmaxy(Win) - 2 - getmaxy(Hdrwin), getmaxx(Win) - 2,
			1 + getmaxy(Hdrwin), 1);
	if (!Subwin)
		error(1, "Can't create l2sr subwindow\n");

	DisplayOn = TRUE;

	strcpy(Symbol, sp->sym);

	wattrset(Hdrwin, A_BOLD);
	mvwprintw(Hdrwin, getmaxy(Hdrwin) - 1, 0,
			"MMID    Bid Size Time");
	mvwprintw(Hdrwin, getmaxy(Hdrwin) - 1, getmaxx(Hdrwin)/3,
			"MMID    Ask Size Time");
	mvwprintw(Hdrwin, getmaxy(Hdrwin) - 1, 2*getmaxx(Hdrwin)/3,
			"  Price   Size Time....");
	wattrset(Hdrwin, A_NORMAL);

	Panel = new_panel(Win);

	Bid.num = Ask.num = 0;
	Sr = sr;

	rc = (*sr->send_l2)(sr, sp->sym, TRUE);
	if (rc == SR_AUTH)
	{
		wattrset(Subwin, A_BOLD);
		mvwcenter(Subwin, getmaxy(Subwin)/2,
			"Sorry, you are not authorized for L2 quotes");
		wattrset(Subwin, A_NORMAL);
	}

	blankrect(LINES-1, 0, LINES-1, COLS-1, 0);
	touchwin(Win);
}

static void
popdown(void)
{
	if (Sr && Sr->send_l2)
		(*Sr->send_l2)(Sr, Symbol, FALSE);

	hide_panel(Panel);
	update_panels();
	del_panel(Panel);
	delwin(Hdrwin);
	delwin(Subwin);
	delwin(Win);
	Win = NULL;
}

int
l2sr_command(int c, STREAMER sr)
{
	MEVENT	m;

	switch (c)
	{
	case '*':
	case 'K':
	case 'O':
		if (c == '*') Code = 0; else Code = c;
		disp_book(&Bid);
		disp_book(&Ask);
		disp_codes();
		touchwin(Win);
		update_panels(); refresh(); // doupdate();
		break;
	case KEY_MOUSE:
		if (getmouse(&m) != OK)
			break;

		// Ignore clicks in our window
		if (m.y >= getbegy(Win)
			&& m.y < getbegy(Win) + getmaxy(Win))
			break;

		// popdown and reprocess clicks in main window
		if (ungetmouse(&m) == OK)
			Ungetch = 1;
		popdown();
		return 2;

	case KEY_F(11):
		if (1) print_rect_troff(getbegy(Win), getbegx(Win),
				getmaxy(Win), getmaxx(Win),
				NULL, "screen.tr");
		break;

	case KEY_UP:
	case KEY_DOWN:
		// Allow cursor motion to select next stock
		popdown();
		return 1;

	default:
		popdown();
		return 2;
	}
	return 0;
}

char *
mmid_lookup(char *mmid)
{
	FILE		*fp;
	char		*home = getenv("HOME");
	static char	buf[BUFSIZ];
	char		*p;
	char		*name = "Unknown";
	int		fileno = 1;

	fp = fopen("mmids.txt", "r");
	if (!fp)
	{
	userfile:
		++fileno;
		sprintf(buf, "%s/." PROGNAMESTR "/players.txt", home);
		fp = fopen(buf, "r");
		if (!fp)
		{
		systemfile:
			++fileno;
			fp = fopen("/usr/share/" PROGNAMESTR "/players.txt",
					"r");
			if (!fp)
				return "Unknown";
		}
	}

	// Find mmid tag in file
	while (fgets(buf, sizeof(buf), fp))
	{
		p = strchr(buf, '\n'); if (p) *p = 0;
		p = strchr(buf, '\r'); if (p) *p = 0;
		p = strchr(buf, '#'); if (p) *p = 0;
		if (!buf[0]) continue;

		name = strchr(buf, '\t');
		if (name) *name++ = 0; else name = "";

		if (strcasecmp(buf, mmid) == 0)
			break;
	}
	if (feof(fp))
	{
		fclose(fp);
		if (fileno < 2)
			goto userfile;
		if (fileno < 3)
			goto systemfile;
		return "Unknown";
	}
	return name;
}
