/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "SoundCard.h"
#include "Exception.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>

#include <wx/debug.h>
#include <wx/log.h>

typedef unsigned char uint8;
typedef signed short  sint16;

CSoundCard::CSoundCard(const wxString& fileName, int sampleRate, int sampleWidth) :
CSoundDev(),
m_fd(-1),
m_fileName(fileName),
m_sampleRate(sampleRate),
m_sampleWidth(sampleWidth)
{
}

CSoundCard::CSoundCard() :
CSoundDev(),
m_fd(-1),
m_fileName(wxEmptyString),
m_sampleRate(11025),
m_sampleWidth(16)
{
}

CSoundCard::~CSoundCard()
{
}

void CSoundCard::openRead(const wxString& fileName, int sampleRate, int sampleWidth)
{
	m_fileName    = fileName;
	m_sampleRate  = sampleRate;
	m_sampleWidth = sampleWidth;

	openRead();
}

void CSoundCard::openRead()
{
	wxASSERT(m_fd == -1);
	wxASSERT(m_sampleRate > 0);
	wxASSERT(m_sampleWidth == 8 || m_sampleWidth == 16);

	::wxLogInfo(wxT("Opening the sound card for read"));

	m_fd = ::open(m_fileName.mb_str(), O_RDONLY, 0);
	if (m_fd == -1) {
		wxString text;
		text.Printf(wxT("Error opening the sound card: %u %s"), errno, ::strerror(errno));
		throw CException(text);
	}

	unsigned int wanted = (m_sampleWidth == 8) ? AFMT_U8 : AFMT_S16_LE;
	unsigned int param  = wanted;
	if (::ioctl(m_fd, SNDCTL_DSP_SETFMT, &param) < 0) {
		close();
		throw CException(wxT("Error opening the sound card: SNDCTL_DSP_SETFMT"));
	}

	if (param != wanted) {
		close();
		throw CException(wxT("Error opening the sound card: Format not supported"));
	}

	wanted = param = 1;
	if (::ioctl(m_fd, SNDCTL_DSP_CHANNELS, &param) < 0) {
		close();
		throw CException(wxT("Error opening the sound card: SNDCTL_DSP_CHANNELS"));
	}

	if (param != wanted) {
		close();
		throw CException(wxT("Error opening the sound card: Cannot set Mono"));
	}

	wanted = param = m_sampleRate;
	if (::ioctl(m_fd, SNDCTL_DSP_SPEED, &param) < 0) {
		close();
		throw CException(wxT("Error opening the sound card: SNDCTL_DSP_SPEED"));
	}

	if (param != wanted) {
		close();
		wxString text;
		text.Printf(wxT("Error opening the sound card: Sample Rate, wanted %u returned %u"), wanted, param);
		throw CException(text);
	}

	wanted  = (m_sampleWidth == 8) ? 0x00000009 : 0x0000000A;
	wanted |= 0x7FFF0000;
	param   = wanted;
	if (::ioctl(m_fd, SNDCTL_DSP_SETFRAGMENT, &param) < 0) {
		close();
		throw CException(wxT("Error opening the sound card: SNDCTL_DSP_SETFRAGMENT"));
	}
}

void CSoundCard::openWrite(const wxString& fileName, int sampleRate, int sampleWidth)
{
	m_fileName    = fileName;
	m_sampleRate  = sampleRate;
	m_sampleWidth = sampleWidth;

	openWrite();
}

void CSoundCard::openWrite()
{
	wxASSERT(m_fd == -1);
	wxASSERT(m_sampleRate > 0);
	wxASSERT(m_sampleWidth == 8 || m_sampleWidth == 16);

	::wxLogInfo(wxT("Opening the sound card for write"));

	m_fd = ::open(m_fileName.mb_str(), O_WRONLY, 0);
	if (m_fd == -1) {
		wxString text;
		text.Printf(wxT("Error opening the sound card: %s"), ::strerror(errno));
		throw CException(text);
	}

	unsigned int wanted = (m_sampleWidth == 8) ? AFMT_U8 : AFMT_S16_LE;
	unsigned int param  = wanted;
	if (::ioctl(m_fd, SNDCTL_DSP_SETFMT, &param) < 0) {
		close();
		throw CException(wxT("Error opening the sound card: SNDCTL_DSP_SETFMT"));
	}

	if (param != wanted) {
		close();
		throw CException(wxT("Error opening the sound card: Format not supported"));
	}

	wanted = param = 1;
	if (::ioctl(m_fd, SNDCTL_DSP_CHANNELS, &param) < 0) {
		close();
		throw CException(wxT("Error opening the sound card: SNDCTL_DSP_CHANNELS"));
	}

	if (param != wanted) {
		close();
		throw CException(wxT("Error opening the sound card: Cannot set Mono"));
	}

	wanted = param = m_sampleRate;
	if (::ioctl(m_fd, SNDCTL_DSP_SPEED, &param) < 0) {
		close();
		throw CException(wxT("Error opening the sound card: SNDCTL_DSP_SPEED"));
	}

	if (param != wanted) {
		close();
		throw CException(wxT("Error opening the sound card: Sample Rate"));
	}

	wanted  = (m_sampleWidth == 8) ? 0x00000009 : 0x0000000A;
	wanted |= 0x00040000;
	param   = wanted;
	if (::ioctl(m_fd, SNDCTL_DSP_SETFRAGMENT, &param) < 0) {
		close();
		throw CException(wxT("Error opening the sound card: SNDCTL_DSP_SETFRAGMENT"));
	}
}

bool CSoundCard::read(double* sample, int& len)
{
	wxASSERT(m_fd != -1);
	wxASSERT(len > 0);
	wxASSERT(sample != NULL);

	if (m_sampleWidth == 8) {
		uint8* data = new uint8[len];

		len = ::read(m_fd, data, len * sizeof(uint8));

		if (len <= 0) {
			delete[] data;
			return false;
		}

		len /= sizeof(uint8);

		for (int i = 0; i < len; i++)
			sample[i] = (double(data[i]) - 128.0) / 127.0;

		delete[] data;
	} else {
		sint16* data = new sint16[len];

		len = ::read(m_fd, data, len * sizeof(sint16));

		if (len <= 0) {
			delete[] data;
			return false;
		}

		len /= sizeof(sint16);

		for (int i = 0; i < len; i++)
			sample[i] = double(data[i]) / 32767.0;

		delete[] data;
	}

	return true;
}

void CSoundCard::write(double* sample, int len, double volume)
{
	wxASSERT(m_fd != -1);
	wxASSERT(len > 0);
	wxASSERT(sample != NULL);

	int n;

	if (m_sampleWidth == 8) {
		uint8* data = new uint8[len];

		for (int i = 0; i < len; i++)
			data[i] = uint8(sample[i] * volume * 127.0 + 128.0);

		n = ::write(m_fd, data, len * sizeof(uint8));

		delete[] data;

		if (n <= 0)
			throw CException(wxT("Error writing to the sound card"));

		n /= sizeof(uint8);
	} else {
		sint16* data = new sint16[len];

		for (int i = 0; i < len; i++)
			data[i] = sint16(sample[i] * volume * 32767.0);

		n = ::write(m_fd, data, len * sizeof(sint16));

		delete[] data;

		if (n <= 0)
			throw CException(wxT("Error writing to the sound card"));


		n /= sizeof(sint16);
	}

	if (n != len)
		throw CException(wxT("Error writing to the sound card"));
}

void CSoundCard::close()
{
	wxASSERT(m_fd != -1);

	::wxLogInfo(wxT("Closing the sound card"));

	if (::ioctl(m_fd, SNDCTL_DSP_SYNC, 0) < 0)
		throw CException(wxT("Cannot set SNDCTL_DSP_SYNC"));

	::close(m_fd);
	m_fd = -1;
}
