import os
import string
import xml.sax
import UserDict

import lodjusetup
import Storage


def protect(str):
    str = u"&amp;".join(str.split(u"&"))
    str = u"&lt;".join(str.split(u"<"))
    str = u"&gt;".join(str.split(u">"))
    assert type(str) == type(u"")
    return str


def real_indent(lines, level):
    return map(lambda line: u"%*s%s" % (level * 4, u"", line), lines)

def indent(lines, level):
    return lines


def lines_to_string(lines):
    return string.join(lines, u"\n") + u"\n"


def xml_front_matter(root_element):
    return ['<?xml version="1.0"?>',
            '<!DOCTYPE %s SYSTEM "lodju.dtd">' % root_element]


def encode_photos(photos):
    list = xml_front_matter(u"photos")
    list.append(u'<photos>')
    for photo in photos:
    	list = list + indent(photo.get_xml_lines(), 1)
    list.append(u'</photos>')
    return lines_to_string(list)


class Element:

    def __init__(self, name):
    	self.name = name
	self.content = u""
	self.children = []
	
    	if name == u"folder":
	    self.meta = Folder()
    	elif name == u"folders":
	    self.meta = Folders()
	elif name == u"photos":
	    self.meta = Photos()
	elif name == u"photo":
	    self.meta = Photo()
	elif name == u"lodju-document":
	    self.meta = Document()
	else:
	    self.meta = None

    def get_child(self, name):
    	for child in self.children:
	    if child.name == name:
	    	assert child.children == []
	    	return child.get_content()

    def add_child(self, child):
    	self.children.append(child)
	
    def add_content(self, content):
    	self.content = self.content + content

    def get_content(self):
    	return self.content.strip()

    def dump(self, level=0):
    	content = self.get_content()
	if content:
	    print u"%*s%s = %s" % (level*2, u"", self.name, content)
	else:
	    print u"%*s%s" % (level*2, u"", self.name)
	for child in self.children:
	    child.dump(level + 1)


class ContentHandler(xml.sax.ContentHandler):

    def __init__(self):
    	xml.sax.ContentHandler.__init__(self)
	self._stack = []
	self.root = None

    def push(self, item):
    	self._stack.append(item)

    def top(self):
    	if self._stack:
	    return self._stack[-1]
	else:
	    return None

    def pop(self):
    	if self._stack:
	    item = self._stack[-1]
	    del self._stack[-1]
	    return item
	else:
	    return None

    def startElement(self, name, attrs):
    	e = Element(name)
    	o = self.top()
	if o:
	    o.add_child(e)
    	else:
	    self.root = e
    	self.push(e)

    def characters(self, content):
    	o = self.top()
	o.add_content(content)

    def endElement(self, name):
    	e = self.pop()
	p = self.top()
	if e.name == u"attribute":
	    p.meta[e.get_child(u"name")] = e.get_child(u"value")
    	elif e.name == u"photo":
	    assert p.name == u"photos"
	    p.meta.add(e.meta)
    	elif e.name == u"photos" and p is not None:
	    assert p.name == u"folder"
	    p.meta.set_photos(e.meta)
    	elif e.name == u"folder" and p is not None:
	    assert p.name == u"folders"
	    p.meta.add(e.meta)
	elif e.name == u"folders" and p is not None:
	    assert p.name in [u"folder", u"lodju-document"]
	    if p.name == u"folder":
		p.meta.set_subfolders(e.meta)
    	    else:
	    	p.meta.set_folders(e.meta)


class EntityResolver(xml.sax.handler.EntityResolver):

    def resolveEntity(self, publicId, systemId):
    	if systemId == "lodju.dtd":
	    return lodjusetup.LODJU_DTD
	else:
	    return systemId


class Parser:

    def __init__(self):
    	self.handler = ContentHandler()
	self.parser = xml.sax.make_parser()
	self.parser.setContentHandler(self.handler)
	self.parser.setEntityResolver(EntityResolver())

    def feed(self, data):
    	self.parser.feed(data)

    def close(self):
    	self.parser.close()
	self.handler.root.meta.clear_dirty()
	return self.handler.root.meta


class Counter:

    def __init__(self):
	self.used = []

    def remember(self, id):
    	if id not in self.used:
	    self.used.append(id)

    def get(self):
    	if self.used == []:
	    id = 1
	else:
	    id = self.used[-1]
	    while id in self.used:
	    	id = id + 1
    	self.remember(id)
	return id

id_counter = Counter()


class Attribute:

    def __init__(self, name, visible_name, editable):
    	self.name = name
	self.visible_name = visible_name
	self.editable = editable

    def get_name(self):
    	return self.name
	
    def get_visible_name(self):
    	return self.visible_name
	
    def set_visible_name(self, new_visible_name):
    	self.visible_name = new_visible_name
	
    def is_editable(self):
    	return self.editable


class UserDefinedAttribute(Attribute):

    def __init__(self, visible_name):
    	name = "user:" + visible_name # XXX shouldn't be derived from vname
    	Attribute.__init__(self, name, visible_name, 1)


class StockAttribute(Attribute):

    # The following defines the list of stock attributes.
    # Eventually this should be read in from a file, I guess.
    stock = {
	u"id": (u"ID", 0),
	u"angle": (u"Angle", 0),
	u"name": (u"Name", 1),
	u"description": (u"Description", 1),
	u"file-size": (u"File size", 0),
	u"imported-from": (u"Imported from", 0),
	u"imported-at": (u"Imported at", 0),

	u"exif:exposure-time": (u"Exposure time", 0),
	u"exif:flash": (u"Flash used", 0),
	u"exif:focal-length": (u"Focal length", 0),
	u"exif:iso-speed": (u"ISO speed", 0),
	u"exif:date-time": (u"Capture time", 0),
    }

    def __init__(self, name):
	x = self.stock[name]
    	Attribute.__init__(self, name, x[0], x[1])


class AttributeList:

    def __init__(self):
	self.attrs = []
	
    def get_all(self):
    	return map(lambda a: a.get_name(), self.attrs)
	
    def get_editable(self):
    	return map(lambda a: a.get_name(), 
	    	   filter(lambda a: a.is_editable(), self.attrs))

    def find(self, name):
    	for attr in self.attrs:
	    if attr.get_name() == name:
	    	return attr
	return None

    def find_visible_name(self, vname):
    	for attr in self.attrs:
	    if attr.get_visible_name() == vname:
	    	return attr
	return None

    def exists(self, name):
    	return self.find(name) != None
    
    def is_editable(self, name):
    	return self.find(name).is_editable()
    	
    def add_user_defined(self, visible_name):
    	if not self.find_visible_name(visible_name):
	    self.attrs.append(UserDefinedAttribute(visible_name))

    def add_stock(self, name):
    	if not self.find(name):
	    self.attrs.append(StockAttribute(name))

    def remove(self, name):
    	attr = self.find(name)
    	if attr:
	    self.attrs.remove(attr)
    
    def get_visible_name(self, name):
    	return self.find(name).get_visible_name()
	
    def set_visible_name(self, name, new_visible_name):
    	self.find(name).set_visible_name(new_visible_name)


class MetaData(UserDict.UserDict):

    type = u"MetaData"
    internals = [u"id"]
    nosave_prefixes = ["dnd:"]

    def __init__(self):
    	UserDict.UserDict.__init__(self)
	self.listeners = []
	self[u"id"] = u"%d" % id_counter.get()
	self.clear_dirty()

    def has_dirty_children(self):
    	return 0

    def is_dirty(self):
    	return self.dirty or self.has_dirty_children()

    def clear_dirty_children(self):
    	pass

    def clear_dirty(self):
    	self.dirty = 0
	self.clear_dirty_children()
	
    def make_dirty(self):
    	self.dirty = 1

    def listen(self, func):
    	if func not in self.listeners:
	    self.listeners.append(func)

    def dont_listen(self, func):
    	if func in self.listeners:
	    self.listeners.remove(func)

    def normalize(self, str):
    	if type(str) == type(u""):
	    return str
	else:
	    return unicode(str)

    def __getitem__(self, key):
	key = self.normalize(key)
	if self.data.has_key(key):
	    return self.data[key]
	else:
	    return u""

    def __setitem__(self, key, value):
	key = self.normalize(key)
	value = self.normalize(value)
	UserDict.UserDict.__setitem__(self, key, value)
	if key == u"id":
	    id_counter.remember(int(value))
	self.make_dirty()
	for func in self.listeners:
	    func(self, key)

    def dump_children(self, level):
    	pass

    def dump(self, level=0):
    	print "%*s%s" % (level * 2, "", self.type)
	for key in self.keys():
	    print "%*s  %s=%s" % (level * 2, "", key, self[key])
    	self.dump_children(level + 1)

    def get_attributes_as_xml_lines(self):
    	list = []
	keys = self.keys()
	for prefix in self.nosave_prefixes:
	    keys = filter(lambda k: k[:len(prefix)] != prefix, keys)
	for key in keys:
	    list.append(
		u"<attribute><name>%s</name><value>%s</value></attribute>" %
	    	    	(protect(key), protect(self[key])))
	return list

    def get_children_as_xml_lines(self):
    	return []

    def get_xml_lines(self):
    	list = []
	list.append(u"<%s>" % self.type)
	list = list + indent(self.get_attributes_as_xml_lines(), 1)
	list = list + indent(self.get_children_as_xml_lines(), 1)
	list.append(u"</%s>" % self.type)
	return list

    def get_xml_document(self):
    	list = xml_front_matter(self.type)
	list = list + self.get_xml_lines()
	return lines_to_string(list)

    def restore(self, dict):
    	for key in self.data.keys():
	    del self.data[key]
	for key in dict.keys():
	    self[key] = dict[key]


class Document(MetaData):

    type = u"lodju-document"

    def __init__(self):
	self.folders = Folders()
	self.storage = Storage.Storage(u"")
	
    	MetaData.__init__(self)
	
	self[u"filename"] = u""
	self.unnamed = 1

    	self.docattrs = AttributeList()

	self.folderattrs = AttributeList()
	self.folderattrs.add_stock(u"name")

    	self.photoattrs = AttributeList()
	self.photoattrs.add_stock(u"id")
	self.photoattrs.add_stock(u"angle")
	self.photoattrs.add_stock(u"name")
	self.photoattrs.add_stock(u"description")
	self.photoattrs.add_stock(u"file-size")
	self.photoattrs.add_stock(u"imported-from")
	self.photoattrs.add_stock(u"imported-at")
	self.photoattrs.add_stock(u"exif:exposure-time")
	self.photoattrs.add_stock(u"exif:flash")
	self.photoattrs.add_stock(u"exif:focal-length")
	self.photoattrs.add_stock(u"exif:iso-speed")
	self.photoattrs.add_stock(u"exif:date-time")

	self.clear_dirty()

    def is_empty(self):
    	return self.folders.get() == [] and \
	       self[u"filename"] == u"" and \
	       not self.is_dirty()

    def has_dirty_children(self):
    	return self.folders.is_dirty()

    def clear_dirty_children(self):
    	self.folders.clear_dirty()

    def dump_children(self, level):
    	self.folders.dump(level)

    def set_folders(self, folders):
    	self.folders = folders

    def get_children_as_xml_lines(self):
    	return self.folders.get_xml_lines()

    def save(self):
    	assert self[u"filename"] != u""
    	if self.unnamed:
	    self.storage.rename(self[u"filename"])
    	elif self.storage.filename != self[u"filename"]:
	    self.storage.save_as(self[u"filename"])
    	xml = self.get_xml_document()
    	self.storage.new_file(u"lodju.xml", xml)
    	self.storage.save()
	self.clear_dirty()
	self.unnamed = 0

class ManyMetaDatas(MetaData):

    type = u"ManyMetaDatas"
    
    def __init__(self):
	self.list = []
    	MetaData.__init__(self)

    def has_dirty_children(self):
    	for child in self.list:
	    if child.is_dirty():
	    	return 1
	return 0
	
    def clear_dirty_children(self):
    	for child in self.list:
	    child.clear_dirty()

    def get_attributes_as_xml_lines(self):
    	return []

    def get_children_as_xml_lines(self):
    	lines = []
	for item in self.list:
	    lines = lines + item.get_xml_lines()
	return lines

    def dump_children(self, level):
    	for item in self.list:
	    item.dump(level)

    def set(self, children):
    	self.list = children
	self.make_dirty()

    def get(self):
    	return self.list

    def has(self, item):
    	return item in self.list

    def add(self, item):
    	self.list.append(item)
	self.make_dirty()
	# XXX reparent?

    def remove(self, item):
    	if item in self.list:
	    self.list.remove(item)
	    self.make_dirty()

    def clear(self):
    	self.set([])


class Folders(ManyMetaDatas):

    type = u"folders"


class Folder(MetaData):

    type = u"folder"
    internals = [u"id", u"name"]

    def __init__(self):
	self.subfolders = Folders()
	self.photos = Photos()
    	MetaData.__init__(self)

    def has_dirty_children(self):
    	return self.photos.is_dirty() or self.subfolders.is_dirty()
	
    def clear_dirty_children(self):
    	self.photos.clear_dirty()
	self.subfolders.clear_dirty()

    def dump_children(self, level):
    	self.photos.dump(level)
	self.subfolders.dump(level)

    def set_subfolders(self, subfolders):
    	assert subfolders.type == u"folders"
    	self.subfolders = subfolders

    def set_photos(self, photos):
    	assert photos.type == u"photos"
    	self.photos = photos
	self.make_dirty()

    def get_children_as_xml_lines(self):
    	return self.photos.get_xml_lines() + self.subfolders.get_xml_lines()


class Photos(ManyMetaDatas):

    type = u"photos"

    def set(self, photos):
    	ManyMetaDatas.set(self, photos)
	self.dirty = 1
    
    def add(self, photo):
    	ManyMetaDatas.add(self, photo)
	self.dirty = 1

    
class Photo(MetaData):

    type = u"photo"
    internals = [u"id", u"angle"]

    def __init__(self):
    	MetaData.__init__(self)
	self[u"angle"] = u"0"
	self.clear_dirty()

    def set_image_filenames(self, storage):
	self[u"dnd:original-filename"] = \
	    storage.get_original_filename(self[u"id"]) or u""
	self[u"dnd:thumbnail-filename"] = \
	    storage.get_thumbnail_filename(self[u"id"]) or u""

    def unset_image_filenames(self):
    	for key in [u"dnd:original-filename", u"dnd:thumbnail-filename"]:
	    if self.has_key(key):
	    	del self[key]

    def get_angle(self):
    	return int(self[u"angle"])

    def set_angle(self, angle):
    	while angle < 0:
	    angle = angle + 360
	while angle >= 360:
	    angle = angle - 360
    	if angle not in [0, 90, 180, 270]:
	    angle = 0
	self[u"angle"] = u"%d" % angle


def test():
    doc = Document()
    folder = Folder()
    folder[u"name"] = u"Wedding pictures: all"
    folder.photos = Photos()
    folder.photos.add(Photo())
    folder.photos.add(Photo())
    folder.photos.add(Photo())
    subfolder = Folder()
    subfolder.photos.add(Photo())
    subfolder[u"name"] = u"Wedding pictures"
    folder.subfolders.add(subfolder)
    doc.folders.set([folder])
    print doc.get_xml_document()

def test2():
    import sys
    import time
    import profile
    import pstats
    
    global doc
    
    p = Parser()
    p.feed(sys.stdin.read())
    doc = p.close()

    doc[u"filename"] = u"test-metadata-save"
    profile.run("doc.save()", "metadata.prof")
    
    p = pstats.Stats("metadata.prof")
    p.strip_dirs().sort_stats("time").print_stats()

def test3():
    import sys
    import time
    import profile
    import pstats
    
    global p, data
    
    data = sys.stdin.read()

    p = Parser()
    profile.run("p.feed(data)", "metadata.prof")
    p.close()

    p = pstats.Stats("metadata.prof")
    p.strip_dirs().sort_stats("time").print_stats()

if __name__ == "__main__":
    test3()
