/* 
 * $Id: init.c,v 1.15 2000/08/25 23:28:00 cbond Exp $
 *
 * libarr - a screen management toolkit
 *
 * Copyright (C) 2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Author: Chris Bond
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <sys/types.h>
#include <stdarg.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>

#include "arr.h"
#include "misc.h"
#include "term_iface.h"
#include "input.h"
#include "types.h"
#include "ktree.h"
#include "tinfo.h"
#include "cap_offset.h"

/* arr_scr is the main (and, for now, the only) screen. */
arr_scr_t *arr_scr;		/* the screen we're writing to */

int ext_table[] = {		/* chars that can't be printed in ext. mode */
	0,	10,	8,	12,	13,	14,	15,	27,	155
};

int norm_table[] = {		/* chars that can't be printed in normal mode */
	0,	1,	2,	3,	4,	5,	6,	7,	8,
	9,	10,	11,	12,	13,	14,	15,	16,	17,
	18,	19,	20,	21,	22,	23,	24,	25,	26,
	27,	28,	29,	30,	31,	127,	155
};

byte_t inverse_color[8],	/* inverse color list */
       bold_color[16];		/* bold color list */

int
arr_scr_init(void)
{
	tsize_t t_size;
	unsigned int i;

	if (!(arr_scr = (arr_scr_t *)MALLOC(arr_scr_t, sizeof(arr_scr_t)))) {
		perror("Unable to allocate memory block");
		return -1;
	}

	if ((arr_init_term(&arr_scr->term) < 0) ||
	    (input_init() < -1)) {
		free(arr_scr);
		return -1;
	}

	t_size = (TERM_SIZE(&arr_scr->term) + 1) * sizeof(struct bchr);

	if ((!(arr_scr->main = (struct bchr *)MALLOC(struct bchr, t_size))) ||
	    (!(arr_scr->back = (struct bchr *)MALLOC(struct bchr, t_size))) ||
	    (!(arr_scr->buf.buf = (char *)MALLOC(char, (WRITE_BUF_LEN+1))))) {
		perror("Unable to allocate buffers");
		return -1;
	}

	/* Set up the write buffer: */
	arr_scr->buf.ptr = arr_scr->buf.buf;
	arr_scr->buf.end = (arr_scr->buf.buf + WRITE_BUF_LEN);

	for (i = 0; i < 256; ++i) {
		arr_ext_conv[i] = i;
		arr_norm_conv[i] = i;
	}

	for (i = 0; i < sizeof(ext_table) / sizeof(int); ++i)
		arr_ext_conv[ext_table[i]] = ' ';
	for (i = 0; i < sizeof(norm_table) / sizeof(int); ++i)
		arr_norm_conv[norm_table[i]] = ' ';

	/* I think this is a good set of defaults.   The colors in
	 * arr_inverse_col_list will be turned to inverse on older terminals.
	 * Conversely, the colors in arr_bold_col_list, will be turned into bold
	 * on the same terminals.
	 */
	arr_inverse_col_list(BG_RED, BG_GREEN, BG_CYAN, CL_INVALID);
	arr_bold_col_list(FG_BWHITE, FG_YELLOW, FG_BRED, FG_BCYAN,
		FG_BMAGENTA, FG_BBLUE, CL_INVALID);

	ktree_init();

	/*
	 * In print.c, we expect all attributes to be reset initially (since the
	 * `attribs' structure is static and thus zero'd at startup).  So
	 * anyway, reset:
	 */
	term_buffer(tinfo_getstr(tc_entry, OFFSET_ME));
	term_buffer(tinfo_getstr(tc_entry, OFFSET_AE));
	term_buffer(tinfo_getstr(tc_entry, OFFSET_SE));

	/* Finally, the cursor is shown until someone turns it off:
	 */
	term_buffer(tinfo_getstr(tc_entry, OFFSET_VE));
	arr_scr->cursor.visible = 1;

	return 0;
}

void
arr_bold_col_list(arr_color_t fc, ...)
{
	arr_color_t cl;
	va_list args;

	memset(&bold_color, 0, sizeof(bold_color));

	if (fc == CL_INVALID)
		return;
	else
		bold_color[((fc & FG_MASK) >> 4)] = 1;

	va_start(args, fc);

	/* Note: we use int, and not arr_color_t because compiler promotes all
	 * numbers to integers when passed to a variadic function.
	 */
	while ((cl = va_arg(args, int)) != CL_INVALID)
		bold_color[((cl & FG_MASK) >> 4)] = 1;

	va_end(args);
}

void
arr_inverse_col_list(arr_color_t fc, ...)
{
	arr_color_t cl;
	va_list args;

	memset(&inverse_color, 0, sizeof(inverse_color));

	if (fc == CL_INVALID)
		return;
	else
		inverse_color[(fc & BG_MASK)] = 1;
	
	va_start(args, fc);
	
	/* Note: we use int, and not arr_color_t because compiler promotes all
	 * numbers to integers when passed to a variadic function.
	 */
	while ((cl = va_arg(args, int)) != CL_INVALID)
		inverse_color[(cl & BG_MASK)] = 1;

	va_end(args);
}

void
arr_buffer_init(byte, color)
	char byte;
	arr_color_t color;
{
	register int i;

	arr_scr->seek.x = arr_scr->seek.y = 0;
	arr_scr->seek.max = TERM_SIZE(&arr_scr->term);

	arr_scr->cursor.x = arr_scr->cursor.y = 0;

	/* Fill the buffers with a character of their choice: */
	memset(arr_scr->back, CL_INVALID, TERM_SIZE(&arr_scr->term) *
					  sizeof(struct bchr));
	for (i = 0; i <= TERM_SIZE(&arr_scr->term); ++i) {
		arr_scr->main[i].chr = byte;
		arr_scr->main[i].color = color;
		arr_scr->main[i].ext = 0;
		arr_scr->main[i].reserved = 0;
	}

	arr_scr->pen_color = color;
}

void
arr_scr_free(void)
{
	input_reset();
	term_reset();

	FREE(arr_scr->main);
	FREE(arr_scr->back);
	FREE(arr_scr->buf.buf);
	FREE(arr_scr);
}
