/*-*-c++-*-*****************************************************************
                                api_c.h
                             -------------------
    begin       : October 2002
    copyright   : (C) 2002, 2003 by Christian Stimming
    email       : stimming@tuhh.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/** @file api_c.h
 *
 * @short Contains the @ref HBCI_API C wrapper functions.  */

#ifndef HBCIAPI_C_H
#define HBCIAPI_C_H

#include <openhbci/dllimport.h>
#include <openhbci/hbci.h>
#include <openhbci/pointer.h>
#include <openhbci/error.h>
#include <openhbci/bank.h>
#include <openhbci/account.h>
#include <openhbci/medium.h>
#include <openhbci/mediumpluginlist.h>
#include <openhbci/progressmonitor.h>
#include <openhbci/outboxjob.h>


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


  /** @defgroup HBCI_APIg HBCI_API functions
   *
   * @short These are the functions for the C-accessible type of the
   * central OpenHBCI class HBCI::API.
   *
   * For the "real" documentation of what you can do with objects of
   * this class, always remember to look up the member method
   * documentation (especially if the C function is not [yet]
   * documented). E.g., for <code>@ref
   * HBCI_API_loadEnvironment</code>, look up the documentation of
   * <code>@ref HBCI::API::loadEnvironment</code>.
   */
  /*@{*/
  /** @name Object Constructor */
  /*@{*/
  /**
   *  Constructor.
   *
   * @author Christian Stimming <stimming@tuhh.de>
   *
   * @param readonly if true then no jobs are allowed which actually
   * do something with your account at the
   * bank. (e.g. JobSingleTransfer)
   *
   * @param retrievalOnly If set to true, OpenHBCI will only store
   * the minimum of data per account required for subsequent
   * retrieval of account data. I.e. it will not store the balance,
   * transactions, and standing orders of an account.
   */
  extern HBCI_API *HBCI_API_new(int readonly,
				int retrievalOnly);
  /**  Default Destructor.
   * @param h The object to destroy.
   */
  extern void HBCI_API_delete(HBCI_API *h);
  /**  Upcast.
   * @param h The object to upcast to HBCI_Hbci.
   */
  extern HBCI_Hbci *HBCI_API_Hbci(HBCI_API *h);
  /*@}*/

  /* methods */

  /** @name File operations
   *
   * Methods to load and save HBCI_API's data to a file. */
  /*@{*/
  /**
   *  Load all banks, users and accounts.
   *
   * @param h The object to perform this action on.
   * @param filename The name of the file where data is loaded from.
   */
  extern HBCI_Error *HBCI_API_loadEnvironment(HBCI_API *h,
					      const char *filename);
  /**
   *  Save all banks, users, accounts.
   *
   * @param h The object to perform this action on.
   * @param filename The name of the file where data is saved to.
   */
  extern HBCI_Error *HBCI_API_saveEnvironment(const HBCI_API *h,
					      const char *filename);
  /*@}*/

  /** @name Object Retrieval
   *
   * Methods that return the objects owned by HBCI_API.*/
  /*@{*/
  /**
   *  Returns the list of known banks.
   *
   * List of known banks. May be used for browsing
   * or saving them to disk.
   * @param h The object to perform this action on.
   * @return The list of all known banks.
   */
  extern const list_HBCI_Bank *HBCI_API_bankList(const HBCI_API *h);
  /**
   *  Search for a bank.
   *
   * Searches for a bank in the internally stored list.
   * No joker or wildcard allowed.
   * @param h The object to perform this action on.
   * @param country Country code
   * @param bankCode Bank code
   */
  extern HBCI_Bank *HBCI_API_findBank(const HBCI_API *h,
				      int country,
				      const char *bankCode);
  /**
   * Searches for a medium of the given name. It therefore checks
   * the media of all already knowncustomers.
   *
   * @param h The object to perform this action on.
   * @param name medium name (card number in case of a chip card (DDV), file
   * path and name in case of a keyfile (RDH)) */
  extern HBCI_Medium *HBCI_API_findMedium(const HBCI_API *h,
					  const char *name);
  /*@}*/

  /** @name Convenience Object Retrieval
   *
   * Methods that return objects owned by a Bank that is owned
   * by HBCI_API. These methods just execute a HBCI_API_findBank
   * first, then execute that Bank's method to find the desired
   * object. */
  /*@{*/
  /**
   * Searches for an account in the internally stored list.
   * No joker or wildcard allowed.
   * @param h The object to perform this action on.
   * @param country Country code of the bank
   * @param bankCode Bank code of the bank
   * @param accnr Account number of the account
   */
  /*extern HBCI_Account *HBCI_API_findAccount(const HBCI_API *h,
                                              int country,
					      const char *bankCode,
					      const char *accnr);*/
  /**  Search for an account.
   *
   * Searches for an account in the internally stored list, with
   * additionally giving the HBCI2.2 account number suffix as
   * argument.  No joker or wildcard allowed.
   *
   * @param h The object to perform this action on.
   * @param country Country code of the bank
   * @param bankCode Bank code of the bank
   * @param accnr Account number of the account
   * @param suffix New in HBCI2.2: Account number suffix */
  /*extern HBCI_Account *HBCI_API_findAccountWSuf(const HBCI_API *h,
                                                  int country,
						  const char *bankCode,
						  const char *accnr,
						  const char *suffix);*/

  /**
   * Searches for a user in the internally stored lists.
   * No joker or wildcard allowed.
   * @param h The object to perform this action on.
   * @param country Country code of the bank
   * @param bankCode Bank code of the bank
   * @param userId User ID of the user
   */
  /*extern HBCI_User *HBCI_API_findUser(const HBCI_API *h,
					int country,
					const char *bankCode,
					const char *userId);*/

  /** Returns the total number of accounts that exist within all
   * banks of this API.
   *
   * @author Christian Stimming <stimming@tuhh.de>
   * @return Number of accounts in this API.  */
  extern int HBCI_API_totalAccounts(const HBCI_API *h);

  /** Returns the total number of users that exist within all banks
   * of this API.
   *
   * @author Christian Stimming <stimming@tuhh.de>
   * @return Number of users in this API.  */
  extern int HBCI_API_totalUsers(const HBCI_API *h);
  /*@}*/

  /** @name Manage Objects
   *
   * Methods to manage the list of objects owned by HBCI_API.
   */
  /*@{*/
  /**
   *  Add a bank.
   *
   * Add a bank to the internally stored list.
   *
   * @return NULL on success, and a HBCI_Error on failure.
   *
   * @param h The object to perform this action on.
   * @param b The bank to add.
   *
   * @param autoDelete If TRUE, then the HBCI_API takes ownership of
   * the HBCI_Bank object and will delete it if the life time is
   * over. If FALSE, the HBCI_API will <i>not</i> delete the
   * HBCI_Bank object; instead, the caller is still responsible for
   * its deletion.
   */
  extern HBCI_Error *
    HBCI_API_addBank(HBCI_API *h, HBCI_Bank *b, int autoDelete);
  /*@}*/

  /** @name OutboxJob Management
   *
   * Methods to set the HBCI_OutboxJob's managed through HBCI_API.
   */
  /*@{*/
  /**
   * Adds a job to the internal outbox queue.
   *
   * Unlike in former versions you still have access to the jobs,
   * since you only supply a pointer with this method. The HBCI_API
   * will take ownership of this job and will delete it when it gets
   * deleted itself, or when you call
   * clearQueueBy{Result,Status}. So please note that your pointer
   * to the OutboxJob might get invalid if you call
   * clearQueueBy{Result,Status}.
   *
   * For now no error is returned, but maybe later this method will check
   * if the given job fits into the current queue.
   * @param h The object to perform this action on.
   *
   * @param j The job to add. The HBCI_API will take ownership of
   * this job and will delete it when it gets deleted itself, or
   * when you call clearQueueBy{Result,Status}.  */
  extern HBCI_Error *HBCI_API_addJob(HBCI_API *h, HBCI_OutboxJob *j);
  /**
   *  Execute jobs in queue.
   *
   * This method executes all jobs in the outbox queue.
   *
   * The jobs remain in the queue until they get removed and deleted
   * via @ref HBCI_API_clearQueueByStatus() or via @ref
   * HBCI_API_clearQueueByResult().
   *
   * Since from C you cannot pass any (reference-counting)
   * HBCI::Pointer to this method, your C pointers will be pointing
   * to garbage after you called
   * HBCI_API_clearQueueBy{Status,Result}. Therefore make sure you
   * do not use the HBCI_OutboxJob... pointers any longer after you
   * called HBCI_API_clearQueue...
   *
   * @param h The object to perform this action on.
   * @param changesAllowed Specifies whether the jobs in the queue
   * are allowed to change your OpenHBCI data. This affects both
   * HBCI-specific data such as the UPD/BPD (user/bank parameter
   * data) [FIXME: is this correct?] as well as financial data of
   * your accounts such as balance, transactions and standing
   * orders. If this flag is false, then NO changes are made in your
   * OpenHBCI data at all.  */
  extern HBCI_Error *HBCI_API_executeQueue(HBCI_API *h,
					   int changesAllowed);
  /**
   *  Remove all jobs with the given result code.
   *
   * @param h The object to perform this action on.
   * @param result The result code, where all jobs with that result
   * code are removed from the queue. If HBCI_JOB_RESULT_NONE is
   * given then ALL jobs are removed.
   */
  extern void HBCI_API_clearQueueByResult(HBCI_API *h,
					  OutboxJob_Result result);
  /**
   *  Remove all jobs with the given status code.
   *
   * @param h The object to perform this action on.
   * @param status The status code, where all jobs with that status
   * code are removed from the queue. If HBCI_JOB_RESULT_NONE is
   * given then ALL jobs are removed.
   */
  extern void HBCI_API_clearQueueByStatus(HBCI_API *h,
					  OutboxJob_Status status);

  /**
   *  Removes a given job from the queue.
   *
   * @param h The object to perform this action on.
   *
   * @param job The job that should be removed. NOTE: Since
   * HBCI_API_addJob() took ownership of that job, it will be
   * delete()d. It is not allowed to use the job pointer after you
   * called this function.
   */
  extern void HBCI_API_removeQueuedJob(HBCI_API *h,
				       HBCI_OutboxJob *job);
  /*@}*/

  /** @name User Interaction
   *
   * Methods to set the user interaction class(es).  */
  /*@{*/
  /**
   *  Set the current progress monitor.
   *
   * Set a new progress monitor to be used.
   *
   * @param h The object to perform this action on.
   * @param m pointer to the monitor to use
   *
   * @param autoDelete If TRUE, then the HBCI_API takes ownership of
   * the HBCI_ProgressMonitor object and will delete it if the life
   * time is over. If FALSE, the HBCI_API will <i>not</i> delete the
   * HBCI_ProgressMonitor object; instead, the caller is still
   * responsible for its deletion.  */
  extern void HBCI_API_setMonitor(HBCI_API *h,
				  HBCI_ProgressMonitor *m,
				  int autoDelete);
  /*@}*/

  /** @name Factory Methods
   *
   * Factory methods for the main objects within OpenHBCI. These are
   * normal members of HBCI_API.  Factory methods are necessary
   * instead of using constructors since the actual implementation
   * classes are not visible for the application (and should never
   * be). */
  /*@{*/
  /**
   *  Creates a bank object.
   *
   * This will NOT be automatically added to internally stored lists.
   * @param h The object to perform this action on.
   * @param country The ISO-3166 country code of this bank. ("280"
   * is for Germany; see also HBCI_Bank_countryCode.)
   * @param bankCode The bank code of this bank.
   * @param server The IP address of the HBCI server of
   * this bank in dotted notation.
   */
  extern HBCI_Bank *HBCI_API_bankFactory(const HBCI_API *h,
					 int country,
					 const char *bankCode,
					 const char *server);
  /**
   *  Creates an medium object.
   *
   * This will NOT be automatically added to internally stored
   * lists.
   *
   * This method rather than HBCI::API::createNewMedium is to be used
   * when the actual medium already exists (i.e. the RDH
   * keyfile). (FIXME: is this correct?)
   *
   * @param h The object to perform this action on.
   * @param name The name of this medium (see HBCI_API_createNewMedium())
   * @param mediumType Chooses the kind of medium to be created.
   */
  /*extern HBCI_Medium* HBCI_API_mediumFactory(HBCI_API *h,
        				     const char *name,
        				     MediumType mediumType);*/

  /**
   *  Creates a new Medium object (and perhaps a new medium file).
   *
   * The newly created object will not be added to internally stored
   * lists; also, no bank, user, or customer object is being created.
   *
   * In case of a DDV medium (chip card) there is not much to be
   * done since all the information is on the DDV chipcard --
   * therefore simply the Medium object is created and returned.
   *
   * In case of a RDH medium (keyfile), this method will perform
   * the following steps:
   * <ul>
   * <li> Create the keyfile if it does not exist.  If a file with
   * the same name does exist already, then this method will assume
   * that this file already is a keyfile and will return the
   * MediumRDHBase object with the path of that file (without further
   * checking, i.e. if that file isn't a keyfile, you will run into
   * errors upon medium mounting).</li>
   * <li> For a new file, create the given user on the medium.</li>
   * <li> For a new file, generate new sign and crypt keys for this user</li>
   * <li> Save the new keyfile and return the MediumRDHBase object pointing
   * to that file.</li>
   * </ul>
   *
   * Please note that under no circumstances this method will
   * overwrite a file. If the file already exists, then this method
   * assumes that the file contains a valid medium. In that case no
   * keys will be created. You can try to mount the medium after
   * calling this method to check if the existing file really is a
   * medium. But beware: when changing data in the medium prior to
   * mounting it (e.g. by creating new keys) then the file may be
   * overwritten by the medium class. (see also @ref HBCI::MediumRDHBase
   * )
   *
   * @return Pointer to the new medium created. If something went
   * wront, NULL is returned and *err contains the pointer to the
   * HBCI_Error describing the error.
   *
   * @param h The object to perform this action on.
   * @param mtype chooses the kind of medium to be created
   * @param readonly FIXME: what is this?
   * @param country ISO-3166 country code. In HBCI, "280" stands for
   * Germany (see also @ref HBCI_Bank_countryCode())
   * @param bankId The bank code of the bank of this user (German
   * "Bankleitzahl")
   * @param userid The user ID of the user; assigned by the bank
   * @param name For DDV media (chip card), this should be the
   * unique card number stored on the card, or simply an empty
   * string. For RDH media (keyfile), this MUST be the full path and
   * filename of the keyfile.
   * @param err Pointer to HBCI_Error* that receives the Error, if one
   * occurred.  */
  extern HBCI_Medium *HBCI_API_createNewMedium(HBCI_API *h,
					       const char *mtype,
					       int readonly,
					       int country,
					       const char *bankId,
					       const char *userid,
					       const char *name,
					       HBCI_Error **err);

  /**
   * This method returns the enum MediumType of the named mediumtype.
   * The responsible plugin is loaded if needed.
   * @author Martin Preuss<martin@libchipcard.de>
   * @param h The object to perform this action on.
   * @param mtype named type (like "DDVCard", "RDHFile" etc)
   */
  extern MediumType HBCI_API_mediumType(HBCI_API *h,
					const char *mtype);

  /** FIXME: Document this 
   *
   * @return Returns a new plugin list which is then owned by the
   * caller, or NULL if an error
   * occurred. HBCI_MediumPluginList_delete() has to be called when
   * this is no longer needed. If an exception occurred and was
   * caught, NULL will be returned. */
  extern HBCI_MediumPluginList *HBCI_API_enumerateMediumPlugins(HBCI_API *h);
  /*@}*/

  /** @name Static Factory Methods
   *
   * Factory methods for the main objects within OpenHBCI. These are
   * static members of HBCI_API.  Factory methods are necessary
   * instead of using constructors since the actual implementation
   * classes are not visible for the application (and should never
   * be). */
  /*@{*/
  /**
   *  Creates a user object.
   *
   * This will NOT be automatically added to internally stored lists.
   * @param b The bank that this user belongs to.
   * @param m The security medium this user owns.
   * @param mediumAutoDelete If TRUE, then the HBCI_API takes ownership of
   * the HBCI_Medium object and will delete it if the life time is
   * over. If FALSE, the HBCI_API will <i>not</i> delete the
   * HBCI_Medium object; instead, the caller is still responsible for
   * its deletion.
   * @param userid The user id that identifies this user to his bank.
   */
  extern HBCI_User *HBCI_API_userFactory(HBCI_Bank *b,
					 HBCI_Medium *m,
					 int mediumAutoDelete,
					 const char *userid);
  /**
   *  Creates a customer object.
   *
   * This will NOT be automatically added to the given user.
   * @param u The user this customer (role) belongs to
   * @param id The customer ID that the bank assigned to this customer.
   * @param custName The name of this customer. You can choose this
   * name as you like, e.g. "business", "private" or whatever you
   * prefer. I.e. you can set this to any arbitrary string you consider
   * helpful.  */
  extern HBCI_Customer *HBCI_API_customerFactory(HBCI_User *u,
						 const char *id,
						 const char *custName);
  /**
   *  Creates an account object.
   *
   * This will NOT be automatically added to internally stored lists.
   * @author Martin Preuss<martin@libchipcard.de>
   * @param b The bank that this account belongs to.
   *
   * @param accountId The account number (id) that identifies this
   * account. May be NULL.
   *
   * @param accountSubId New in HBCI2.2: The suffix of the account
   * number. May be NULL.
   */
  extern HBCI_Account *HBCI_API_accountFactory(HBCI_Bank *b,
					       const char *accountId,
					       const char *accountSubId);
  /*@}*/
  /*@}*/

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif
